package com.aniways.blur;

import com.aniways.Log;
import com.aniways.R;
import com.aniways.blur.BlurRenderer;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.util.AttributeSet;
import android.view.View;
import android.widget.RelativeLayout;

/**
 * LinearLayout which applies blur effect into its background
 */
public class BlurRelativeLayout extends RelativeLayout implements IBlurLayout {

    private static final String TAG = "AniwaysBlurRelativeLayout";
	// Blur renderer instance
    private BlurRenderer mBlurRenderer;
    private View mViewBehindLayout;

    /**
     * Default constructor
     */
    public BlurRelativeLayout(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(attrs);
    }

    /**
     * Default constructor
     */
    public BlurRelativeLayout(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(attrs);
    }

    /**
     * Initialize layout to handle background blur effect
     */
    private void init(AttributeSet attrs) {
    	if(this.isInEditMode()){
			return;
		}
    	
    	mBlurRenderer = new BlurRenderer(this);

        // Read blur radius from layout variables
        TypedArray a = getContext().obtainStyledAttributes(attrs, R.styleable.AniwaysBlurView);
        for (int i = 0; i < a.getIndexCount(); ++i) {
            int attr = a.getIndex(i);
            if (attr == R.styleable.AniwaysBlurView_aniways_blur_radius) {
                float radius = a.getFloat(attr, 1f);
                mBlurRenderer.setBlurRadius(radius);
            }
        }
        a.recycle();
    }

    @Override
	protected void onAttachedToWindow() {
		if(this.isInEditMode()){
			return;
		}
		
		try{
			super.onAttachedToWindow();
			mBlurRenderer.onAttachedToWindow();
		}
		catch (Throwable e) {
			Log.e(true, TAG, "Caught Exception in onAttachToWindow", e);
		}
	}

	@Override
	protected void onDetachedFromWindow() {
		if(this.isInEditMode()){
			return;
		}
		
		try{
			super.onDetachedFromWindow();
			mBlurRenderer.onDetachedFromWindow();
		}
		catch (Throwable e) {
			Log.e(true, TAG, "Caught Exception in onDetachFromWindow", e);
		}
	}

	@Override
	public void draw(Canvas canvas){
		if(this.isInEditMode()){
			super.draw(canvas);
			return;
		}
		
		try{
			if(mBlurRenderer.isOffscreenCanvas(canvas)){
				// Do nothing
			}

			else{
				mBlurRenderer.drawToCanvas(canvas);
				super.dispatchDraw(canvas);
			}
		}
		catch (Throwable e) {
			Log.e(true, TAG, "Caught Exception in draw", e);
		}
	}

	@Override
	protected void dispatchDraw(Canvas canvas){
		if(this.isInEditMode()){
			return;
		}
		
		try{
			if(mBlurRenderer.isOffscreenCanvas(canvas)){
				// Do not draw the children if we are in the off-screen canvas
				return;
			}
			super.dispatchDraw(canvas);
		}
		catch (Throwable e) {
			Log.e(true, TAG, "Caught Exception in dispatchDraw", e);
		}
	}

	/**
	 * Set blur radius in pixels
	 */
	public void setBlurRadius(float radius) {
		try{
			mBlurRenderer.setBlurRadius(radius);

			// TODO: after the code changes I've made, make sure invalidate actually causes the blur to be re-applied
			invalidate();
		}
		catch (Throwable e) {
			Log.e(true, TAG, "Caught Exception in dispatchDraw", e);
		}
	}

	@Override
	public void setViewBehindPopup(View v) {
		this.mViewBehindLayout = v;
	}
	
	@Override
	public View getViewBehindPopup() {
		return this.mViewBehindLayout;
	}

}
