package com.aniways.billing;

import java.util.ArrayList;
import java.util.List;

import com.aniways.Aniways;
import com.aniways.Log;
import com.aniways.R;
import com.aniways.Utils;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.AnalyticsReporter.StoreEventAction;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.billing.utils.IabHelper;
import com.aniways.billing.utils.IabResult;
import com.aniways.billing.utils.Inventory;
import com.aniways.billing.utils.Purchase;
import com.aniways.billing.utils.SkuDetails;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysStoreManager;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnCompletionListener;
import android.os.Bundle;
import android.os.Handler;
import android.text.TextUtils;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.WindowManager.LayoutParams;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;
import android.view.animation.AnimationSet;
import android.view.animation.DecelerateInterpolator;
import android.view.animation.Animation.AnimationListener;
import android.widget.Button;
import android.widget.ImageButton;
import android.widget.TextView;

public class AniwaysCreditsStoreActivity extends Activity {
	public static class Price {
		protected static final Price NOT_AVAILABLE = new Price(AniwaysCreditsStoreActivity.NOT_AVAILABLE ,AniwaysCreditsStoreActivity.NOT_AVAILABLE, AniwaysCreditsStoreActivity.NOT_AVAILABLE, AniwaysCreditsStoreActivity.NOT_AVAILABLE);
		public String readablePrice, priceAmount, priceCurrency, sku;
		
		public Price(String readablePrice, String priceMicros, String priceCurrency, String sku){
			this.sku = sku;
			setReadablePrice(readablePrice);
			setPriceAmount(priceMicros);
			setPriceCurrency(priceCurrency);
		}
		
		private void setReadablePrice(String price){
			if(TextUtils.isEmpty(price)){
				Log.e(true, TAG, "Readable price empty for SKU: " + sku);
				price = AniwaysCreditsStoreActivity.NOT_AVAILABLE;
			}
			readablePrice = price;
		}
		
		private void setPriceAmount(String amount){
			if(TextUtils.isEmpty(amount)){
				Log.e(true, TAG, "Price amount empty for SKU: " + sku);
				priceAmount = AniwaysCreditsStoreActivity.NOT_AVAILABLE;
				return;
			}
			try{
				if(AniwaysCreditsStoreActivity.NOT_AVAILABLE.equals(amount)){
					priceAmount = amount;
					return;
				}
				double micros = Double.parseDouble(amount);
				double regularUnits = micros / 1000000;
				priceAmount = Double.toString(regularUnits);
			}
			catch(Exception ex){
				Log.e(true, TAG, "Error converting price micros: " + amount + ". for SKU: " + sku, ex);
				priceAmount = AniwaysCreditsStoreActivity.NOT_AVAILABLE;
			}
		}
		
		public void setPriceCurrency(String currency){
			if(TextUtils.isEmpty(currency)){
				Log.e(true, TAG, "Price currency empty for SKU: " + sku);
				currency = AniwaysCreditsStoreActivity.NOT_AVAILABLE;
			}
			priceCurrency = currency;;
		}
		
		@Override
		public String toString(){
			return "Readable price: " + readablePrice + ". PriceAmount: " + priceAmount + " PriceCurrency: " + priceCurrency;
		}
	}

	// Debug tag, for logging
	static final String TAG = "AniwaysInAppPurchaseActivity";

	// (arbitrary) request code for the purchase flow
	static final int RC_REQUEST = 10001;

	public static final String CREDITS_PURCHASE_CATEGORY = "Store";
	
	public static boolean sIsRunning;
	
	private static String sLastPurchaseFlowSku = null;

	// If starting activity this way then need to close it after a successful purchase
	public static final String KEY_GENERATE_BY_TRYIMG_TO_UNLOCK_ICON = "com.aniways.GENERATED_BY_TRYING_TO_UNLOCK_ICON";
	public static final String KEY_ICON_TRYING_TO_UNLOCK = "com.aniways.ICON_TRYING_TO_UNLOCK";
	public static final String KEY_SOURCE_SCREEN = "com.aniways.SOURCE_SCREEN";
	public static final String KEY_ID = "com.aniways.ID";

	protected static final String NOT_AVAILABLE = "N/A";

	private Price mCreditsPrice100 = Price.NOT_AVAILABLE;
	private Price mCreditsPrice250 = Price.NOT_AVAILABLE;
	private Price mCreditsPrice500 = Price.NOT_AVAILABLE;
	private Price mCreditsPrice1000 = Price.NOT_AVAILABLE;
	private Price mCreditsPrice2000 = Price.NOT_AVAILABLE;
	private Price mCreditsPrice3000 = Price.NOT_AVAILABLE;

	private ProgressDialog mProgressDialog;

	// The helper object
	IabHelper mHelper;

	private boolean mStartedFollowingIconUnlock;

	private String mIconTryingToUnlock;

	private String mSource;
	
	private String mStoreSessionId;

	@SuppressLint("NewApi")
	@Override
	public void onCreate(Bundle savedInstanceState) {
		try{
			super.onCreate(savedInstanceState);

			if(!Utils.isAndroidVersionAtLeast(11)){
				// Make us modal, so that others cant receive touch events, and hopefully the store will not close on outside touch
				getWindow().clearFlags(LayoutParams.FLAG_NOT_TOUCH_MODAL);
			}

			setContentView(R.layout.aniways_credits_store_activity);

			// Make sure touch outside the activity will not close it
			if(Utils.isAndroidVersionAtLeast(11)){
				this.setFinishOnTouchOutside(false);
			}

			if (this.getIntent() == null || this.getIntent().getExtras() == null || !this.getIntent().getExtras().containsKey(KEY_GENERATE_BY_TRYIMG_TO_UNLOCK_ICON)){
				Log.e(true, TAG, "AniwaysCreditsStoreActivity started without indication if it was a result of trying to unlock an icon or not..");
			}

			mStartedFollowingIconUnlock = this.getIntent().getExtras().getBoolean(KEY_GENERATE_BY_TRYIMG_TO_UNLOCK_ICON);
			mIconTryingToUnlock = this.getIntent().getExtras().getString(KEY_ICON_TRYING_TO_UNLOCK);
			mSource = this.getIntent().getExtras().getString(KEY_SOURCE_SCREEN);
			mStoreSessionId = this.getIntent().getExtras().getString(KEY_ID);
			Log.i(TAG, "Creating AniwaysCreditsStoreActivity " + (mStartedFollowingIconUnlock ? "" : "not ") + "following trying to unlock an icon");

			// Report view
			int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
			AnalyticsReporter.reportStoreEvent(StoreEventAction.openedStore, mStoreSessionId, creditsLeft, creditsLeft, 0, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, null, null, getPaymentMethodName());

			// Set the close button
			((ImageButton)findViewById(R.id.aniways_credits_store_close_store_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onCloseButtonClicked(v);
				}
			});

			// Set the purchase buttons
			((Button)findViewById(R.id.aniways_credits_store_100_coins_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onBuyCreditsButtonClicked(v, AniwaysPrivateConfig.getInstance().creditsSku100, 100);
				}
			});
			((Button)findViewById(R.id.aniways_credits_store_250_coins_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onBuyCreditsButtonClicked(v, AniwaysPrivateConfig.getInstance().creditsSku250, 250);
				}
			});
			((Button)findViewById(R.id.aniways_credits_store_500_coins_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onBuyCreditsButtonClicked(v, AniwaysPrivateConfig.getInstance().creditsSku500, 500);
				}
			});
			((Button)findViewById(R.id.aniways_credits_store_1000_coins_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onBuyCreditsButtonClicked(v, AniwaysPrivateConfig.getInstance().creditsSku1000, 1000);
				}
			});
			((Button)findViewById(R.id.aniways_credits_store_2000_coins_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onBuyCreditsButtonClicked(v, AniwaysPrivateConfig.getInstance().creditsSku2000, 2000);
				}
			});
			((Button)findViewById(R.id.aniways_credits_store_3000_coins_button)).setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					// No need for try-catch here cause the called method swallows everything
					onBuyCreditsButtonClicked(v, AniwaysPrivateConfig.getInstance().creditsSku3000, 3000);
				}
			});

			// Update the UI, without Animations
			updateUi(false, false);


			// Start querying of Store
			
			IAniwaysPaymentCallback paymentCallback = Aniways.getPaymentCallback();
			if(paymentCallback == null){
				setWaitScreen(true);
				setupIabHelper();
			} else {
				AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
				mCreditsPrice100 = paymentCallback.getPriceForSKU(config.creditsSku100);
				mCreditsPrice250 = paymentCallback.getPriceForSKU(config.creditsSku250);
				mCreditsPrice500 = paymentCallback.getPriceForSKU(config.creditsSku500);
				mCreditsPrice1000 = paymentCallback.getPriceForSKU(config.creditsSku1000);
				mCreditsPrice2000 = paymentCallback.getPriceForSKU(config.creditsSku2000);
				mCreditsPrice3000 = paymentCallback.getPriceForSKU(config.creditsSku3000);
				updateUi(false, false);
			}

		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in onCreate of Credit Store Activity", ex);
			setWaitScreen(false);
			updateUi(false,false);
		}
	}


	private String getPaymentMethodName() {
		IAniwaysPaymentCallback cb = Aniways.getPaymentCallback();
		if (cb != null){
			return cb.GetPaymentMethodName();
		}
		return "Google IAP";
	}


	private void setupIabHelper() {
		String base64EncodedPublicKey = AniwaysPrivateConfig.getInstance().appPublicKeyForCreditsStore;

		// Create the helper, passing it our context and the public key to verify signatures with
		Log.d(TAG, "Creating IAB helper.");
		mHelper = new IabHelper(this, base64EncodedPublicKey);

		// enable debug logging (for a production application, you should set this to false).
		mHelper.enableDebugLogging(true);

		// Start setup. This is asynchronous and the specified listener
		// will be called once setup completes.
		Log.d(TAG, "Starting setup.");
		mHelper.startSetup(new IabHelper.OnIabSetupFinishedListener() {
			public void onIabSetupFinished(IabResult result) {
				try{
					Log.d(TAG, "Setup finished.");
					
					if(result == null){
						complain("Problem setting up the store.\n\nPlease try again later.", "Problem setting in app billing. Null result");
						updateUi(false, false);
						setWaitScreen(false);
						return;
					}
					
					if (!result.isSuccess()) {
						if(result.getResponse() == IabHelper.BILLING_RESPONSE_RESULT_BILLING_UNAVAILABLE){
							complain("In-App Purchase is not setup on your device. \n\nPlease install the latest Google Play version.", "Problem setting up in-app billing: " + result + " Code: " + result.getResponse() + " . Message: " + result.getMessage());
						}
						else{
							complain("Problem setting up In-App Purchase: " + result + "\n\nPlease try again later.", "Problem setting up in-app billing: " + result + " Code: " + result.getResponse() + " . Message: " + result.getMessage());
						}
					
						updateUi(false, false);
						setWaitScreen(false);
						return;
					}

					// Hooray, IAB is fully set up. Now, let's get an inventory of stuff we own.
					Log.d(TAG, "Setup successful. Querying inventory.");
					startQueryInventory(false);
				}
				catch(Throwable ex){
					Log.e(true, TAG, "Caught an Exception on IabSetupFinished", ex);
					setWaitScreen(false);
					updateUi(false,false);
				}
			}
		});
	}


	@Override   
	public boolean onTouchEvent(MotionEvent event) {   
		
		
		
		try{
			// If we've received a touch notification that the user has touched   
			// outside the app, say that the event is handled to prevent the activity from closing.   
			if (MotionEvent.ACTION_OUTSIDE == event.getAction()) {     
				return true;   
			}
			// Delegate everything else to Activity.   
			return super.onTouchEvent(event);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in onTouchEvent", ex);
			return true;
		}
	}   

	// Listener that's called when we finish querying the items and subscriptions we own
	IabHelper.QueryInventoryFinishedListener mGotInventoryListener = new IabHelper.QueryInventoryFinishedListener() {

		public void onQueryInventoryFinished(final IabResult result, final Inventory inventory, final boolean noNeedToComplainIfThereIsNoConnection) {
			try{
				Log.d(TAG, "Query inventory finished.");
				if (result.isFailure() && result.getResponse() != IabHelper.BILLING_RESPONSE_RESULT_ERROR) { //this means that it is not a 'no connection' error
					complain("Failed to get prices from Google Play: " + result, "Failed to query inventory: " + result + " Code: " + result.getResponse() + " . Message: " + result.getMessage());
					updateUi(false, false);
				} 
				else if (result.isFailure() && result.getResponse() == IabHelper.BILLING_RESPONSE_RESULT_ERROR) { // this means that it is a 'no connection' error, so just log it, no need to inform the user - he will get 
					// Try to query again in 2 seconds in case the connection was restored. This effectively creates a loop.
					if(!noNeedToComplainIfThereIsNoConnection){
						complain("Failed to get prices from Google Play\n\nPlease make sure there is an internet connection and try again", null);
					}
					Log.w(false, TAG, "Failed to query inventory: " + result + " Code: " + result.getResponse() + " . Message: " + result.getMessage() + ". Trying again in 2 seconds");
					
					Handler handler = new Handler();
					handler.postDelayed(new NonThrowingRunnable(TAG, "while starting to query inventory after previous failure", ""){

						@Override
						public void innerRun() {
							try{
								if(AniwaysCreditsStoreActivity.this.isFinishing() || mHelper == null){
									// The activity is already destroyed
									return;
								}
								startQueryInventory(true);
							}
							catch(IllegalStateException ex){
								Log.w(false, TAG, "Cannot start the async operation since there is another one taking place, so start again in 2 secs");
								onQueryInventoryFinished(result, inventory, true);
							}
						}
					}, 2000);
					
					updateUi(false, false);
					
				}  
				else{
					/*
					 * Check for items we own. Notice that for each purchase, we check
					 * the developer payload to see if it's correct! See
					 * verifyDeveloperPayload().
					 */

					// Get the prices of the SKUs
					AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
					mCreditsPrice100 = getSkuPrice(inventory, config.creditsSku100);
					mCreditsPrice250 = getSkuPrice(inventory, config.creditsSku250);
					mCreditsPrice500 = getSkuPrice(inventory, config.creditsSku500);
					mCreditsPrice1000 = getSkuPrice(inventory, config.creditsSku1000);
					mCreditsPrice2000 = getSkuPrice(inventory, config.creditsSku2000);
					mCreditsPrice3000 = getSkuPrice(inventory, config.creditsSku3000);

					// Check for unconsumed credits. If we own credits, we should consume them immediately so the user could purchase again..
					boolean startedConsumptionFlow = false;
					startedConsumptionFlow |= startConsumptionFlowIfNecessary(inventory, config.creditsSku100, 100);
					startedConsumptionFlow |= startConsumptionFlowIfNecessary(inventory, config.creditsSku250, 250);
					startedConsumptionFlow |= startConsumptionFlowIfNecessary(inventory, config.creditsSku500, 500);
					startedConsumptionFlow |= startConsumptionFlowIfNecessary(inventory, config.creditsSku1000, 1000);
					startedConsumptionFlow |= startConsumptionFlowIfNecessary(inventory, config.creditsSku2000, 2000);
					startedConsumptionFlow |= startConsumptionFlowIfNecessary(inventory, config.creditsSku3000, 3000);

					if(startedConsumptionFlow){
						Log.i(TAG, "Started consumption flow(s), so waiting for them to finish before allowing the UI to respond");
						return;
					}
					
					// Update UI with animation
					updateUi(false, true);
					Log.d(TAG, "Query inventory was successful.");
				}

				// Update the UI if not in consumption flow
				
				setWaitScreen(false);
				Log.d(TAG, "Initial inventory query finished; enabling main UI.");
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught an Exception in onQueryInventoryFinished", ex);
				setWaitScreen(false);
				updateUi(false,false);
			}
		}

		/**
		 * @param inventory
		 * @param mSku
		 */
		private Price getSkuPrice(Inventory inventory, String sku) {
			SkuDetails details = inventory.getSkuDetails(sku);
			if(details == null){
				Log.e(true, TAG, "Received null for details of SKU: " + sku);
				return Price.NOT_AVAILABLE;
			}
			return new Price(details.getPrice() ,details.getPriceMicros(), details.getPriceCurrency(), sku);
		}

		/**
		 * @param inventory
		 * @param mSku
		 */
		private boolean startConsumptionFlowIfNecessary(Inventory inventory, String sku, int amount) {
			// Check for unconsumed credits. If we own credits, we should consume them immediately so the user could purchase again..
			Purchase creditsPurchase = inventory.getPurchase(sku);
			if (creditsPurchase != null && verifyDeveloperPayload(creditsPurchase)) {
				Log.d(TAG, "We have credits. Consuming them.");
				GoogleAnalyticsReporter.reportEvent(Verbosity.Info, CREDITS_PURCHASE_CATEGORY, "Start Consumption Flow From Inventory Query-"+getPaymentMethodName(), sku, amount);
				int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
				AnalyticsReporter.reportStoreEvent(StoreEventAction.startConsumptionFlowFromInventoryQuery, mStoreSessionId, creditsLeft, creditsLeft, amount, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, null, getPaymentMethodName());
				mHelper.consumeAsync(inventory.getPurchase(sku), mConsumeFinishedListener);
				return true;
			}

			return false;
		}
	};

	// User clicked a "Buy Credits" button
	private void onBuyCreditsButtonClicked(View button, String sku, int amount) {
		try{
			Log.d(TAG, "Buy credits button clicked. SKU: " + sku + ". amount: " + amount);

			GoogleAnalyticsReporter.reportEvent(Verbosity.Info, CREDITS_PURCHASE_CATEGORY, "Start Purchase Credits-"+getPaymentMethodName(), sku, amount);
			int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
			AnalyticsReporter.reportStoreEvent(StoreEventAction.startPurchseCredits, mStoreSessionId, creditsLeft, creditsLeft, amount, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, null, getPaymentMethodName());

			// launch the credits purchase UI flow.
			// We will be notified of completion via mPurchaseFinishedListener
			setWaitScreen(true);
			Log.d(TAG, "Launching purchase flow for credits.");

			/* TODO: for security, generate your payload here for verification. See the comments on 
			 *        verifyDeveloperPayload() for more info. Since this is a SAMPLE, we just use 
			 *        an empty string, but on a production app you should carefully generate this. */
			String payload = ""; 
			
			try{
				sLastPurchaseFlowSku = sku;
				IAniwaysPaymentCallback paymentCallback = Aniways.getPaymentCallback();
				if(paymentCallback != null){
					payWithCallback(sku, paymentCallback);
				} else {
					mHelper.launchPurchaseFlow(this, sku, RC_REQUEST, mPurchaseFinishedListener, payload);
				}
			}
			catch(IllegalStateException ex){
				complain("In App Purchase is not setup on your device. \n\nPlease install the latest Google Play version.", null);
				setWaitScreen(false);
				updateUi(false,false);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in onBuyCreditsButton. SKU: " + sku + " . Amount: " + amount, ex);
			setWaitScreen(false);
			updateUi(false,false);
		}
	}


	private void payWithCallback(String sku,
			IAniwaysPaymentCallback paymentCallback) {
		PaymentStatus paymentStatus = paymentCallback.pay(sku);
		switch (paymentStatus){
		case Error:
			Log.e(true, TAG, "Error paying for: " + sku);
			mConsumeFinishedListener.onConsumeFinished(new Purchase(sku), new IabResult(IabHelper.BILLING_RESPONSE_RESULT_ERROR, "Error."));
			break;
		case OK:
			AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
			if (	sku.equalsIgnoreCase(config.creditsSku100) ||
					sku.equalsIgnoreCase(config.creditsSku250) ||
					sku.equalsIgnoreCase(config.creditsSku500) ||
					sku.equalsIgnoreCase(config.creditsSku1000) ||
					sku.equalsIgnoreCase(config.creditsSku2000) ||
					sku.equalsIgnoreCase(config.creditsSku3000) ) {

				// Bought credits. So consume them.
				Log.d(TAG, "Purchase is credits. Starting credits consumption.");
				Price price = getCreditsSkuToPrice(sku);
				int amountOfCredits = getCreditsSkuToAmount(sku);
				int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
				GoogleAnalyticsReporter.reportEvent(Verbosity.Billing, CREDITS_PURCHASE_CATEGORY, "Credits Purchased-" + getPaymentMethodName(), price.toString(), amountOfCredits);
				AnalyticsReporter.reportStoreEvent(StoreEventAction.creditsPurchased, mStoreSessionId, creditsLeft, creditsLeft, amountOfCredits, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, price, getPaymentMethodName());
				GoogleAnalyticsReporter.reportEvent(Verbosity.Info, CREDITS_PURCHASE_CATEGORY, "Start Consumption Flow After Purchase-" + getPaymentMethodName(), sku, amountOfCredits);
				AnalyticsReporter.reportStoreEvent(StoreEventAction.startConsumptionFlowAfterPurchase, mStoreSessionId, creditsLeft, creditsLeft, amountOfCredits, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, price, getPaymentMethodName());
				mConsumeFinishedListener.onConsumeFinished(new Purchase(sku), new IabResult(IabHelper.BILLING_RESPONSE_RESULT_OK, "Inventory refresh successful."));
			}
			break;
		case Rejected:
			Log.e(true, TAG, "Payment rejected for: " + sku);
			mConsumeFinishedListener.onConsumeFinished(new Purchase(sku), new IabResult(IabHelper.BILLING_RESPONSE_RESULT_BILLING_UNAVAILABLE, "Rejected."));
			break;
		}
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		try{
			Log.d(TAG, "onActivityResult(" + requestCode + "," + resultCode + "," + data);

			// Pass on the activity result to the helper for handling
			if (!mHelper.handleActivityResult(requestCode, resultCode, data)) {
				// not handled, so handle it ourselves (here's where you'd
				// perform any handling of activity results not related to in-app
				// billing...
				super.onActivityResult(requestCode, resultCode, data);
			}
			else {
				Log.d(TAG, "onActivityResult handled by IABUtil.");
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in onActivityResult", ex);
			setWaitScreen(false);
			updateUi(false,false);
		}
	}

	/** Verifies the developer payload of a purchase. */
	private boolean verifyDeveloperPayload(Purchase p) {
		//String payload = p.getDeveloperPayload();

		/*
		 * TODO: verify that the developer payload of the purchase is correct. It will be
		 * the same one that you sent when initiating the purchase.
		 * 
		 * WARNING: Locally generating a random string when starting a purchase and 
		 * verifying it here might seem like a good approach, but this will fail in the 
		 * case where the user purchases an item on one device and then uses your app on 
		 * a different device, because on the other device you will not have access to the
		 * random string you originally generated.
		 *
		 * So a good developer payload has these characteristics:
		 * 
		 * 1. If two different users purchase an item, the payload is different between them,
		 *    so that one user's purchase can't be replayed to another user.
		 * 
		 * 2. The payload must be such that you can verify it even when the app wasn't the
		 *    one who initiated the purchase flow (so that items purchased by the user on 
		 *    one device work on other devices owned by the user).
		 * 
		 * Using your own server to store and verify developer payloads across app
		 * installations is recommended.
		 */

		return true;
	}
	
	@Override
	protected void onStop() {
		super.onStop();
		sIsRunning = false;
	}
	
	@Override
	protected void onResume() {
		super.onResume();
		sIsRunning = true;
	}

	// Callback for when a purchase is finished
	IabHelper.OnIabPurchaseFinishedListener mPurchaseFinishedListener = new IabHelper.OnIabPurchaseFinishedListener() {
		public void onIabPurchaseFinished(IabResult result, Purchase purchase) {
			try{
				Log.d(TAG, "Purchase finished: " + result + ", purchase: " + purchase);
				// when there is failure sometimes Google do not report the SKU, so we guess it was the last one
				// we tried to purchase
				String sku = (purchase == null ? AniwaysCreditsStoreActivity.sLastPurchaseFlowSku : purchase.getSku());
				if(result == null){
					complain("Failed to purchase credits.\n\nPlease try again later.", "Error purchasing credits. Result is null. SKU was: " + sku);
					setWaitScreen(false);
					updateUi(false,false);
					return;
				}
				if (result.isFailure()) {
					int response = result.getResponse();
					if(response == IabHelper.IABHELPER_USER_CANCELLED){
						Log.w(false, TAG, "Credits purchase completed with Failure due to user cancelling. SKU was: " + sku + ". result code: " + result.getResponse() + " . Result message: " + result.getMessage());
						Price price = getCreditsSkuToPrice(sku);
						int amountOfCredits = getCreditsSkuToAmount(sku);
						int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
						AnalyticsReporter.reportStoreEvent(StoreEventAction.userCancelled, mStoreSessionId, creditsLeft, creditsLeft, amountOfCredits, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, price, getPaymentMethodName());
					}
					else{
						Log.e(true, TAG, "Credits purchase completed with Failure. SKU was: " + sku + ". result code: " + result.getResponse() + " . Result message: " + result.getMessage());
						complain("Failed to purchase credits. Reason is: " + result.getMessage() + "\n\nPlease try again later.", null);
					}
					setWaitScreen(false);
					updateUi(false,false);
					return;
				}
				if (!verifyDeveloperPayload(purchase)) {
					complain("Error purchasing credits. Authenticity verification failed.\n\nPlease try again later.", "Credits purchase completed with Failure to verify developer payload. SKU was: " + sku + ". result code: " + result.getResponse() + " . Result message: " + result.getMessage() + ". Payload was: " + purchase == null ? "null" : purchase.getDeveloperPayload());
					setWaitScreen(false);
					updateUi(false,false);
					return;
				}
				if(purchase == null){
					complain("Error purchasing credits. Reason is: " + result.getMessage() + "\n\nPlease try again later.", "Purchase is null although credits purchase did not completed with Failure. SKU was: " + sku + ". result code: " + result.getResponse() + " . Result message: " + result.getMessage());
					setWaitScreen(false);
					updateUi(false,false);
					return;
				}

				Log.d(TAG, "Purchase successful.");
				AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
				if (	sku.equalsIgnoreCase(config.creditsSku100) ||
						sku.equalsIgnoreCase(config.creditsSku250) ||
						sku.equalsIgnoreCase(config.creditsSku500) ||
						sku.equalsIgnoreCase(config.creditsSku1000) ||
						sku.equalsIgnoreCase(config.creditsSku2000) ||
						sku.equalsIgnoreCase(config.creditsSku3000) ) {

					// Bought credits. So consume them.
					Log.d(TAG, "Purchase is credits. Starting credits consumption.");
					Price price = getCreditsSkuToPrice(sku);
					int amountOfCredits = getCreditsSkuToAmount(sku);
					GoogleAnalyticsReporter.reportEvent(Verbosity.Billing, CREDITS_PURCHASE_CATEGORY, "Credits Purchased-"+getPaymentMethodName(), price.toString(), amountOfCredits);
					int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
					AnalyticsReporter.reportStoreEvent(StoreEventAction.creditsPurchased, mStoreSessionId, creditsLeft, creditsLeft, amountOfCredits, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, price, getPaymentMethodName());
					GoogleAnalyticsReporter.reportEvent(Verbosity.Info, CREDITS_PURCHASE_CATEGORY, "Start Consumption Flow After Purchase-"+getPaymentMethodName(), sku, amountOfCredits);
					AnalyticsReporter.reportStoreEvent(StoreEventAction.startConsumptionFlowAfterPurchase, mStoreSessionId, creditsLeft, creditsLeft, amountOfCredits, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, sku, price, getPaymentMethodName());
					mHelper.consumeAsync(purchase, mConsumeFinishedListener);
				}
				else{
					Log.e(true, TAG, "Finished purchase of an unknown SKU: " + sku);
					setWaitScreen(false);
					updateUi(false,false);
				}
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught an Exception in onIabPurchaseFinished", ex);
				setWaitScreen(false);
				updateUi(false,false);
			}
		}
	};

	// Called when consumption is complete
	IabHelper.OnConsumeFinishedListener mConsumeFinishedListener = new IabHelper.OnConsumeFinishedListener() {

		public void onConsumeFinished(Purchase purchase, IabResult result) {
			try{
				Log.d(TAG, "Consumption finished. Purchase: " + purchase + ", result: " + result);

				if (result.isSuccess()) {
					// Successfully consumed, so we apply the effects of the item in the app
					AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
					String sku = purchase.getSku();
					if(sku.equalsIgnoreCase(config.creditsSku100)){
						onFinishConsumingCredits(100);
					}
					else if (sku.equalsIgnoreCase(config.creditsSku250)){
						onFinishConsumingCredits(250);
					}
					else if (sku.equalsIgnoreCase(config.creditsSku500)){
						onFinishConsumingCredits(500);
					}
					else if (sku.equalsIgnoreCase(config.creditsSku1000)){
						onFinishConsumingCredits(1000);
					}
					else if (sku.equalsIgnoreCase(config.creditsSku2000)){
						onFinishConsumingCredits(2000);
					}
					else if (sku.equalsIgnoreCase(config.creditsSku3000)){
						onFinishConsumingCredits(3000);
					}
					else {
						Log.e(true, TAG, "Finished consumption of an unknown SKU: " + sku);
					}
				}
				else {
					complain("Could not complete the purchase successfully.\n\nPlease close and reopen the store to try again\n\n" + result, "Error while consuming SKU: " + purchase.getSku() + ". Result code: " + result.getResponse() + " . Result message: " + result.getMessage());
				}
				updateUi(true, false);
				setWaitScreen(false);
				Log.d(TAG, "End consumption flow.");
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught an Exception in onConsumeFinished", ex);
				setWaitScreen(false);
				updateUi(false,false);
			}
		}

		/**
		 * 
		 */
		private void onFinishConsumingCredits(int amount) {
			int creditsLeft = AniwaysStoreManager.getUserCreditsLeft();
			AniwaysStoreManager.addUserCreditsAvailable(amount);
			GoogleAnalyticsReporter.reportEvent(Verbosity.Info, CREDITS_PURCHASE_CATEGORY, "Finish Consumption Flow-"+getPaymentMethodName(), "Credits before raise: " + Integer.toString(creditsLeft), amount);
			AnalyticsReporter.reportStoreEvent(StoreEventAction.finishedConsumptionFlow, mStoreSessionId, creditsLeft, creditsLeft + amount, amount, mStartedFollowingIconUnlock, mIconTryingToUnlock, mSource, null, null, getPaymentMethodName());
			Log.i(TAG, "Finished consumption flow of " + amount + " credits");

			// Start the bought credits sound
			try{
				final MediaPlayer mp = MediaPlayer.create(AniwaysCreditsStoreActivity.this, R.raw.aniways_bought_credits_sound);
				if(mp == null){
					Log.e(true, TAG, "Failed to create media player for the bought credits sound");
				}
				else{
					mp.start();
					mp.setOnCompletionListener(new OnCompletionListener(){
						@Override
						public void onCompletion(MediaPlayer mp) {
							try{
								mp.release();
							}
							catch(Throwable ex){
								Log.e(true, TAG, "Caught Exception while completing playing a sound");
							}
						}
					});
				}
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught exception while creating media player for the bought credits sound");
			}

			// TODO: If this is consumption following a buy in this session then close the store after all the animations

			//alert("You successfully purchased more credits. Your now have: " + String.valueOf(AniwaysStoreManager.getUserCreditsLeft()));
		}
	};

	// Drive button clicked. Burn gas!
	private void onCloseButtonClicked(View arg0) {
		try{
			this.finish();
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception while closing Credit Store Activity", ex);
		}
	}

	// We're being destroyed. It's important to dispose of the helper here!
	@Override
	public void onDestroy() {
		super.onDestroy();

		try{
			// very important:
			Log.d(TAG, "Destroying helper.");
			if (mHelper != null) mHelper.dispose();
			mHelper = null;
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in onDestroy", ex);
		}
	}

	// updates UI to reflect model
	private void updateUi(boolean animateCredits, boolean animatePrices) {		
		// Animate current balance if needed
		if(animateCredits){
			AnimationSet s = new AnimationSet(false);
			AlphaAnimation creditsBalancefadeInAnimation = new AlphaAnimation(0.0f, 1.0f);
			creditsBalancefadeInAnimation.setDuration(600);
			creditsBalancefadeInAnimation.setStartOffset(0);
			creditsBalancefadeInAnimation.setInterpolator(new AccelerateInterpolator());
			AlphaAnimation creditsBalanceFadeOutAnimation = new AlphaAnimation(1.0f, 0.0f);
			creditsBalanceFadeOutAnimation.setAnimationListener(new AnimationListener(){

				@Override
				public void onAnimationStart(Animation animation) {
					// TODO Auto-generated method stub
					try{

					}
					catch(Throwable ex){
						Log.e(true, TAG, "Caught an Exception in onAnimationStart", ex);
					}
				}

				@Override
				public void onAnimationEnd(Animation animation) {
					try{
						((TextView)findViewById(R.id.aniways_credits_store_credits_balance)).setText(Integer.toString(AniwaysStoreManager.getUserCreditsLeft()));
					}
					catch(Throwable ex){
						Log.e(true, TAG, "Caught an Exception in onAnimationEnd", ex);
					}

				}

				@Override
				public void onAnimationRepeat(Animation animation) {
					// TODO Auto-generated method stub
					try{

					}
					catch(Throwable ex){
						Log.e(true, TAG, "Caught an Exception in onAnimationRepeat", ex);
					}

				}

			});
			creditsBalanceFadeOutAnimation.setDuration(600);
			creditsBalanceFadeOutAnimation.setStartOffset(0);
			creditsBalanceFadeOutAnimation.setInterpolator(new DecelerateInterpolator());
			s.addAnimation(creditsBalanceFadeOutAnimation);
			s.addAnimation(creditsBalancefadeInAnimation);
			s.setRepeatCount(0);
			((TextView)findViewById(R.id.aniways_credits_store_credits_balance)).startAnimation(s);
		}
		else{
			// Update current balance without animation
			((TextView)findViewById(R.id.aniways_credits_store_credits_balance)).setText(Integer.toString(AniwaysStoreManager.getUserCreditsLeft()));
		}

		// Update prices
		((Button)findViewById(R.id.aniways_credits_store_100_coins_button)).setText(mCreditsPrice100.readablePrice);
		((Button)findViewById(R.id.aniways_credits_store_250_coins_button)).setText(mCreditsPrice250.readablePrice);
		((Button)findViewById(R.id.aniways_credits_store_500_coins_button)).setText(mCreditsPrice500.readablePrice);
		((Button)findViewById(R.id.aniways_credits_store_1000_coins_button)).setText(mCreditsPrice1000.readablePrice);
		((Button)findViewById(R.id.aniways_credits_store_2000_coins_button)).setText(mCreditsPrice2000.readablePrice);
		((Button)findViewById(R.id.aniways_credits_store_3000_coins_button)).setText(mCreditsPrice3000.readablePrice);

		// Animate prices if needed
		if(animatePrices){
			AlphaAnimation fadeInAnimation = new AlphaAnimation(AniwaysPrivateConfig.getInstance().lockedIconAlpha, 1.0f);
			fadeInAnimation.setDuration(1200);
			fadeInAnimation.setStartOffset(0);
			fadeInAnimation.setInterpolator(new AccelerateInterpolator());

			((Button)findViewById(R.id.aniways_credits_store_100_coins_button)).startAnimation(fadeInAnimation);
			((Button)findViewById(R.id.aniways_credits_store_250_coins_button)).startAnimation(fadeInAnimation);
			((Button)findViewById(R.id.aniways_credits_store_500_coins_button)).startAnimation(fadeInAnimation);
			((Button)findViewById(R.id.aniways_credits_store_1000_coins_button)).startAnimation(fadeInAnimation);
			((Button)findViewById(R.id.aniways_credits_store_2000_coins_button)).startAnimation(fadeInAnimation);
			((Button)findViewById(R.id.aniways_credits_store_3000_coins_button)).startAnimation(fadeInAnimation);
		}
	}

	// Enables or disables the "please wait" screen.
	void setWaitScreen(boolean set) {
		if(mProgressDialog == null){
			mProgressDialog = new ProgressDialog(this);
			mProgressDialog.setTitle("Connecting to Google Play");
			mProgressDialog.setMessage("Please wait..");
			mProgressDialog.setCanceledOnTouchOutside(false);

			// If the user presses 'back' in the wait dialog then close the activity because the store is not ready..
			mProgressDialog.setOnKeyListener(new Dialog.OnKeyListener() {
				@Override
				public boolean onKey(DialogInterface arg0, int keyCode, KeyEvent event) {
					try{
						// TODO Auto-generated method stub
						if (keyCode == KeyEvent.KEYCODE_BACK) {
							finish();
							mProgressDialog.dismiss();
						}
					}
					catch(Throwable ex){
						Log.e(true, TAG, "Caught an Exception in onBack of the wait dialog", ex);
					}
					return true;
				}
			});
		}

		if(set){
			Log.i(TAG, "Setting wait screen");
			mProgressDialog.show();
		}
		else{
			if(mProgressDialog.isShowing()){
				Log.i(TAG, "Removing wait screen");
				mProgressDialog.dismiss();
			}
		}
		//findViewById(R.id.screen_main).setVisibility(set ? View.GONE : View.VISIBLE);
		//findViewById(R.id.screen_wait).setVisibility(set ? View.VISIBLE : View.GONE);
	}

	void complain(String message, String messageForLogs) {
		// TODO: find out where exactly this is used and see if usage of an alert is really necessary..
		if(messageForLogs != null){
			Log.e(true, TAG, messageForLogs);
		}
		alert(message);
	}

	void alert(String message) {
		AlertDialog.Builder bld = new AlertDialog.Builder(this);
		bld.setTitle("Error");
		bld.setMessage(message);
		bld.setNeutralButton("OK", null);
		Log.d(TAG, "Showing alert dialog: " + message);
		bld.create().show();
	}

	private int getCreditsSkuToAmount(String sku){
		AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
		if(sku.equalsIgnoreCase(config.creditsSku100)){
			return 100;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku250)){
			return 250;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku500)){
			return 500;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku1000)){
			return 1000;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku2000)){
			return 2000;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku3000)){
			return 3000;
		}
		else {
			Log.e(true, TAG, "Unknown SKU: " + sku);
			return 0;
		}
	}

	private Price getCreditsSkuToPrice(String sku){
		AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
		
		if(sku.equalsIgnoreCase(config.creditsSku100)){
			return mCreditsPrice100;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku250)){
			return mCreditsPrice250;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku500)){
			return mCreditsPrice500;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku1000)){
			return mCreditsPrice1000;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku2000)){
			return mCreditsPrice2000;
		}
		else if (sku.equalsIgnoreCase(config.creditsSku3000)){
			return mCreditsPrice3000;
		}
		else {
			Log.e(true, TAG, "Unknown SKU: " + sku);
			return Price.NOT_AVAILABLE;
		}
	}


	/**
	 * 
	 */
	private void startQueryInventory(boolean noNeedToComplainIfThereIsNoConnection) {
		List<String> allSkus = new ArrayList<String>();
		AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
		
		allSkus.add(config.creditsSku100);
		allSkus.add(config.creditsSku250);
		allSkus.add(config.creditsSku500);
		allSkus.add(config.creditsSku1000);
		allSkus.add(config.creditsSku2000);
		allSkus.add(config.creditsSku3000);
		
		Log.i(TAG, "Starting to query inventory");
		
		// It is important to have all SKUs here, otherwise it will only retrieve details if they are cached in the Google Play service..
		mHelper.queryInventoryAsync(true, allSkus, mGotInventoryListener, noNeedToComplainIfThereIsNoConnection);
	}
}

