package com.aniways.analytics.db;

import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.analytics.db.IPayloadDatabaseLayer.ErrorPayloadCallback;
import com.aniways.analytics.db.IPayloadDatabaseLayer.EnqueueCallback;
import com.aniways.analytics.db.IPayloadDatabaseLayer.PayloadCallback;
import com.aniways.analytics.db.IPayloadDatabaseLayer.RemoveCallback;
import com.aniways.analytics.models.BasePayload;
import com.aniways.analytics.utils.LooperThreadWithHandler;
import com.aniways.data.AniwaysPrivateConfig;

import java.util.LinkedList;
import java.util.List;

import android.util.Pair;

/**
 * The DatabaseThread is a singleton handler thread that is
 * statically created to assure a single entry point into the
 * database to achieve SQLLite thread safety.
 *
 */
public class PayloadDatabaseThread extends LooperThreadWithHandler {

	private static final String TAG = "AniwaysPayloadDatabaseThread";
	private PayloadDatabase database;

	public PayloadDatabaseThread(PayloadDatabase database) { 
		super(TAG);

		this.database = database;
	}

	public void enqueue(final String payload, final EnqueueCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "enqueue", "", true) {

			@Override
			public void innerRun() {
				boolean success = database.addPayload(payload);
				long rowCount = database.getRowCount();

				if (callback != null) callback.onEnqueue(success, rowCount);
			}
		});
	}

	public void enqueueErrorPayload(final String payload, final EnqueueCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "enqueueError", "", true) {

			@Override
			public void innerRun() {
				boolean success = database.addErrorPayload(payload);
				long rowCount = database.getErrorsRowCount();

				if (callback != null) callback.onEnqueue(success, rowCount);
			}
		});
	}

	public void nextPayload(final PayloadCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "nextPayload", "", true) {

			@Override
			public void innerRun() {
				List<Pair<Long, BasePayload>> pairs = database.getEvents(AniwaysPrivateConfig.getInstance().analyticsMaxFlushSize);

				long minId = 0;
				long maxId = 0;

				List<BasePayload> payloads = new LinkedList<BasePayload>();

				if (pairs.size() > 0) {
					minId = pairs.get(0).first;
					maxId = pairs.get(pairs.size() - 1).first;

					for (Pair<Long, BasePayload> pair : pairs) {
						payloads.add(pair.second);
					}
				}

				if (callback != null){
					callback.onPayload(minId, maxId, payloads);
				}
			}
		});	
	}

	public void nextErrorPayload(final ErrorPayloadCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "nextErrorPayload", "", true) {

			@Override
			public void innerRun() {
				List<Pair<Long, String>> pairs = database.getErrorEvents(1);

				long minId = 0;
				long maxId = 0;

				List<String> payloads = new LinkedList<String>();

				if (pairs.size() > 0) {
					minId = pairs.get(0).first;
					maxId = pairs.get(pairs.size() - 1).first;

					for (Pair<Long, String> pair : pairs) {
						payloads.add(pair.second);
					}
				}

				if (callback != null){
					callback.onPayload(minId, maxId, payloads == null || payloads.size() == 0 ? null : payloads.get(0));
				}
			}
		});	
	}

	public void removePayloads(final long minId, final long maxId, final RemoveCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "removePayloads", "", true) {

			@Override
			public void innerRun() {
				int removed = database.removeEvents(minId, maxId);

				if (callback != null){
					callback.onRemoved(removed);
				}
			}
		});	
	}

	public void removeErrorPayloads(final long minId, final long maxId, final RemoveCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "removeErrorPayloads", "", true) {

			@Override
			public void innerRun() {
				int removed = database.removeErrorEvents(minId, maxId);

				if (callback != null){
					callback.onRemoved(removed);
				}
			}
		});	
	}
}
