/**
 * 
 */
package com.aniways;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import com.aniways.analytics.AnalyticsReporter;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysConfiguration.Verbosity;

import android.annotation.SuppressLint;
import android.content.Context;

/**
 * @author Shai
 * TODO: Add try catch to all methods to make sure that the log doesn't hinder normal app function. (these methods are already called from try-catch, but better to put them too in order not to make other methods not perform, just because of an error in the logs 
 */
public class Log {
	private static Context sContext;
	private static final String WARNING = "Warning";
	private static final String ERROR = "Error";
	private static final String ASSERT = "Assert";
	private static final String TAG = "AniwaysLog";

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Verbose} log message.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 */
	public static int v(String tag, String msg) {
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Verbose.ordinal()){
			return android.util.Log.v(tag, msg);
		}
		return 0;
	}

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Verbose} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log
	 */
	public static int v(String tag, String msg, Throwable tr) {
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Verbose.ordinal()){
			return android.util.Log.v(tag, msg, tr);
		}
		return 0;
	}

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Debug} log message.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 */
	public static int d(String tag, String msg) {
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Debug.ordinal()){
			return android.util.Log.d(tag, msg);
		}
		return 0;
	}

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Debug} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log
	 */
	public static int d(String tag, String msg, Throwable tr) {
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Debug.ordinal()){
			return android.util.Log.d(tag, msg, tr);
		}
		return 0;
	}

	/**
	 * Send an {@link com.aniways.data.AniwaysConfiguration.Verbosity#Info} log message.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 */
	public static int i(String tag, String msg) {
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Info.ordinal()){
			return android.util.Log.i(tag, msg);
		}
		return 0;
	}

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Info} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log
	 */
	public static int i(String tag, String msg, Throwable tr) {
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Info.ordinal()){
			return android.util.Log.i(tag, msg, tr);
		}
		return 0;
	}

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Warning} log message.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 */
	public static int w(boolean sendToAnalytics, String tag, String msg) {
		return w(sendToAnalytics, tag, msg, null);
	}

	/**
	 * Send a {@link com.aniways.data.AniwaysConfiguration.Verbosity#Warning} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log
	 */
	public static int w(boolean sendToAnalytics, String tag, String msg, Throwable tr) {
		if(sendToAnalytics){
			reportToAnalytics(false, WARNING, tag, msg, tr);
		}
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Info.ordinal()){
			return android.util.Log.w(tag, msg, tr);
		}
		return 0;
	}

	public static int wToGaOnly(boolean sendToAnalytics, String tag, String msg, Throwable tr) {
		if(sendToAnalytics){
			reportToAnalytics(true, WARNING, tag, msg, tr);
		}
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Info.ordinal()){
			return android.util.Log.w(tag, msg, tr);
		}
		return 0;
	}

	/*
	 * Send a {@link #WARN} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param tr An exception to log
	 */
	public static int w(boolean sendToAnalytics, String tag, Throwable tr) {
		return w(sendToAnalytics, tag, "", tr);
	}

	/**
	 * Send an {@link #ERROR} log message.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 */
	public static int e(boolean sendToAnalytics, String tag, String msg) {
		return e(sendToAnalytics, tag, msg, null);
	}

	/**
	 * Send a {@link #ERROR} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log
	 */
	public static int e(boolean sendToAnalytics, String tag, String msg, Throwable tr) {
		try{
			if(sendToAnalytics){
				reportToAnalytics(false, ERROR, tag, msg, tr);
			}
			if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Error.ordinal()){
				return android.util.Log.e(tag, msg, tr);
			}
		}
		catch(Throwable ex){
			android.util.Log.e(TAG, "Caught error while reporting an error", ex);
		}
		return 0;
	}

	/**
	 * Send a {@link #ERROR} log message and log the exception.
	 * @param tag Used to identify the source of a log message.  It usually identifies
	 *        the class or activity where the log call occurs.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log
	 */
	public static int eToGaOnly(boolean sendToAnalytics, String tag, String msg, Throwable tr) {
		if(sendToAnalytics){
			reportToAnalytics(true, ERROR, tag, msg, tr);
		}
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Error.ordinal()){
			return android.util.Log.e(tag, msg, tr);
		}
		return 0;
	}

	/**
	 * What a Terrible Failure: Report a condition that should never happen.
	 * The error will always be logged at level ASSERT with the call stack.
	 * Depending on system configuration, a report may be added to the
	 * {@link android.os.DropBoxManager} and/or the process may be terminated
	 * immediately with an error dialog.
	 * @param tag Used to identify the source of a log message.
	 * @param msg The message you would like logged.
	 */
	public static int wtf(String tag, String msg) {
		return wtf(tag, msg, new Exception());
	}

	/**
	 * What a Terrible Failure: Report an exception that should never happen.
	 * Similar to {@link #wtf(String, String)}, with an exception to log.
	 * @param tag Used to identify the source of a log message.
	 * @param tr An exception to log.
	 */
	public static int wtf(String tag, Throwable tr) {
		return wtf(tag, "", tr);
	}

	/**
	 * What a Terrible Failure: Report an exception that should never happen.
	 * Similar to {@link #wtf(String, Throwable)}, with a message as well.
	 * @param tag Used to identify the source of a log message.
	 * @param msg The message you would like logged.
	 * @param tr An exception to log.  May be null.
	 */
	@SuppressLint("NewApi")
	public static int wtf(String tag, String msg, Throwable tr) {
		reportToAnalytics(false, ASSERT, tag, msg, tr);
		if(Utils.isAndroidVersionAtLeast(8)){
			// TODO: consider adding WTF verbosity
			if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Error.ordinal()){
				return android.util.Log.wtf(tag, msg, tr);
			}
			return 0;
		}
		if(AniwaysPrivateConfig.getLogsVerbosity().ordinal() <= Verbosity.Error.ordinal()){
			return android.util.Log.e(tag, msg, tr);
		}
		return 0;

	}

	/**
	 * Handy function to get a loggable stack trace from a Throwable
	 * @param tr An exception to log
	 */
	public static String getStackTraceString(Throwable tr) {
		return android.util.Log.getStackTraceString(tr);
	}

	public static void forceInit(Context context) {
		sContext = context;	
	}

	private static void reportToAnalytics(boolean toGaOnly, String severity, String tag, String msg, Throwable tr) {
		if(sContext == null){
			android.util.Log.w(TAG, "Trying to send log to GA before the Log was initialized. Severity: " + severity + " tag: " + tag + " msg: " + msg ,tr);
			return;
		}

		if(tr == null){
			// Generate a stack trace
			tr = new Exception();
			StackTraceElement[] stackTrace = tr.getStackTrace();
			List<StackTraceElement> stackTraceList = new ArrayList<>(Arrays.asList(stackTrace));
			for (StackTraceElement e : stackTrace){
				if (e.getClassName().equals(Log.class.getCanonicalName())){
					stackTraceList.remove(0);
				} else {
					break;
				}
			}
			tr.setStackTrace(stackTraceList.toArray(new StackTraceElement[stackTraceList.size()]));
		}

		// TODO: why these conversions, why doesn't the method receive the Enum??
		Verbosity verbosity;
		if(severity.equalsIgnoreCase(ASSERT)){
			verbosity = Verbosity.Error;
		}else if(severity.equalsIgnoreCase(WARNING)){
			verbosity = Verbosity.Warning;
		}else if (severity.equalsIgnoreCase(ERROR)){
			verbosity = Verbosity.Error;
		}else{
			Log.e(true, TAG, "severity doesn't equal one of the 3 allowed one");
			verbosity = Verbosity.Error;
		}

		if(AnalyticsReporter.isInitialized()){
			AnalyticsReporter.reportError(verbosity, tag, msg, tr, toGaOnly);
		}
	}
}
