/**
 *
 */
        package com.aniways;

        import android.app.Activity;
        import android.content.Context;
        import android.graphics.Canvas;
        import android.graphics.Paint;
        import android.text.Editable;
        import android.text.Spanned;
        import android.text.style.ForegroundColorSpan;
        import android.text.style.ReplacementSpan;
        import android.util.TypedValue;
        import android.view.LayoutInflater;
        import android.widget.EditText;
        import android.widget.TextView;

/**
 * Uses a TextView to render a span on the span's canvas.
 * Takes care of correct placement of the text at the base line of the containing text line.
 */
class AniwaysSelectedWordMarkerSpan extends ReplacementSpan implements IAniwaysWordMarkerSpan
{
    // Holds the textView that is used for rendering the text on this instance's canvas.
    private final TextView textView;
    private final int paddingWidth;
    private int textHeight;
    private AniwaysNonSelectedWordMarkerSpan nonSelectedSpan;

    public AniwaysSelectedWordMarkerSpan(Context context, String text, float textSizeRawPixels, AniwaysNonSelectedWordMarkerSpan nonSelectedSpan)
    {
        this.nonSelectedSpan = nonSelectedSpan;
        LayoutInflater inflater = (LayoutInflater)context.getSystemService(Activity.LAYOUT_INFLATER_SERVICE);
        this.textView = (TextView)inflater.inflate(R.layout.aniways_contextual_suggestions_marker, null);
        this.textView.setText(text);
        this.textView.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeRawPixels);
        this.textView.setSelected(true);
        this.paddingWidth = (int)context.getResources().getDimension(R.dimen.aniways_suggestion_box_padding);
    }

    private void prepView()
    {
        this.textView.measure(0, 0);
        this.textView.layout(0, 0, textView.getMeasuredWidth(), textView.getMeasuredHeight());
    }

    public void draw(Canvas canvas, CharSequence text, int start, int end, float x, int top, int y, int bottom, Paint paint)
    {
        prepView();

        Paint.FontMetricsInt metrics = paint.getFontMetricsInt();

        // y is the base line of the text. Using y + descent to get to the bottom of the text,
        // then moving back up by textHeight to reach the desired top of the text view.
        y += metrics.descent - this.textHeight;

        if (start > 0)
        {
            x -= this.paddingWidth;
        }

        canvas.save();

        canvas.translate(x, y);
        this.textView.draw(canvas);

        canvas.restore();
    }

    public int getSize(Paint paint, CharSequence charSequence, int start, int end, Paint.FontMetricsInt fm)
    {
        prepView();

        this.textHeight = this.textView.getHeight();
        int retWidth = this.textView.getWidth();

        // By taking textView's padding into account, we can avoid the unnecessary space between the
        // text itself and the text before/after this textView.
        if (start > 0)
        {
            retWidth -= 2 * this.paddingWidth;
            this.textView.setPadding(this.paddingWidth, 0, this.paddingWidth, 0);
        }
        else
        {
            retWidth -= this.paddingWidth;
            this.textView.setPadding(0, 0, this.paddingWidth, 0);
        }

        return retWidth;
    }

    @Override
    public void setSelected(EditText textView, boolean selected)
    {
        if(selected){
            return;
        }

        Editable text = textView.getText();
        int end = text.getSpanEnd(this);
        int start = text.getSpanStart(this);

        if (start == -1 || end == -1)
        {
            return;
        }

        text.removeSpan(this);
        text.setSpan(this.nonSelectedSpan, start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
    }
}

class AniwaysNonSelectedWordMarkerSpan extends ForegroundColorSpan implements IAniwaysWordMarkerSpan {
    private final Context context;
    private final String text;
    private final float textSizeRawPixels;
    private AniwaysSelectedWordMarkerSpan selectedSpan;

    public AniwaysNonSelectedWordMarkerSpan(Context context, String text, float textSizeRawPixels) {
        super(context.getResources().getColor(R.color.aniways_word_marker_text_color));

        this.context = context;
        this.text = text;
        this.textSizeRawPixels = textSizeRawPixels;
    }

    @Override
    public void setSelected(EditText textView, boolean selected)
    {
        if(!selected){
            return;
        }
        if(this.selectedSpan == null){
            this.selectedSpan = new AniwaysSelectedWordMarkerSpan(context,text,textSizeRawPixels,this);
        }

        Editable text = textView.getText();
        int end = text.getSpanEnd(this);
        int start = text.getSpanStart(this);

        if (start == -1 || end == -1)
        {
            return;
        }

        text.removeSpan(this);
        text.setSpan(this.selectedSpan, start, end, Spanned.SPAN_EXCLUSIVE_EXCLUSIVE);
    }
}
