package com.aniways;

import java.util.HashMap;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.os.Handler;
import android.os.Looper;
import android.text.Spannable;
import android.text.TextUtils;
import android.text.style.ReplacementSpan;
import android.util.TypedValue;
import android.view.View;
import android.view.View.MeasureSpec;
import android.widget.TextView;

import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.data.AniwaysNetworkStateChecker;
import com.aniways.data.Phrase;
import com.aniways.volley.VolleyError;
import com.aniways.volley.toolbox.IResponseListener;
import com.aniways.volley.toolbox.ImageLoader;
import com.aniways.volley.toolbox.ImageLoader.ImageContainer;
import com.aniways.volley.toolbox.ImageLoader.ImageListener;

public class AniwaysLoadingImageSpan extends ReplacementSpan implements IAniwaysDynamicImageSpan {
	protected static final String TAG = "AniwaysLoadingImageSpan";
	private int mMaxWidth, mMaxHeight;

	/** The URL of the network image to load */
	private String mUrl;

	/** Local copy of the ImageLoader. */
	private ImageLoader mImageLoader;

	/** Current ImageContainer. (either in-flight or finished) */
	private ImageContainer mImageContainer;

	private HashMap<AniwaysDynamicImageSpansContainer,IResponseListener> mRequestListeners = new HashMap<AniwaysDynamicImageSpansContainer,IResponseListener>();

	private Phrase mPhrase;
	private IconData mIcon;
	private ImageSpanMetadata mImageSpanMetadata;
	private Handler mHandler = null;
	private int mMaxWidthForCache;
	private int mMaxHeightForCache;
	protected View textView;
	
	// This is used for debug - to know that a listener was registered at least once
	private boolean mListenerRegistered = false;
	protected Object mData;

	public AniwaysLoadingImageSpan(int maxWidth, int maxHeight, Context ctx, Phrase phrase, IconData icon, ImageSpanMetadata imageSpanMetadata, float textSizeRawPixels){
		mHandler = new Handler(Looper.getMainLooper());

		// for the replacement
		final TextView button = new TextView(ctx);
		button.setText(phrase.getPartToReplace());
		if(textSizeRawPixels > 0){
			button.setTextSize(TypedValue.COMPLEX_UNIT_PX, textSizeRawPixels);
		}
		button.setBackgroundColor(ctx.getResources().getColor(R.color.aniways_transparent));
		textView = button;
		
		// for image span
		mMaxWidth = maxWidth;
		mMaxHeight = maxHeight;
		mIcon = icon;
		mPhrase = phrase;
        mImageSpanMetadata = imageSpanMetadata;
	}

	private void prepView() {
		int widthSpec = MeasureSpec.makeMeasureSpec(0, MeasureSpec.UNSPECIFIED);
		int heightSpec = MeasureSpec.makeMeasureSpec(0, MeasureSpec.UNSPECIFIED);

		textView.measure(widthSpec, heightSpec);
		textView.layout(0, 0, textView.getMeasuredWidth(), textView.getMeasuredHeight());
	}

	public void draw(Canvas canvas, CharSequence text, int start, int end, float x, int top, int y, int bottom, Paint paint) {
		prepView();

		canvas.save();
		//Aligning the token to the bottom of the text
		canvas.translate(x, bottom - textView.getBottom());
		textView.draw(canvas);
		canvas.restore();
	}

	public int getSize(Paint paint, CharSequence charSequence, int i, int i2, Paint.FontMetricsInt fm) {
		prepView();

		if (fm != null) {
			//We need to make sure the layout allots enough space for the view
			int height = textView.getMeasuredHeight();
			int need = height - (fm.descent - fm.ascent);
			if (need > 0) {
				int ascent = need / 2;
				//This makes sure the text drawing area will be tall enough for the view
				fm.descent += need - ascent;
				fm.ascent -= ascent;
				fm.bottom += need - ascent;
				fm.top -= need / 2;
			}
		}

		return textView.getRight();
	}

	public void setImageUrl(String url, ImageLoader imageLoader, int maxWidthForCache, int maxHeightForCache) {
		mUrl = url;
		mImageLoader = imageLoader;
		mMaxWidthForCache = maxWidthForCache;
		mMaxHeightForCache = maxHeightForCache;
		// We do not try to load until this span is actually a part of a string that is set to be inside a text view
		// in order to prevent too many unnecessary loads when lots of messages are being prepared for possible insertion
		// into text views..
		// The URL has potentially changed. See if we need to load it.
		//loadImageIfNecessary(false);
	}

	public void registerResponseListener(AniwaysDynamicImageSpansContainer key, IResponseListener responseListener){
		Log.d(TAG, "Registering response listener. Url: " + mUrl);
		mRequestListeners.put(key, responseListener);
		this.mListenerRegistered  = true;
	}

	public void unregisterResponseListener(AniwaysDynamicImageSpansContainer key){
		Log.d(TAG, "Un-Registering response listener. Before remove: " + mRequestListeners.size() + " .Url: " + mUrl);
		IResponseListener removed = mRequestListeners.remove(key);
		Log.d(TAG, "After remove: " + mRequestListeners.size() + ". Removed: " + (removed != null) + " .Url: " + mUrl);
	}

	/**
	 * Loads the image for the view if it isn't already loaded.
	 * @param isInLayoutPass True if this was invoked from a layout pass, false otherwise.
	 */
	void loadImageIfNecessary(final boolean isInLayoutPass) {
		// if the URL to be loaded in this view is empty, cancel any old requests and clear the
		// currently loaded image.
		if (TextUtils.isEmpty(mUrl)) {
			if (mImageContainer != null) {
				mImageContainer.cancelRequest();
				mImageContainer = null;
			}
			setDefaultImageOrNull();
			return;
		}

		// if there was an old request in this span, check if it needs to be canceled.
		if (mImageContainer != null && mImageContainer.getRequestUrl() != null) {
			if (mImageContainer.getRequestUrl().equals(mUrl)) {
				// if the request is from the same URL, return.
				return;
			} else {
				mImageContainer.cancelRequest();
				setDefaultImageOrNull();
			}
		}

		// The pre-existing content of this view didn't match the current URL. Load the new image
		// from the network.
		ImageContainer newContainer = mImageLoader.get(mUrl,
				new ImageListener() {

			@Override
			public void onErrorResponse(final VolleyError error) {
				try{
					AniwaysNetworkStateChecker.checkInternet(new AniwaysNetworkStateChecker.AniwaysNetworkSateCheckCallback() {
						
						@Override
						public void run(boolean connectionAvailable) {
							if (connectionAvailable){
	    						Log.e(true, TAG, "Error loading image from url: " + mUrl + ". Error: " + error.getMessage(), error);
	    					}
	    					else{
	    						Log.w(false, TAG, "Error loading image, but its OK since there us no internet connection. Url: " + mUrl + ". Error: " + error.getMessage(), error);
	    					}
						}
					});
					
					for(IResponseListener listener : mRequestListeners.values()){
						listener.onError();
					}
				}
				catch(Throwable tr){
					Log.e(true, TAG, "Caught exception in onErrorResponse. Url: " + mUrl, tr);
				}
			}

			@Override
			public void onResponse(final ImageContainer response, final boolean isImmediate) {
				try{
					// If this was an immediate response that was delivered inside of a layout
					// pass do not set the image immediately as it will trigger a requestLayout
					// inside of a layout. Instead, defer setting the image by posting back to
					// the main thread.
					if (isImmediate  && isInLayoutPass) {
						if(response.getBitmap() == null && response.getRawData() == null){
							// We do this because if it is null we are meant to
							// put the intermediate image, but since we do not have
							// an option in this span to do so it is not needed.
							// Also, if we post this, and then the real image arrives then
							// the listeners are detached and when the posted runnable runs the
							// image is no longer null and it tries to call the listeners again,
							// yet there aren't any..
							return;
						}
						mHandler.post(new NonThrowingRunnable(TAG, "onResponse", "") {
							@Override
							public void innerRun() {
								responseInternal(response, isImmediate);
							}
						});
						return;
					}
					responseInternal(response, isImmediate);
				}
				catch(Throwable tr){
					try{
						Log.e(true, TAG, "Caught exception in onResponse. Url: " + mUrl, tr);
						for(IResponseListener listener : mRequestListeners.values()){
							listener.onError();
						}
					}
					catch(Throwable tr2){
						Log.e(true, TAG, "Caught exception in onResponse error response. Url: " + mUrl, tr2);
					}
				}
			}
			
			void responseInternal(final ImageContainer response, boolean wasImmediate){
                if(mIcon != null && mIcon.isAnimated()){
                	Object data = response.getRawData();
                	if(data == null){
                		if(wasImmediate){
                			Log.v(TAG, "Received immediate null response for animated gif: " + mUrl);
                			return;
                		}
                		else{
                			Log.w(true, TAG, "Received non-immediate null response for animated gif: " + mUrl);
                			for(IResponseListener listener : mRequestListeners.values()){
                            	listener.onError();
                            }
                			return;
                		}
                	}
                	else{
                		if(data instanceof byte[]){
                			mData = data;
                		}
                		else{
                			Log.e(true, TAG, "Received non byte array for animated gif: " + mUrl);
                			for(IResponseListener listener : mRequestListeners.values()){
                            	listener.onError();
                            }
                			return;
                		}
                	}
                }
                else if (response.getRawData() != null){ //Yay supporting animated gifs :)
                    //todo: move the methos from IconData to AniwaysImageUtil
                    mData = response.getRawData();
                }
                else if (response.getBitmap() != null) {
                    mData = response.getBitmap();
                }
                else if(response.getBitmap() == null){
                    if(wasImmediate){
                        Log.v(TAG, "Received immediate null response for bitmap: " + mUrl);
                        return;
                    }
                    Log.e(true, TAG, "Bitmap is null. Url: " + mUrl);
                	for(IResponseListener listener : mRequestListeners.values()){
                    	listener.onError();
                    }
                	return;
                }
                if(mData != null){
                	if (mRequestListeners.size() > 0) {
						for(IResponseListener listener : mRequestListeners.values()){
							listener.onSuccess();
						}
						return;
					}
					else{
						Log.w(true, TAG, "No listeners. Was registered at least once: " + mListenerRegistered + ". Url: " + mUrl);
						// Detaching on next UI thread run in order not to touch the requests queue 
						// while it might be iterated upon to get here
						mHandler.post(new NonThrowingRunnable(TAG, "detatching from window", "") {
							@Override
							public void innerRun() {
								onDetachedFromWindowCalled();
							}
						});
					}
                } 
                else{
					Log.e(true, TAG, "Data is null. Url: " + mUrl);
					for(IResponseListener listener : mRequestListeners.values()){
						listener.onError();
					}
				}
			}
		}, mMaxWidthForCache, mMaxHeightForCache, false);

		// update the ImageContainer to be the new bitmap container.
		mImageContainer = newContainer;
	}

	boolean replaceWithImageSpan(Spannable spannable, Context ctx, IIconInfoDisplayer iconInfoDisplayer, AniwaysDynamicImageSpansContainer dynamicSpansContainer){
		int start = spannable.getSpanStart(AniwaysLoadingImageSpan.this);
		int end = spannable.getSpanEnd(AniwaysLoadingImageSpan.this);

		if(start < 0 || end < 0){
			Log.v(TAG, "Span is not in the spannable. Url: " + mUrl);
			return false;
		}
		AniwaysImageSpan is = mIcon.createImageSpan(mData, mPhrase, mImageSpanMetadata.iconSelectionOrigin, ctx, mMaxWidth, mMaxHeight, iconInfoDisplayer);
		// Add the new span to the container that made the call
		if(is instanceof IAniwaysDynamicImageSpan){
			dynamicSpansContainer.addDynamicImageSpan((IAniwaysDynamicImageSpan)is);
		}
		spannable.setSpan(is, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
		spannable.removeSpan(AniwaysLoadingImageSpan.this);
		return true;
	}

	private void setDefaultImageOrNull() {
		//    if(mDefaultImageId != 0) {
		//        setImageResource(mDefaultImageId);
		//    }
		//    else {
		//        setImageBitmap(null);
		//    }
		//TODO: maybe need to remove the span
	}

	// Call this in the onLayout of the editText
	public void onLayoutCalled() {
		//super.onLayout(changed, left, top, right, bottom);
		Log.d(TAG, "layout called. Url: " + mUrl);
		loadImageIfNecessary(true);
	}

	public void onDetachedFromWindowCalled() {
		Log.d(TAG, "detach called. Url: " + mUrl);

		if(mRequestListeners.size() > 0){
			Log.d(TAG, "Not detaching cause this span is still hooked to something. Url: " + mUrl);
			return;
		}

		if (mImageContainer != null) {
			// If the view was bound to an image request, cancel it and clear
			// out the image from the view.
			mImageContainer.cancelRequest();
			//setImageBitmap(null);
			// also clear out the container so we can reload the image if necessary.
			mImageContainer = null;
		}
	}

	@Override
	public Phrase getPhrase() {
		return this.mPhrase;
	}

	@Override
	public IconData getIcon() {
		return this.mIcon;
	}

    @Override
    public ImageSpanMetadata getImageSpanMetadata() {
        return this.mImageSpanMetadata;
    }

	@Override
	public boolean isLoadingImageSpan() {
		return true;
	}

	@Override
	public void onAddedToContainer(AniwaysDynamicImageSpansContainer container) {
		this.loadImageIfNecessary(false);
	}

	@Override
	public void onRemovedFromContainer(AniwaysDynamicImageSpansContainer aniwaysDynamicImageSpansContainer) {
		this.unregisterResponseListener(aniwaysDynamicImageSpansContainer);
		this.onDetachedFromWindowCalled();
	}
}
