/*
 * Copyright (C) 2011 The Android Open Source Project
 * Copyright (C) 2011 Jake Wharton
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aniways.viewpagerindicator;

import com.aniways.Log;
import com.aniways.R;
import com.aniways.analytics.NonThrowingRunnable;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.support.v4.view.AniwaysDirectionalViewPager;
import android.support.v4.view.PagerAdapter;
import android.util.AttributeSet;
import android.view.ContextThemeWrapper;
import android.view.View;
import android.view.ViewGroup;
import android.widget.HorizontalScrollView;
import android.widget.ImageView;
import android.widget.ImageView.ScaleType;
import android.widget.LinearLayout;
import android.widget.TextView;

import static android.view.ViewGroup.LayoutParams.MATCH_PARENT;
import static android.view.ViewGroup.LayoutParams.WRAP_CONTENT;

/**
 * This widget implements the dynamic action bar tab behavior that can change
 * across different configurations or circumstances.
 */
public class TabPageIndicator extends HorizontalScrollView implements PageIndicator {
	/** Title text used when no title is provided by the adapter. */
	private static final CharSequence EMPTY_TITLE = "";

	private static final String TAG = "AniwaysTabPageIndicator";

	private NonThrowingRunnable mTabSelector;

	private final OnClickListener mTabClickListener = new OnClickListener() {
		public void onClick(View view) {
			try{
				TabView tabView = (TabView)view;
				final int newSelected = tabView.getIndex();
				mViewPager.setCurrentItem(newSelected);
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught Exception in onClick", ex);
			}
		}
	};

	private final IcsLinearLayout mTabLayout;

	private AniwaysDirectionalViewPager mViewPager;
	private AniwaysDirectionalViewPager.OnPageChangeListener mListener;

	private int mMaxTabWidth;
	private int mSelectedTabIndex;

	public TabPageIndicator(Context context) {
		this(context , null);
	}

	public TabPageIndicator(Context context, AttributeSet attrs) {
		super(context, attrs);
		setHorizontalScrollBarEnabled(false);

		mTabLayout = new IcsLinearLayout(context, R.attr.aniways_vpiTabPageIndicatorStyle);
		addView(mTabLayout, new ViewGroup.LayoutParams(WRAP_CONTENT, MATCH_PARENT));
	}

	@Override
	public void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
		try{
			final int widthMode = MeasureSpec.getMode(widthMeasureSpec);
			final boolean lockedExpanded = widthMode == MeasureSpec.EXACTLY;
			setFillViewport(lockedExpanded);

			final int childCount = mTabLayout.getChildCount();
			if (childCount > 1 && (widthMode == MeasureSpec.EXACTLY || widthMode == MeasureSpec.AT_MOST)) {
				if (childCount > 2) {
					mMaxTabWidth = (int)(MeasureSpec.getSize(widthMeasureSpec) * 0.4f);
				} else {
					mMaxTabWidth = MeasureSpec.getSize(widthMeasureSpec) / 2;
				}
			} else {
				mMaxTabWidth = -1;
			}

			final int oldWidth = getMeasuredWidth();
			super.onMeasure(widthMeasureSpec, heightMeasureSpec);
			final int newWidth = getMeasuredWidth();

			if (lockedExpanded && oldWidth != newWidth) {
				// Recenter the tab display if we're at a new (scrollable) size.
				setCurrentItem(mSelectedTabIndex);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onMeasure", ex);
		}
	}

	private void animateToTab(final int position) {
		final View tabView = mTabLayout.getChildAt(position);
		if (mTabSelector != null) {
			removeCallbacks(mTabSelector);
		}
		mTabSelector = new NonThrowingRunnable(TAG, "run of AnimateToTab", "") {
			public void innerRun() {
				final int scrollPos = tabView.getLeft() - (getWidth() - tabView.getWidth()) / 2;
				smoothScrollTo(scrollPos, 0);
				mTabSelector = null;
			}
		};
		post(mTabSelector);
	}

	@Override
	public void onAttachedToWindow() {
		try{
			super.onAttachedToWindow();
			if (mTabSelector != null) {
				// Re-post the selector we saved
				post(mTabSelector);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onAttachedToWindow", ex);
		}
	}

	@Override
	public void onDetachedFromWindow() {
		try{
			super.onDetachedFromWindow();
			if (mTabSelector != null) {
				removeCallbacks(mTabSelector);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onDetachedFromWindow", ex);
		}
	}

	private void addTab(int index, CharSequence text, Drawable icon) {

		if (icon != null) {
			//tabView.setCompoundDrawablesWithIntrinsicBounds(icon, null, null, null);

			ImageTabView imageTabView = new ImageTabView(getContext());
			imageTabView.setImageDrawable(icon);
			imageTabView.setScaleType(ScaleType.CENTER_INSIDE);
			imageTabView.mIndex = index;

			imageTabView.setFocusable(true);
			imageTabView.setOnClickListener(mTabClickListener);

            // Convert dp to pixels.
            final float scale = getResources().getDisplayMetrics().density;
            mTabLayout.addView(imageTabView, new LinearLayout.LayoutParams((int)(40f * scale + 0.5f), MATCH_PARENT, 1));
		}
		else{

			TextTabView textTabView = new TextTabView(getContext());
			textTabView.setText(text);
			textTabView.mIndex = index;

			textTabView.setFocusable(true);
			textTabView.setOnClickListener(mTabClickListener);
			mTabLayout.addView(textTabView, new LinearLayout.LayoutParams(WRAP_CONTENT, MATCH_PARENT, 0));
		}

	}

	@Override
	public void onPageScrollStateChanged(int arg0) {
		try{
			if (mListener != null) {
				mListener.onPageScrollStateChanged(arg0);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onPageScrollStateChanged", ex);
		}
	}

	@Override
	public void onPageScrolled(int arg0, float arg1, int arg2) {
		try{
			if (mListener != null) {
				mListener.onPageScrolled(arg0, arg1, arg2);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onPageScrolled", ex);
		}
	}

	@Override
	public void onPageSelected(int arg0) {
		try{
			setCurrentItem(arg0);
			if (mListener != null) {
				mListener.onPageSelected(arg0);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onPageSelected", ex);
		}
	}

	@Override
	public void setViewPager(AniwaysDirectionalViewPager view) {
		PagerAdapter adapter = null;
		try{
			if (mViewPager == view) {
				return;
			}
			if (mViewPager != null) {
				mViewPager.setOnPageChangeListener(null);
			}
			adapter = view.getAdapter();
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in setViewPager", ex);
		}
		if (adapter == null) {
			throw new IllegalStateException("ViewPager does not have adapter instance.");
		}
		try{
			mViewPager = view;
			view.setOnPageChangeListener(this);
			notifyDataSetChanged();
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in setViewPager", ex);
		}
	}

	public void notifyDataSetChanged() {
		try{
			mTabLayout.removeAllViews();
			PagerAdapter adapter = mViewPager.getAdapter();
			IconPagerAdapter iconAdapter = null;
			if (adapter instanceof IconPagerAdapter) {
				iconAdapter = (IconPagerAdapter)adapter;
			}
			final int count = adapter.getCount();
			for (int i = 0; i < count; i++) {
				CharSequence title = adapter.getPageTitle(i);
				if (title == null) {
					title = EMPTY_TITLE;
				}
				Drawable icon = null;
				if (iconAdapter != null) {
					icon = iconAdapter.getIconDrawable(i);
				}
				addTab(i, title, icon);
			}
			if (mSelectedTabIndex > count) {
				mSelectedTabIndex = count - 1;
			}
			setCurrentItem(mSelectedTabIndex);
			requestLayout();
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in notifyDataSetChanged", ex);
		}
	}

	@Override
	public void setViewPager(AniwaysDirectionalViewPager view, int initialPosition) {
		try{
			setViewPager(view);
			setCurrentItem(initialPosition);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in setViewPager", ex);
		}
	}

	@Override
	public void setCurrentItem(int item) {
		try{
			if(this.isInEditMode()){
				return;
			}

			if (mViewPager == null) {
                Log.w(false, TAG, "Tab Pag Indicator ViewPager is not set: Current=" + mSelectedTabIndex + ", New=" + item);
				return;
			}
			mSelectedTabIndex = item;
			mViewPager.setCurrentItem(item);

			final int tabCount = mTabLayout.getChildCount();
			for (int i = 0; i < tabCount; i++) {
				final View child = mTabLayout.getChildAt(i);
				final boolean isSelected = (i == item);
				child.setSelected(isSelected);
				if (isSelected) {
					animateToTab(item);
				}
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in setCurrentItem", ex);
		}
	}

	@Override
	public void setOnPageChangeListener(AniwaysDirectionalViewPager.OnPageChangeListener listener) {
		mListener = listener;
	}

	private interface TabView{
		int getIndex();
	}

	private class ImageTabView extends ImageView implements TabView {
		private int mIndex;
		private static final String TAG = "AniwaysImageTabView";

		public ImageTabView(Context context) {
			super(new ContextThemeWrapper(context, R.style.Aniways_StyledIndicators), null, R.attr.aniways_vpiImageTabPageIndicatorStyle);
		}

		@Override
		public void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
			try{
				super.onMeasure(widthMeasureSpec, heightMeasureSpec);

				// Re-measure if we went beyond our maximum size.
				if (mMaxTabWidth > 0 && getMeasuredWidth() > mMaxTabWidth) {
					super.onMeasure(MeasureSpec.makeMeasureSpec(mMaxTabWidth, MeasureSpec.EXACTLY),
							heightMeasureSpec);
				}
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught Exception in onMeasure", ex);
			}
		}

		public int getIndex() {
			return mIndex;
		}
	}

	private class TextTabView extends TextView implements TabView {
		private int mIndex;
		private static final String TAG = "AniwaysTextTabView";

		public TextTabView(Context context) {
			super(new ContextThemeWrapper(context, R.style.Aniways_StyledIndicators), null, R.attr.aniways_vpiTextTabPageIndicatorStyle);
		}

		@Override
		public void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
			try{
				super.onMeasure(widthMeasureSpec, heightMeasureSpec);

				// Re-measure if we went beyond our maximum size.
				if (mMaxTabWidth > 0 && getMeasuredWidth() > mMaxTabWidth) {
					super.onMeasure(MeasureSpec.makeMeasureSpec(mMaxTabWidth, MeasureSpec.EXACTLY),
							heightMeasureSpec);
				}
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught Exception in onMeasure", ex);
			}
		}

		public int getIndex() {
			return mIndex;
		}
	}
}
