package com.aniways.ui.views;

import android.content.Context;
import android.content.Intent;
import android.content.res.Resources;
import android.graphics.Color;
import android.util.AttributeSet;
import android.view.View;
import android.widget.LinearLayout;

import com.aniways.R;
import com.aniways.contextual.ContextualPopup;
import com.aniways.data.AniwaysStatics;
import com.aniways.nineoldandroids.view.ViewHelper;
import com.aniways.settings.AniwaysSettingsActivity;

/**
 * Presents the settings card part of the contextual popup.
 */
public class ContextualSettingsView extends LinearLayout
{
    private int parallaxMaxPadding;
    private int cardBaseColor;
    private int revealWidth;
    private View settingsCardContent;
    private View settingsCardHolder;

    public ContextualSettingsView(Context context)
    {
        super(context);
    }

    public ContextualSettingsView(Context context, AttributeSet attrs)
    {
        super(context, attrs);
    }

    @Override
    public void onFinishInflate()
    {
        super.onFinishInflate();

        Resources res = getResources();
        this.cardBaseColor = res.getColor(R.color.aniways_contextual_asset_card_background_color);
        this.parallaxMaxPadding = (int)res.getDimension(R.dimen.aniways_contextual_card_setting_padding);
        this.settingsCardContent = findViewById(R.id.aniways_contextual_settings_view);
        this.settingsCardHolder = findViewById(R.id.aniways_contextual_settings_card_holder);
        this.revealWidth = (int)res.getDimension(R.dimen.aniways_contextual_card_reveal_width);

        View icon = findViewById(R.id.aniways_contextual_settings_card_icon);
        View text = findViewById(R.id.aniways_contextual_settings_card_title);

        ViewHelper.setAlpha(icon, ContextualAssetView.IconMaxAlpha);
        ViewHelper.setAlpha(text, ContextualAssetView.IconMaxAlpha);

        this.setOnClickListener(new OnClickListener()
        {
            @Override
            public void onClick(View v)
            {
                ContextualPopup.dismissAllOpenPopups();

                Intent settingsIntent = new Intent(getContext(), AniwaysSettingsActivity.class);
                settingsIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                AniwaysStatics.getApplicationContext().startActivity(settingsIntent);
            }
        });
    }

    public void initContent(int pagerHeight)
    {
        LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams)this.settingsCardHolder.getLayoutParams();
        int cardHeight = (int)getResources().getDimension(R.dimen.aniways_contextual_card_min_height);
        lp.topMargin = pagerHeight - cardHeight;

        this.settingsCardHolder.setLayoutParams(lp);
    }

    /**
     * Updates the visuals of this instance to match its horizontal scroll position.
     * @param position The view's horizontal position along a scrolling view.
     *                 (-Infinity, -1] means the view is not visible.
     *                 (-1, 0) means the view's peeking in from the left, partially visible.
     *                 [0, 0] means the view is completely visible.
     *                 (0, Infinity) means the view's leftmost content is partially covered.
     * @param pageWidth The full page width.
     */
    public void updateByPosition(float position, int pageWidth)
    {
        int targetColor = getShadedColorByPosition(position, cardBaseColor);
        this.settingsCardContent.setBackgroundColor(targetColor);

        float oneOverRelativeWidth = 1 - ((float)(this.settingsCardContent.getWidth() - this.revealWidth) / pageWidth);

        if (position <= oneOverRelativeWidth)
        {
            this.settingsCardContent.setPadding(this.parallaxMaxPadding, 0, 0, 0);
        }
        else if (position < 1.0f)
        {
            float relativePadding = (position - oneOverRelativeWidth) / (1 - oneOverRelativeWidth);
            this.settingsCardContent.setPadding((int)(this.parallaxMaxPadding * (1 - relativePadding)), 0, 0, 0);
        }
        else
        {
            this.settingsCardContent.setPadding(0, 0, 0, 0);
        }
    }

    private static int getShadedColorByPosition(float position, int baseColor)
    {
        final float LightCutOff = 0.7f;
        final float BrightnessStepSize = 0.12f;

        int retColor;

        if (position <= 1.0f)
        {
            retColor = changeColorBrightness(baseColor, 1.0f - BrightnessStepSize);
        }
        else
        {
            int stepsDown = (int)position;
            float remainder = position % 1;
            if (remainder > LightCutOff) remainder = LightCutOff;

            float ratio = 1.0f - (stepsDown * BrightnessStepSize) - (remainder / LightCutOff) * BrightnessStepSize;

            retColor = changeColorBrightness(baseColor, ratio);
        }

        return retColor;
    }

    private static int changeColorBrightness(int color, float brightnessRatio)
    {
        float[] hsv = new float[3];
        Color.colorToHSV(color, hsv);
        hsv[2] *= brightnessRatio;

        return Color.HSVToColor(hsv);
    }
}
