package com.aniways.service;

import com.aniways.Log;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.data.AniwaysStatics;
import com.aniways.service.utils.AniwaysServiceUtils;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;

/**
 * Receiver class that listens to the BOOT_COMPLETED action and the pending intent,
 * used to schedule the update service at boot time, and start the updates
 * when pending intent is fired.
 */
public class AlarmReceiver extends BroadcastReceiver {
	private static final String TAG = "AniwaysBroadcastReceiver";
	static final String ANIWAYS_HEARTBEAT_ACTION = "com.aniways.heartbeat.action";

	/**
	 * Start the or schedule the update service when pending intent is fired or
	 * BOOT_COMPLETED action received.
	 */
	@Override
	public void onReceive(Context context, Intent intent) {
		try{
			// Although this is called on the app process, we init as if on service process since we only need this
			// to start the service if necessary.
			AniwaysStatics.init(context.getApplicationContext(), context, true);
			
			AlarmListener alarmListener = new AniwaysAlarmListener();

			// Check if its a pending intent and start work.
			if (null == intent.getAction()) {
				Log.i(TAG, "Broadcast received with null intent action - performing work in the service");

				// Execute update
				alarmListener.sendWakefulWork(context, true);
			} else if(intent.getAction().equals(ANIWAYS_HEARTBEAT_ACTION)){
				sendAppInstalledEvent(context, true);
			} else {
				Log.i(TAG, "Broadcast received with " + intent.getAction() + " intent action - schedueling alarms");
				// If BOOT_COMPLETED action intent just schedule the updates
				WakefulIntentService.scheduleAlarms(alarmListener, context);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in onReceive", ex);
		}
	}

	static void sendAppInstalledEvent(Context context, boolean fromAlarm) {
		Log.i(TAG, (fromAlarm ? "Broadcast received" : "") + " with heartbeat intent action - sending app (still) installed event");
		long now = System.currentTimeMillis();
		long installTime = AniwaysServiceUtils.getAppInstallTime(context, now);
		AnalyticsReporter.reportAppStillInstalled(installTime, now);
		if(installTime == now){
			Log.i(TAG, "Setting app install time to be: " + now);
			AniwaysServiceUtils.setAppInstallTime(context, now);
		}
	}
}