/**
 *
 */
package com.aniways.data;

import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import android.annotation.SuppressLint;
import android.text.TextUtils;

import aniways.com.google.gson.Gson;
import aniways.com.google.gson.reflect.TypeToken;
import aniways.com.google.gson.stream.JsonReader;

import com.aniways.IconData;
import com.aniways.AssetType;
import com.aniways.Log;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.data.AniwaysConfiguration.Verbosity;

/**
 * @author Shai
 */
public class JsonParser {
    private static final String TAG = "AniwaysKeywordsJsonParser";
    private static final String REVERT_TO_TEXT_ICON_NAME = "revert_to_text.png";
    private static final String UNKNOWN = "UNKNOWN";

    private IconsHashSet mIcons;
    private IconsHashSet mCommercialIcons;
    private EmojiHashSet mEmoji;
    private EmojiHashSet mEmojiWithVarientSelector;
    private PhrasesHashSet mPhrases;
    private PhrasesHashSet mCommercialPhrases;
    private Chars mChars;
    private Chars mCommercialChars;
    private LinkedHashMap<AssetType, LinkedHashMap<String, LinkedHashMap<String, List<IconData>>>> mAssetsCategoriesFamiliesIcons;

    private KeywordsFile mKeywordsFile;

    static JsonParser emptyInstance = new JsonParser();

    //TODO: add dealing with substitutes
    public JsonParser() {
        // init all members, even thoough we re-create them later in order not to have null members in case of an Exception
        // in the middle of the init later
        init();
    }

    public interface InputStreamProvider {
        InputStream getStream() throws FileNotFoundException;
    }

    void parseDataFile(InputStreamProvider inputStreamProvider) throws FileNotFoundException {
        if (inputStreamProvider == null) {
            Log.e(true, TAG, "Received null input stream provider, not parsing.");
            return;
        }
        InputStream inputStream = inputStreamProvider.getStream();

        if (inputStream == null) {
            Log.e(true, TAG, "Received null input stream, not parsing.");
            return;
        }
        KeywordsFile keywordsFile = null;
        try {
            long startTime = System.currentTimeMillis();
            String jsontext = loadTextFile(inputStream);
            long jsonTextLength = jsontext.length();
            if (TextUtils.isEmpty(jsontext)) {
                Log.e(true, TAG, "Empty json text, doing nothig");
                return;
            }
            keywordsFile = KeywordsFile.parseFromJson(jsontext);
            extractInfoFromKeywordsFile(keywordsFile);
            // Report timing
            if (AnalyticsReporter.isInitialized()) {
                AnalyticsReporter.ReportTiming(Verbosity.Info, startTime, "Performance", "Create Data From Json with gson", "num Json chars: " + jsonTextLength, TAG, "num chars");
            }
            return;
        } catch (OutOfMemoryError ex) {
            Log.w(true, TAG, "Received out of memory error while parsing json - trying to parse stream. Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex);
            keywordsFile = null;
            try {
                long startTime = System.currentTimeMillis();
                init();
                try {
                    inputStream.close();
                }
                catch (Throwable tr) {
                }
                inputStream = inputStreamProvider.getStream();
                if (inputStream == null) {
                    Log.e(true, TAG, "Received null input stream after out of mem error, not parsing.");
                    return;
                }
                parseStream(inputStream);
                // Report timing
                if (AnalyticsReporter.isInitialized()) {
                    AnalyticsReporter.ReportTiming(Verbosity.Info, startTime, "Performance", "Create Data From Json from stream", "num Json chars: " + "unknown", TAG, "num chars");
                }
                return;
            } catch (Throwable ex2) {
                Log.e(true, TAG, "Could not parse keywords JSON from stream after out of memory!! Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex2);
                init();
                return;
            }
        } catch (Throwable ex) {
            Log.w(true, TAG, "Could not parse keywords JSON!! with gson. Trying manually. Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex);
            keywordsFile = null;
            try {
                // If we are here then parsing was not successful..
                long startTime = System.currentTimeMillis();
                init();
                try {
                    inputStream.close();
                } catch (Throwable tr) {
                }
                inputStream = inputStreamProvider.getStream();
                if (inputStream == null) {
                    Log.e(true, TAG, "Received null input stream after error, not parsing.");
                    return;
                }
                String jsontext = loadTextFile(inputStream);
                long jsonTextLength = jsontext.length();
                if (TextUtils.isEmpty(jsontext)) {
                    Log.i(TAG, "Empty json text, doing nothig");
                    return;
                }
                keywordsFile = parseManually(jsontext);
                extractInfoFromKeywordsFile(keywordsFile);
                // Report timing
                if (AnalyticsReporter.isInitialized()) {
                    AnalyticsReporter.ReportTiming(Verbosity.Info, startTime, "Performance", "Create Data From Json manually", "num Json chars: " + jsonTextLength, TAG, "num chars");
                    GoogleAnalyticsReporter.reportEvent(Verbosity.Info, TAG, "Finished manual parse successfully after Exception: " + ex.getMessage(), keywordsFile.version, 0);
                }
                return;
            } catch (OutOfMemoryError ex2) {
                Log.w(true, TAG, "Received out of memory error while parsing json manually - trying to parse stream. Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex2);
                keywordsFile = null;
                try {
                    long startTime = System.currentTimeMillis();
                    init();
                    try {
                        inputStream.close();
                    } catch (Throwable tr) {
                    }
                    inputStream = inputStreamProvider.getStream();
                    if (inputStream == null) {
                        Log.e(true, TAG, "Received null input stream after out of mem error when parsing manually, not parsing.");
                        return;
                    }
                    parseStream(inputStream);
                    // Report timing
                    if (AnalyticsReporter.isInitialized()) {
                        AnalyticsReporter.ReportTiming(Verbosity.Info, startTime, "Performance", "Create Data From Json from stream after oom exception when parsing manually", "num Json chars: " + "unknown", TAG, "num chars");
                    }
                    return;
                } catch (Throwable ex3) {
                    Log.e(true, TAG, "Could not parse keywords JSON from stream after out of memory when trying to parse manually!! Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex3);
                    init();
                    return;
                }
            } catch (Throwable ex2) {
                Log.w(true, TAG, "Received error while parsing json manually - trying to parse stream. Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex2);
                keywordsFile = null;
                try {
                    long startTime = System.currentTimeMillis();
                    init();
                    try {
                        inputStream.close();
                    } catch (Throwable tr) {
                    }
                    inputStream = inputStreamProvider.getStream();
                    if (inputStream == null) {
                        Log.e(true, TAG, "Received null input stream after error when parsing manually, not parsing.");
                        return;
                    }
                    parseStream(inputStream);
                    // Report timing
                    if (AnalyticsReporter.isInitialized()) {
                        AnalyticsReporter.ReportTiming(Verbosity.Info, startTime, "Performance", "Create Data From Json from stream after exception when parsing manually", "num Json chars: " + "unknown", TAG, "num chars");
                        GoogleAnalyticsReporter.reportEvent(Verbosity.Info, TAG, "Finished stream parse successfully after Exception in manual parse: " + ex2.getMessage(), mKeywordsFile.version, 0);
                    }
                    return;
                } catch (Throwable ex3) {
                    Log.e(true, TAG, "Could not parse keywords JSON from stream after error when trying to parse manually!! Version is: " + (keywordsFile == null ? "null" : keywordsFile.version), ex3);
                    init();
                    return;
                }
            }
        }
    }

    @SuppressLint("UseSparseArrays")
    private void extractInfoFromKeywordsFile(KeywordsFile keywordsFile) throws IOException {
        // Json parsing might be successful, but not complete, so check that there are no null fields, and if there are
        // then create empty ones, so we do not get null reference exceptions later
        // TODO: Make all these verifications automatic using reflection - notice that some of them (like locked icon and version) are different

        if (keywordsFile.version == null) {
            Log.e(true, TAG, "No kw version definition in kw file");
            // If there is not even a keywords version then something went horribly wrong, so just create an empty map that will give
            // an experience as if Aniways is not there..
            keywordsFile = new KeywordsFile();
        }
        if (keywordsFile.superCategoriesToIcons == null || (!keywordsFile.version.equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION) && keywordsFile.superCategoriesToIcons.isEmpty())) {
            Log.e(true, TAG, "No super categories to tab icons definition in kw file with version: " + keywordsFile.version);
            keywordsFile.superCategoriesToIcons = new LinkedHashMap<String, String>();
        }
        if (keywordsFile.primaryPhrases == null || (!keywordsFile.version.equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION) && keywordsFile.primaryPhrases.isEmpty())) {
            Log.e(true, TAG, "No primary phrases definition in kw file with version: " + keywordsFile.version);
            keywordsFile.primaryPhrases = new HashMap<Long, String>();
        }
        // This can actually be empty..
        if (keywordsFile.lockedIcons == null /*|| (keywordsFile.version != AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION && keywordsFile.lockedIcons.isEmpty())*/) {
            Log.w(keywordsFile.lockedIcons == null ? true : false, TAG, "No locked icons definition in kw file with version: " + keywordsFile.version);
            keywordsFile.lockedIcons = new long[0];
        }
        if (keywordsFile.phrasesToIcons == null || (!keywordsFile.version.equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION) && keywordsFile.phrasesToIcons.isEmpty())) {
            Log.e(true, TAG, "No phrases to icons definition in kw file with version: " + keywordsFile.version);
            keywordsFile.phrasesToIcons = new HashMap<String, HashMap<String, Long[]>>();
        }
        if (keywordsFile.iconFamilies == null || (!keywordsFile.version.equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION) && keywordsFile.iconFamilies.isEmpty())) {
            Log.e(true, TAG, "No families definition in kw file with version: " + keywordsFile.version);
            keywordsFile.iconFamilies = new HashMap<String, ArrayList<Long>>();
        }
        if (keywordsFile.iconIdsToEmojiUnicodes == null || (!keywordsFile.version.equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION) && keywordsFile.iconIdsToEmojiUnicodes.isEmpty())) {
            Log.e(true, TAG, "No icon ids to emojis definition in kw file with version: " + keywordsFile.version);
            keywordsFile.iconIdsToEmojiUnicodes = new HashMap<Long, String>();
        }
        if (keywordsFile.superCategoriesToIconFamilies == null || (!keywordsFile.version.equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION) && keywordsFile.superCategoriesToIconFamilies.isEmpty())) {
            Log.e(true, TAG, "No super categories to icon families definition in kw file with version: " + keywordsFile.version);
            keywordsFile.superCategoriesToIconFamilies = new LinkedHashMap<String, LinkedHashMap<String, Long[]>>();
        }

        if (keywordsFile.phrasesToCommercialIcons == null ) {
            Log.e(true, TAG, "No phrases to commercial icons defined in kw file with version: " + keywordsFile.version);
            keywordsFile.phrasesToCommercialIcons = new HashMap<String, Long>();
        }

        if (keywordsFile.phraseToAnimatedGifs == null ) {
            Log.e(true, TAG, "No phrase To Animated Gifs defined in kw file with version: " + keywordsFile.version);
            keywordsFile.phraseToAnimatedGifs = new HashMap<>();
        }

        // Parse the Keywords definitions

        // Parse icons
        if (keywordsFile.iconsCount < 0) {
            keywordsFile.iconsCount = 0;
            for (ArrayList<Long> value : keywordsFile.iconFamilies.values()) {
                keywordsFile.iconsCount += value.size();
            }
        }
        parseIconFamilies(keywordsFile);

        // Parse phrases
        if (keywordsFile.numPhrases < 0) {
            keywordsFile.numPhrases = keywordsFile.phrasesToIcons.size();
        }
        if (keywordsFile.numPhraseChars < 0) {
            keywordsFile.numPhraseChars = keywordsFile.numPhrases * 10;
        }
        parsePhrases(keywordsFile, null, null);

        // Parse primary phrases
        parsePrimaryPhrases(keywordsFile);

        // put locked icons
        parseLockedIcons(keywordsFile);

        // build icon categories to families to icons
        parseSuperCategoriesToIconFamilies(keywordsFile);
        
        // parse commercial phrases
        parseCommercialPhrases(keywordsFile);

        parsePhraseToAnimatedGifs(keywordsFile);

        nullLeftovers(keywordsFile);

        // Set the keywords file
        mKeywordsFile = keywordsFile;
    }

    private void parsePhraseToAnimatedGifs(KeywordsFile keywordsFile) {
        if (keywordsFile.phraseToAnimatedGifs == null || keywordsFile.phraseToAnimatedGifs.isEmpty() ){
            addDefaultAnimatedGifProviderToPhrase();
            return;
        }

        if (mPhrases == null){
            return;
        }

        for (Entry<String, HashMap<String, String[]>> entry : keywordsFile.phraseToAnimatedGifs.entrySet()){
            String key = entry.getKey();
            HashMap<String, String[]> value = entry.getValue();

            Phrase phrase = mPhrases.get(key);

            if (phrase == null){
                continue;
            }

            phrase.animatedGifs = new AnimatedGifsDataForPhrase(value);
        }
    }

    private void addDefaultAnimatedGifProviderToPhrase() {
        if (!AniwaysPrivateConfig.getInstance().addDefaultPhraseToAnimatedGif){
            return;
        }

        if (mPhrases == null){
            return;
        }

        for (Phrase phrase : mPhrases){
            phrase.animatedGifs = AnimatedGifsDataForPhrase.SEARCH_PHRASE_IN_ALL_PROVIDERS;
        }
    }

    private void parseSuperCategoriesToIconFamilies(KeywordsFile keywordsFile) {

        LinkedHashMap<String, LinkedHashMap<String, IconData[]>> oldCategoriesToIconFamilies = new LinkedHashMap<String, LinkedHashMap<String, IconData[]>>();

        for (Entry<String, LinkedHashMap<String, Long[]>> entry : keywordsFile.superCategoriesToIconFamilies.entrySet()) {
            String catName = entry.getKey();
            LinkedHashMap<String, Long[]> familiesToIcons = entry.getValue();
            LinkedHashMap<String, IconData[]> familiesToIconDatas = new LinkedHashMap<String, IconData[]>();
            for (Entry<String, Long[]> entry2 : familiesToIcons.entrySet()) {
                String familyName = entry2.getKey();
                Long[] iconIds = entry2.getValue();
                int numIcons = iconIds.length;
                IconData[] icons = new IconData[numIcons];
                for (int i = 0; i < numIcons; i++) {
                    IconData icon = mIcons.get(iconIds[i].intValue());
                    if (icon == null) {
                        Log.e(true, TAG, "Null icon: " + iconIds[i].intValue());
                    }
                    icons[i] = icon;
                }
                familiesToIconDatas.put(familyName, icons);
            }
            oldCategoriesToIconFamilies.put(catName, familiesToIconDatas);
        }

        // Build the AssetTypes-to-Categories-to-Icons tree.
        for (Entry<String, LinkedHashMap<String, IconData[]>> originalCategoryToFamiliesAndIcons : oldCategoriesToIconFamilies.entrySet()) {
            String catName = originalCategoryToFamiliesAndIcons.getKey();
            LinkedHashMap<String, IconData[]> originalFamiliesToIcons = originalCategoryToFamiliesAndIcons.getValue();
            for (Entry<String, IconData[]> originalFamilyToIcons : originalFamiliesToIcons.entrySet()) {
                IconData[] originalFamilyIcons = originalFamilyToIcons.getValue();

                for (int i = 0; i < originalFamilyIcons.length; i++) {

                    AssetType currentAsset = originalFamilyIcons[i].assetType;
                    LinkedHashMap<String, LinkedHashMap<String, List<IconData>>> categoriesToFamiliesAndIcons = mAssetsCategoriesFamiliesIcons.get(currentAsset);

                    // First time encountering this asset type.
                    if (categoriesToFamiliesAndIcons == null) {
                        categoriesToFamiliesAndIcons = new LinkedHashMap<String, LinkedHashMap<String, List<IconData>>>();
                        LinkedHashMap<String, List<IconData>> recentCategory = new LinkedHashMap<String, List<IconData>>();
                        recentCategory.put("Default", new ArrayList<IconData>());

                        categoriesToFamiliesAndIcons.put("Recent", recentCategory);

                        mAssetsCategoriesFamiliesIcons.put(originalFamilyIcons[i].assetType, categoriesToFamiliesAndIcons);
                    }

                    LinkedHashMap<String, List<IconData>> familiesToIcons = categoriesToFamiliesAndIcons.get(catName);

                    // First time encountering this category, under this asset type.
                    if (familiesToIcons == null) {
                        familiesToIcons = new LinkedHashMap<String, List<IconData>>();

                        categoriesToFamiliesAndIcons.put(catName, familiesToIcons);
                    }

                    String familyName = originalFamilyToIcons.getKey();
                    List<IconData> familyIcons = familiesToIcons.get(familyName);

                    // First time encountering this family, under this category and asset type.
                    if (familyIcons == null) {
                        familyIcons = new ArrayList<IconData>();

                        familiesToIcons.put(familyName, familyIcons);
                    }

                    familyIcons.add(originalFamilyIcons[i]);
                }
            }
        }

        keywordsFile.superCategoriesToIconFamilies = null;
    }

    private void parseLockedIcons(KeywordsFile keywordsFile) {
        for (Long iconId : keywordsFile.lockedIcons) {
            IconData icon = mIcons.get(iconId.intValue());
            icon.isLocked = true;
        }
        keywordsFile.lockedIcons = null;
    }

    private void parsePrimaryPhrases(KeywordsFile keywordsFile) {
        // put primary phrase
        for (Entry<Long, String> entry : keywordsFile.primaryPhrases.entrySet()) {
            Long iconId = entry.getKey();
            String phraseName = entry.getValue();
            if (phraseName == null) {
                Log.w(false, TAG, "Phrase is null for icon id: " + iconId);
                continue;
            }
            Phrase phrase = mPhrases.get(phraseName);
            IconData icon = mIcons.get(iconId.intValue());

            if (icon == null) {
                Log.e(true, TAG, "Null icon found when parsing primary phrases: " + iconId.intValue() + " . Phrase: " + phrase);
                int id = iconId.intValue();
                boolean isEmoji = isEmoji(keywordsFile, id);
                IconData newIcon = IconDataFactory.iconData(iconId, UNKNOWN, keywordsFile, isEmoji);
                mIcons.put(newIcon);
                if (isEmoji) {
                    mEmoji.put(newIcon);
                }          
            }

            if (phrase == null) {
                // This means that either the primary phrase is not a phrase of the icon for some reason,
                // or that it is not present in the json because the icon is not enabled for contextual interaction
                // Either way, we create the phrase here and assign only to the primary phrase and not to the
                // general phrases pool in order not to make it available in contextual engine
                phrase = new Phrase(phraseName, icon);
            }

            icon.primaryPhrase = phrase;
        }
    }

    private void parseCommercialPhrases(KeywordsFile keywordsFile) {
        if (noCommercialDatasToParse(keywordsFile))
            return;

        int commercialPhrasesLength = keywordsFile.numOfCommercialPhrases;

        mCommercialIcons = new IconsHashSet(keywordsFile.commercialIconsCount);
        mCommercialPhrases = new PhrasesHashSet(commercialPhrasesLength);
        mCommercialChars = new Chars();
        mCommercialChars.chars = new char[commercialPhrasesLength * 20];

        for (Entry<String, Long> entry : keywordsFile.phrasesToCommercialIcons.entrySet()) {
            parseCommercialPhrase(keywordsFile, entry);
        }
    }

    private void parseCommercialPhrase(KeywordsFile keywordsFile, Entry<String, Long> commercialPharse) {
        String phraseName = commercialPharse.getKey();
        Long iconId = commercialPharse.getValue();

        //Todo: ask if commercial icons are separate from mIcons - for now assume not
        IconData icon = mCommercialIcons.get(iconId.intValue());
        if (icon == null) {
            //Log.e(true, TAG, "Null commercial icon: " + iconId.intValue());
            boolean isEmoji = isEmoji(keywordsFile, iconId.intValue());
            icon = IconDataFactory.iconData(iconId, UNKNOWN, keywordsFile, isEmoji);
            mCommercialIcons.put(icon);
        }

        int nameStartInArray = mCommercialChars.size;
        int nameLength = phraseName.length();

        mCommercialChars.add(phraseName);

        Phrase p = new Phrase(mCommercialChars, nameStartInArray, nameLength, new IconData[]{icon});
        mCommercialPhrases.put(p);
    }

    private boolean noCommercialDatasToParse(KeywordsFile keywordsFile) {
        return keywordsFile.numOfCommercialPhrases <= 0 || keywordsFile.commercialIconsCount <= 0;
    }

    private void parsePhrases(KeywordsFile keywordsFile, JsonReader jsonReader, Gson gson) throws IOException {
        mChars = new Chars();

        if (keywordsFile.numPhrases < 0) {
            throw new IllegalArgumentException("numPhrases < 0");
        }
        if (keywordsFile.numPhraseChars < 0) {
            throw new IllegalArgumentException("numPhraseChars < 0");
        }

        mPhrases = new PhrasesHashSet(keywordsFile.numPhrases);

        // TODO: Use this when the numPhraseChars in the json is correct
        //mChars.chars = new char[keywordsFile.numPhraseChars];
        mChars.chars = new char[keywordsFile.numPhrases * 10];

        if (jsonReader == null) {
            for (Entry<String, HashMap<String, Long[]>> entry : keywordsFile.phrasesToIcons.entrySet()) {
                String phraseName = entry.getKey();
                HashMap<String, Long[]> subPhrasesToIcons = entry.getValue();
                parsePhrase(keywordsFile, phraseName, subPhrasesToIcons);
            }
            keywordsFile.phrasesToIcons = null;
            // Trim mChars
            mChars.trim();
            Log.i(TAG, "Num of chars: " + mChars.size);
        } else {
            jsonReader.beginObject();
            while (jsonReader.hasNext()) {
                String phraseName = jsonReader.nextName();
                HashMap<String, Long[]> subPhrasesToIcons = gson.fromJson(jsonReader, new TypeToken<HashMap<String, Long[]>>() {
                }.getType());
                parsePhrase(keywordsFile, phraseName, subPhrasesToIcons);
            }
            jsonReader.endObject();

            // TODO: when the chars array will be the correct size from the get go then no longer need this
            mChars.trim();
            Log.i(TAG, "Num of chars: " + mChars.size);
        }
    }

    private void parsePhrase(KeywordsFile keywordsFile, String phraseName,
                             HashMap<String, Long[]> subPhrasesToIcons) {
        if (subPhrasesToIcons != null && !subPhrasesToIcons.isEmpty()) {
            for (Entry<String, Long[]> entry2 : subPhrasesToIcons.entrySet()) {
                String subPhrase = entry2.getKey();
                Long[] iconIds = entry2.getValue();
                int numIcons = iconIds.length;
                IconData[] icons = new IconData[numIcons];
                // Build icons array

                for (int i = 0; i < numIcons; i++) {
                    IconData icon = mIcons.get(iconIds[i].intValue());
                    if (icon == null) {
                        Log.e(true, TAG, "Null icon: " + iconIds[i].intValue());
                        int id = iconIds[i].intValue();
                        boolean isEmoji = isEmoji(keywordsFile, id);
                        icon = IconDataFactory.iconData(iconIds[i], UNKNOWN, keywordsFile, isEmoji);
                    }
                    icons[i] = icon;
                }
                int nameStartInArray = mChars.size;
                int nameLength = phraseName.length();
                mChars.add(phraseName);
                Phrase p = null;
                if (subPhrase.length() == 0) {
                    p = new Phrase(mChars, nameStartInArray, nameLength, icons);
                } else {
                    int startOfSubPhrase = phraseName.indexOf(subPhrase);
                    if (startOfSubPhrase == -1) {
                        Log.e(true, TAG, "Start index of subphrase < -1: name: " + phraseName + " . ptr: " + subPhrase);
                        continue;
                    }
                    int lengthOfSubphrase = subPhrase.length();
                    if (startOfSubPhrase == 0 && lengthOfSubphrase == nameLength) {
                        Log.w(true, TAG, "Discovered subphrase equals to phrase. Name: " + phraseName + " . Sub: " + subPhrase);
                        p = new Phrase(mChars, nameStartInArray, nameLength, icons);
                    } else {
                        p = new PhraseWithPartToReplace(mChars, nameStartInArray, nameLength, startOfSubPhrase, lengthOfSubphrase, icons);
                    }
                }
                mPhrases.put(p);
            }
        } else {
            if (!phraseName.equals(REVERT_TO_TEXT_ICON_NAME)) {
                // No need to check if KW version is 0.0 here because if it was we would have no entries and we would not be here..
                Log.w(true, TAG, "No icons for phrase with name: " + phraseName + " in kw file with version: " + keywordsFile.version);
            }
        }
    }

    private boolean isEmoji(KeywordsFile keywordsFile, int id) {
        return id >= keywordsFile.emojiMinIndex && id <= keywordsFile.emojiMaxIndex;
    }

    private void parseIconFamilies(KeywordsFile keywordsFile) {
        if (keywordsFile.iconsCount < 0) {
            throw new IllegalArgumentException("icons count < 0");
        }

        mIcons = new IconsHashSet(keywordsFile.iconsCount);
        mEmoji = new EmojiHashSet(keywordsFile.numberOfEmojis);
        mEmojiWithVarientSelector = new EmojiHashSet(keywordsFile.numberOfEmojis);

        for (Entry<String, ArrayList<Long>> entry : keywordsFile.iconFamilies.entrySet()) {
            String familyName = entry.getKey();
            for (Long icon : entry.getValue()) {
                int id = icon.intValue();
                boolean isEmoji = isEmoji(keywordsFile, id);
                IconData newIcon = IconDataFactory.iconData(icon, familyName, keywordsFile, isEmoji);
                mIcons.put(newIcon);
                if (isEmoji) {
                    mEmoji.put(newIcon);
                }
            }
        }

        keywordsFile.iconFamilies = null;
    }

    static String getVersionFromStream(InputStream stream) {
        String result = null;

        try {
            JsonReader jsonReader = new JsonReader(new InputStreamReader(stream, "UTF-8"));
            jsonReader.beginObject();
            //stay in loop as long as there are more data elements
            while (jsonReader.hasNext()) {
                //get the element name
                String name = jsonReader.nextName();

                if (name.equals("version")) {
                    result = jsonReader.nextString();
                    Log.d(TAG, "Discovered version: " + result);
                }
                break;
            }
            //end reader and close the stream
            //jsonReader.endObject();
            jsonReader.close();
        } catch (Throwable ex) {
            Log.e(true, TAG, "Caught exception while trying to get version from stream", ex);
        }
        return result;
    }

    /**
     * This is the order of the json:
     * version
     * iconsCount
     * numberOfEmojis = -1;
     * emojiMinIndex = -1;
     * emojiMaxIndex = -1;
     * iconsToExternalApps
     * iconsToExternalWebsite
     * iconIdsToEmojiUnicodes
     * animatedGifs
     * iconFamilies
     * numPhrases
     * numPhraseChars
     * phrasesToIcons
     * primaryPhrases
     * lockedIcons
     * superCategoriesToIconFamilies
     * superCategoriesToIcons
     * superCategoriesToFamiliesOrder
     * superCategoriesOrder
     * interactivePhrases
     *
     * @param stream
     * @throws IOException
     */
    private void parseStream(InputStream stream) throws IOException {
        Gson gson = new Gson();
        KeywordsFile file = new KeywordsFile();
        //create a new JSON reader from the response input stream
        JsonReader jsonReader = new JsonReader(new InputStreamReader(stream, "UTF-8"));
        //begin parsing
        jsonReader.beginObject();
        //stay in loop as long as there are more data elements
        while (jsonReader.hasNext()) {
            //get the element name
            String name = jsonReader.nextName();

            if (name.equals("version")) {
                file.version = jsonReader.nextString();
            } else if (name.equals("iconsCount")) {
                file.iconsCount = jsonReader.nextInt();
            } else if (name.equals("numberOfEmojis")) {
                file.numberOfEmojis = jsonReader.nextInt();
            } else if (name.equals("emojiMinIndex")) {
                file.emojiMinIndex = jsonReader.nextLong();
            } else if (name.equals("emojiMaxIndex")) {
                file.emojiMaxIndex = jsonReader.nextLong();
            } else if (name.equals("iconsToExternalApps")) {
                file.iconsToExternalApps = gson.fromJson(jsonReader, new TypeToken<HashMap<Long, String>>() {
                }.getType());
            } else if (name.equals("iconsToExternalWebsite")) {
                file.iconsToExternalWebsite = gson.fromJson(jsonReader, new TypeToken<HashMap<Long, String>>() {
                }.getType());
            } else if (name.equals("iconIdsToEmojiUnicodes")) {
                file.iconIdsToEmojiUnicodes = gson.fromJson(jsonReader, new TypeToken<HashMap<Long, String>>() {
                }.getType());
            } else if (name.equals("animatedGifs")) {
                // Since Gson jas a problem with HashSet<Long> we pass through an intermidiary of an array list
                ArrayList<Long> temp = gson.fromJson(jsonReader, new TypeToken<ArrayList<Long>>() {
                }.getType());
                file.animatedGifs.addAll(temp);
            } else if (name.equals("iconFamilies")) {
                file.iconFamilies = gson.fromJson(jsonReader, new TypeToken<HashMap<String, ArrayList<Long>>>() {
                }.getType());
                // This parses the families and nulls whatever can be nulled in it..
                parseIconFamilies(file);
            } else if (name.equals("numPhrases")) {
                file.numPhrases = jsonReader.nextInt();
            } else if (name.equals("numPhraseChars")) {
                file.numPhraseChars = jsonReader.nextInt();
            } else if (name.equals("commercialIconsCount")) {
                file.commercialIconsCount = jsonReader.nextInt();
            } else if (name.equals("numOfCommercialPhrases")) {
                file.numOfCommercialPhrases = jsonReader.nextInt();
            } else if (name.equals("phrasesToIcons")) {
                parsePhrases(file, jsonReader, gson);
            } else if (name.equals("primaryPhrases")) {
                file.primaryPhrases = gson.fromJson(jsonReader, new TypeToken<HashMap<Long, String>>() {
                }.getType());
                parsePrimaryPhrases(file);
            } else if (name.equals("lockedIcons")) {
                file.lockedIcons = gson.fromJson(jsonReader, new TypeToken<long[]>() {
                }.getType());
                parseLockedIcons(file);
            } else if (name.equals("superCategoriesToIconFamilies")) {
                file.superCategoriesToIconFamilies = gson.fromJson(jsonReader, new TypeToken<LinkedHashMap<String, LinkedHashMap<String, Long[]>>>() {
                }.getType());
                parseSuperCategoriesToIconFamilies(file);
            } else if (name.equals("superCategoriesToIcons")) {
                file.superCategoriesToIcons = gson.fromJson(jsonReader, new TypeToken<LinkedHashMap<String, String>>() {
                }.getType());
            } else if (name.equals("phrasesToCommercialIcons")) {
                file.phrasesToCommercialIcons = gson.fromJson(jsonReader, new TypeToken<HashMap<String,Long>>(){}.getType());
                parseCommercialPhrases(file);
            }else if (name.equals("phraseToAssetIds")) {
                file.phraseToAnimatedGifs = gson.fromJson(jsonReader, new TypeToken<HashMap<String, HashMap<String, String[]>>>(){}.getType());
                parsePhraseToAnimatedGifs(file);
            } else {
                jsonReader.skipValue();
            }
            // TODO: After this we have interactivePhrases
        }
        //end reader and close the stream
        jsonReader.endObject();
        jsonReader.close();
        stream.close();
        nullLeftovers(file);
        // Set the keywords file
        mKeywordsFile = file;
    }

    private void nullLeftovers(KeywordsFile file) {
    	file.primaryPhrases = null;
        file.iconIdsToEmojiUnicodes = null;
    	file.animatedGifs = null;
        file.iconsToExternalApps = null;
        file.iconsToExternalWebsite = null;
        file.phrasesToCommercialIcons = null;
	}

    @SuppressWarnings("unchecked")
	private KeywordsFile parseManually(String jsontext) throws JSONException {
        Gson gson = new Gson();

        KeywordsFile file = null;
        Log.d(TAG, "Start manual parse");
        JSONObject obj = new JSONObject(jsontext);
        file = new KeywordsFile();
        file.version = obj.getString("version");
        file.numberOfEmojis = obj.optInt("numberOfEmojis", 845);
        file.numPhraseChars = obj.optInt("numPhraseChars", -1);
        file.numPhrases = obj.optInt("numPhrases", -1);
        file.iconsCount = obj.optInt("iconsCount", -1);
        file.emojiMinIndex = obj.optLong("emojiMinIndex", 1000);
        file.emojiMaxIndex = obj.optLong("emojiMaxIndex", 2999);
        file.commercialIconsCount = obj.optInt("commercialIconsCount", -1);
        file.numOfCommercialPhrases = obj.optInt("numOfCommercialPhrases", -1);

        // Get icon families
        JSONObject families = obj.getJSONObject("iconFamilies");
        Iterator<String> keys = families.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            JSONArray icons = families.getJSONArray(key);
            ArrayList<Long> ids = new ArrayList<Long>(icons.length());
            for (int i = 0; i < icons.length(); i++) {
                ids.add(icons.getLong(i));
            }
            ids.trimToSize();
            file.iconFamilies.put(key, ids);
        }

        // Get iconIdsToEmojiUnicodes
        JSONObject iconIdsToEmojiUnicodes = obj.getJSONObject("iconIdsToEmojiUnicodes");
        keys = iconIdsToEmojiUnicodes.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            String value = iconIdsToEmojiUnicodes.getString(key);

            file.iconIdsToEmojiUnicodes.put(Long.decode(key), value);
        }

        // Get lockedIcons
        JSONArray lockedIcons = obj.getJSONArray("lockedIcons");
        int numLockedIcons = lockedIcons.length();
        file.lockedIcons = new long[numLockedIcons];
        for (int i = 0; i < numLockedIcons; i++) {
            file.lockedIcons[i] = lockedIcons.getLong(i);
        }

        // Get animated gifs
        JSONArray animatedGifs = obj.getJSONArray("animatedGifs");
        int numAnimatedGifs = animatedGifs.length();
        for (int i = 0; i < numAnimatedGifs; i++) {
            file.animatedGifs.add(animatedGifs.getLong(i));
        }

        // Get phrasesToIcons
        JSONObject phrasesToIcons = obj.getJSONObject("phrasesToIcons");
        keys = phrasesToIcons.keys();
        while (keys.hasNext()) {
            String phraseName = keys.next();
            JSONObject subphrasesToIcons = phrasesToIcons.optJSONObject(phraseName);
            if (subphrasesToIcons == null) {
                Log.e(true, TAG, "Found a phrase with no definition: " + phraseName);
                continue;
            }
            HashMap<String, Long[]> map = new HashMap<String, Long[]>();
            Iterator<String> keys2 = subphrasesToIcons.keys();

            while (keys2.hasNext()) {
                String subphraseName = keys2.next();
                JSONArray icons = subphrasesToIcons.getJSONArray(subphraseName);

                Long[] ids = new Long[icons.length()];
                for (int i = 0; i < icons.length(); i++) {
                    ids[i] = (icons.getLong(i));
                }

                map.put(subphraseName, ids);
            }
            file.phrasesToIcons.put(phraseName, map);
        }

        // Get primaryPhrases
        JSONObject primaryPhrases = obj.getJSONObject("primaryPhrases");
        keys = primaryPhrases.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            String value = primaryPhrases.optString(key);

            file.primaryPhrases.put(Long.decode(key), value);
        }

        // Get icons to external apps
        JSONObject externalApps = obj.getJSONObject("iconsToExternalApps");
        keys = externalApps.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            String value = externalApps.optString(key);

            file.iconsToExternalApps.put(Long.decode(key), value);
        }

        // Get icons to external websites
        JSONObject externalWebsites = obj.getJSONObject("iconsToExternalWebsite");
        keys = externalWebsites.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            String value = externalWebsites.optString(key);

            file.iconsToExternalWebsite.put(Long.decode(key), value);
        }

        // Get superCategoriesToIcons
        JSONObject superCategoriesToIcons = obj.getJSONObject("superCategoriesToIcons");
        JSONArray superCategoriesOrdered = obj.getJSONArray("superCategoriesOrder");
        for(int i = 0; i< superCategoriesOrdered.length(); i++) {
            String key = superCategoriesOrdered.getString(i);
            String value = superCategoriesToIcons.getString(key);

            file.superCategoriesToIcons.put(key, value);
        }

        // Get superCategoriesToIconFamilies
        JSONObject superCategoriesToIconFamilies = obj.getJSONObject("superCategoriesToIconFamilies");
        for(int i = 0; i< superCategoriesOrdered.length(); i++) {
            String catName = superCategoriesOrdered.getString(i);
            JSONObject familiesToIcons = superCategoriesToIconFamilies.getJSONObject(catName);
            LinkedHashMap<String, Long[]> familyToIcons = new LinkedHashMap<String, Long[]>();

            JSONObject superCategoriesToFamiliesOrder = obj.getJSONObject("superCategoriesToFamiliesOrder");
            JSONArray familiesOrder = superCategoriesToFamiliesOrder.getJSONArray(catName);

            for(int k = 0; k< familiesOrder.length(); k++) {
                String familyName = familiesOrder.getString(k);
                JSONArray icons = familiesToIcons.getJSONArray(familyName);

                Long[] ids = new Long[icons.length()];
                for (int j = 0; j < icons.length(); j++) {
                    ids[j] = (icons.getLong(j));
                }

                familyToIcons.put(familyName, ids);
            }
            file.superCategoriesToIconFamilies.put(catName, familyToIcons);
        }

        // Get phrasesToCommercialIcons
        JSONObject phrasesToCommercialIcons = obj.optJSONObject("phrasesToCommercialIcons");
        if (phrasesToCommercialIcons != null){
            keys = phrasesToCommercialIcons.keys();
            while (keys.hasNext()) {
                String phraseName = keys.next();
                Long iconId = phrasesToCommercialIcons.optLong(phraseName);
                file.phrasesToCommercialIcons.put(phraseName,iconId);
            }
        }

        // Get phrasesToAnimatedGifs
        JSONObject phrasesToAnimatedGifs = obj.getJSONObject("phraseToAssetIds");
        if (phrasesToAnimatedGifs != null){
            keys = phrasesToAnimatedGifs.keys();
            while (keys.hasNext()) {
                String phraseName = keys.next();
                JSONObject provider = phrasesToAnimatedGifs.getJSONObject(phraseName);

                Iterator<String> innerKeys = provider.keys();
                HashMap<String,String[]> providerToAssetsIds = new HashMap<>();
                while (innerKeys.hasNext()){
                    String providerName = keys.next();
                    String[] ids = gson.fromJson(provider.getJSONArray(providerName).toString(),new TypeToken<ArrayList<String>>(){}.getType());

                    providerToAssetsIds.put(providerName,ids);
                }

                file.phraseToAnimatedGifs.put(phraseName, providerToAssetsIds);
            }
        }

        Log.d(TAG, "End manual parse");
        return file;
    }

    public Phrase getPhraseByName(String name) {
        return mPhrases.get(name);
    }

    public PhrasesHashSet getCommercialPhrases() {
        return mCommercialPhrases;
    }

    public String getKeywordsVersion() {
        return mKeywordsFile.version;
    }

    public AssetType[] getAssetTypesInOrder() {
        Set<AssetType> assetTypes = this.mAssetsCategoriesFamiliesIcons.keySet();
        AssetType[] retAssetTypes = new AssetType[assetTypes.size()];
        assetTypes.toArray(retAssetTypes);

        return retAssetTypes;
    }

    public String[] getIconCategoriesInOrderByAssetType(AssetType assetType) {
        LinkedHashMap<String, LinkedHashMap<String, List<IconData>>> categoriesToIcons = this.mAssetsCategoriesFamiliesIcons.get(assetType);

        if (categoriesToIcons == null) {
            return new String[0];
        }

        Set<String> categories = categoriesToIcons.keySet();
        String[] result = new String[categories.size()];
        return categories.toArray(result);
    }

    public String getSuperCategoryTabIconName(String category) {
        return mKeywordsFile.superCategoriesToIcons.get(category);
    }

    public HashMap<String, List<IconData>> getFamiliesToIconsByAssetTypeAndCategory(AssetType assetType, String category) {
        LinkedHashMap<String, LinkedHashMap<String, List<IconData>>> categoriesToFamiliesAndIcons = mAssetsCategoriesFamiliesIcons.get(assetType);

        if (categoriesToFamiliesAndIcons == null) {
            return new HashMap<String, List<IconData>>();
        }

        return categoriesToFamiliesAndIcons.get(category);
    }

    public ArrayList<String> getFamiliesInOrderForSuperCategory(AssetType assetType, String superCategory) {

        ArrayList<String> result = new ArrayList<String>();

        LinkedHashMap<String, LinkedHashMap<String, List<IconData>>> categoriesToFamiliesAndIcons = mAssetsCategoriesFamiliesIcons.get(assetType);

        if (categoriesToFamiliesAndIcons != null) {
            LinkedHashMap<String, List<IconData>> familiesToIcons = categoriesToFamiliesAndIcons.get(superCategory);

            if (familiesToIcons != null) {
                result.addAll(familiesToIcons.keySet());
            }
        }

        return result;
    }

    public IconData getIconById(int id) {
        return mIcons.get(id);
    }

    public IconData getEmoji(int unicode) {
        return mEmoji.get(unicode);
    }

    public IconData getEmoji(int[] unicodes) {
        return mEmoji.get(unicodes);
    }

    public IconData getEmojiWithVarientSelector(int unicode) {
        return mEmojiWithVarientSelector.get(unicode);
    }

    public IconData getEmojiWithVarientSelector(int[] unicodes) {
        return mEmojiWithVarientSelector.get(unicodes);
    }

    public void addEmojiWithVarientSelector(IconData icon) {
        mEmojiWithVarientSelector.put(icon);
    }

    private String loadTextFile(InputStream inputStream) throws IOException {
        ByteArrayOutputStream byteStream = new ByteArrayOutputStream();
        byte[] bytes = new byte[4096];
        int len = 0;
        while ((len = inputStream.read(bytes)) > 0)
            byteStream.write(bytes, 0, len);
        inputStream.close();
        return new String(byteStream.toByteArray(), "UTF-8");
    }

    private void init() {
        mPhrases = new PhrasesHashSet(0);
        mIcons = new IconsHashSet(0);
        mEmoji = new EmojiHashSet(0);
        mEmojiWithVarientSelector = new EmojiHashSet(0);
        mChars = new Chars();
        mAssetsCategoriesFamiliesIcons = new LinkedHashMap<AssetType, LinkedHashMap<String, LinkedHashMap<String, List<IconData>>>>();
        mKeywordsFile = new KeywordsFile();
    }

    public boolean isEmpty() {
        boolean result = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION.equals(this.getKeywordsVersion());
        return result;
    }
    
    void setKeywordsVersion(String newVersion) {
        this.mKeywordsFile.version = newVersion;

    }
}
