/**
 * 
 */
package com.aniways.data;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.os.Environment;

import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.service.helper.KeywordsHelper;
import com.aniways.service.utils.AniwaysServiceUtils;

/**
 * @author Shai
 * This class is transactional - it represents either the external or internal storage throughout its lifetime. If the storage medium changes, need a new instance of this class.
 * Therefore, all methods in the chain of a storage operation (for example, downloading keywords) need to chain the same instance of this class to each other.
 * When a new version of keywords is downloaded then need to call recordLocationOfNewKeywords() in order for future instances to know where the latest version
 * is and deal with it accordingly.
 */
public class AniwaysStorageManager {
	public static final String NO_MEDIA = ".nomedia";

	private static final String TAG = "AniwaysStorageManager";
	private static final String CNOFIG_DIR = "/aniways/config";
	private static final String CNOFIG_FILE = "configuration";
	private static String sPackageName;	
	private static Context sContext = null;
	private static AniwaysStorageManager sInstance;
	private static boolean sIsInit;

	private boolean mExternalStorageReadable = false; 
	private boolean mExternalStorageWriteable = false;
	private boolean mUsingExternal;
	private File mKeywordsCacheDir;
	private File mKeywordsDir;

	private File mExternalRootCacheDir;
	private File mExternalRootFilesDir;
	private File mInternalRootCacheDir;
	private File mInternalRootFilesDir;

	private Context mContext;

	private boolean notAllDirsInitialized;

	static void forceInit(Context applicationContext) {
		if(isInit()){
			Log.e(true, TAG, "Calling init of AniwaysStorageManager more than once");
		}

		sContext = applicationContext;
		if(sContext.getApplicationInfo() == null || sContext.getApplicationInfo().packageName == null){
			Log.e(true, TAG, "Could not resolve package name. Using app ID");
			sPackageName = AniwaysPrivateConfig.getInstance().appId;
		}
		else{
			sPackageName = sContext.getApplicationInfo().packageName;
		}

		// Delete files from old location names and reset storage storage..
		// (in first version the aniways assets didn't sit in an 'aniways' folder - which means that there could be collisions with app assets
		resetStorageIfNamesFromOldVersionExist();

		// Create the instance to use
		onUseExternalStorageConfigurationChange(sContext);
		
		setInit();
	}
	
	private static boolean isInit(){
		return sIsInit;
	}
	
	private static void setInit(){
		sIsInit = true;
	}

	/**
	 * @param isService
	 */
	public static void onUseExternalStorageConfigurationChange(Context context) {
		AniwaysStorageManager instance = new AniwaysStorageManager(context); // The service needs writable storage, the app doesn't
		setInstance(instance);
	}

	public static AniwaysStorageManager getInstance(Context context){
		AniwaysStorageManager instance = sInstance;

		// This is a hack!!!
		// Sometimes when the first instance is initialized early then not all storage dirs initialized properly, 
		// so if this happens we init now again in the getter, and hope that it will not happen again
		if(instance.notAllDirsInitialized){
			instance = new AniwaysStorageManager(context);
			if(instance.notAllDirsInitialized){
				Log.e(true, TAG, "Not all dirs initialized after creating new instance again");
			}
			else{
				Log.w(true, TAG, "All dirs initialized only after creating new instance again");
			}
			setInstance(instance);
		}

		return instance;
	}

	/**
	 * If the operation succeeds then the directory exists as a non media directory
	 * @return
	 * @throws java.io.IOException
	 */
	public File getKeywordsDir() throws IOException{
		createNonMediaDirectoryIfDoesntExist(mKeywordsDir);
		return this.mKeywordsDir;
	}

	/**
	 * If the operation succeeds then the directory exists as a non media directory
	 * @return
	 * @throws java.io.IOException
	 */
	public File getKeywordsCacheDir() throws IOException{
		createNonMediaDirectoryIfDoesntExist(mKeywordsCacheDir);
		return this.mKeywordsCacheDir;
	}

	private AniwaysStorageManager(Context context){
		mContext = context;

		// Generate file pointers to storage dirs according to config, external storage availability, and where icons are currently present.
		determineUseInternalOrExternal();
		Log.i(TAG, "Is app installed on sd card:" + isAppInstalledOnSdCard() + " . Use external: " + this.mUsingExternal);
		generateStorageDirs();

		// If the app is installed on the external storage, and it is configured to use it, and it is empty - but if there are already files on the internal storage, then move them to the external storage.
		// Else, if the internal storage in empty - but there are already files on the external storage, then move them to the internal storage.
		boolean result = true;
		try{
			result = this.moveFilesToInternalOrExternalIfNecessary();
		}
		catch(IOException ex){
			result = false;
			Log.e(true, TAG, "Moving file to or from external storage failed", ex);
		}
		if(result){
			// Mark for next time that the keywords are according to where this instance is pointing to
			this.recordLocationOfNewKeywords();
		}
		if(!result){
			// If the moving of files failed, then need to reset the Keywords version, so it will be downloaded again to the correct location
			resetKeywordsVersions();
		}
	}

	private static void setInstance(AniwaysStorageManager instance){
		sInstance = instance; 
	}

	private static void resetKeywordsVersions() {
		KeywordsHelper keywordsHelper = KeywordsHelper.getInstance();
		keywordsHelper.setKeywordsVersion(sContext, AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION, "");
		AniwaysServiceUtils.setLastSuccessfulUpdate(sContext, 0);
		removeAreEmoticonsStoredOnExternalStorageKey();
	}

	private boolean moveFilesToInternalOrExternalIfNecessary() throws IOException{
		boolean success = true;

		Boolean areStoredOnExternal = getAreEmoticonsStoredOnExternalStorage();
		if (areStoredOnExternal == null){
			// there is no data from previous runs. Can be that:
			// 1. This is a first run of the application
			// 2. No version has finished downloading yet.
			// 3. A move operation failed and the data was reset
			// 4. This is a first run after and upgrade to a version that writes this data
			// Look at the 
			// TODO: this is not correct, there is no guarenty that files are indeed on internal, and worse - if the service is running and downloading files to the external storage
			// then the moving of files will cause a disruption. Need to only move the files if the app is restarting after an upgrade or after move from external to internal, etc.
			areStoredOnExternal = false;
		}

		// If the files are stored on internal storage and we are using external, then try and move them to external
		//TODO: currently if we use external storage then its available, so its OK
		if(!areStoredOnExternal && this.mUsingExternal){
			//TODO: this condition is only because of the setting of the boolean to false above if it is null. Need to resolve this
			if(this.getStorageDir(false, false).exists()){
				Log.i(TAG, "Move files to external");
				this.moveRecursive(this.getStorageDir(false, false), this.getKeywordsDir(), true);
			}
		}
		// If the files are stored on external and we are using internal then try and move to internal if external storage is available
		else if(areStoredOnExternal && !this.mUsingExternal && this.isExternalStorageReadable()){
			Log.i(TAG, "Move files to internal");
			this.moveRecursive(this.getStorageDir(true, false), this.getKeywordsDir(), true);
		}

		return success;

	}

	/**
	 * If the app is configured to use internal storage then use that one.
	 * If it is configured to try and use external storage, then only use it if the app is installed on the external storage (to make sure that the icons are always available).
	 * TODO: Need to be able to use the external storage if it is available, even if the app is on internal storage (alot of end cases dealing with external storage being available or unavailable while the app is running and also because the service and the app need to be sync'd). Also in such cases storage dirs can be null in current implementation, which is another headache (see generateStorageDir())
	 * @param isWritable
	 */
	private void determineUseInternalOrExternal() {
		if (!AniwaysPrivateConfig.getInstance().tryUsingExternalStorageForIconCaching){
			// Icons are configured to be stored on internal storage
			this.mUsingExternal = false;
			return;
		}

		// If we are here, then the app is configured to try and use external storage

		// If the app is available on the SD card then use this
		if (isAppInstalledOnSdCard()){
			this.mUsingExternal = true;
			return;
		}

		//If the app is not installed on sd card (phone storage, or internal memory) then use internal memory only
		this.mUsingExternal = false;
	}

	/**
	 * @param forWriting
	 */
	private void generateStorageDirs(){
		generateStorageDirRoots();

		mKeywordsCacheDir = getStorageDir(mUsingExternal, true);
		mKeywordsDir = getStorageDir(mUsingExternal, false);
	}

	private void generateStorageDirRoots() {
		mExternalRootCacheDir = getExternalFilesDir(true);
		mInternalRootCacheDir = getInternalFilesDir(true);
		mExternalRootFilesDir = getExternalFilesDir(false);
		mInternalRootFilesDir = getInternalFilesDir(false);

		if(this.mUsingExternal){
			if(mExternalRootCacheDir == null || mExternalRootFilesDir == null){
				Log.e(true, TAG, "One of the root dirs is null. mUsingExternal:" + mUsingExternal + " . cache is null:" + (mExternalRootCacheDir == null) + " . files is null:" + (mExternalRootFilesDir == null));
				this.notAllDirsInitialized = true;
			}
			else{
				this.notAllDirsInitialized = false;
			}
		}
		else{
			if(mInternalRootCacheDir == null || mInternalRootFilesDir == null){
				Log.e(true, TAG, "One of the root dirs is null. mUsingExternal:" + mUsingExternal + " . cache is null:" + (mInternalRootCacheDir == null) + " . files is null:" + (mInternalRootFilesDir == null));
				this.notAllDirsInitialized = true;
			}
			else{
				this.notAllDirsInitialized = false;
			}
		}
	}

	private File getStorageDir(boolean useExternal, boolean isCache){
		File dir = getStorageDirRoot(useExternal, isCache);
		if(dir != null){
			return new File(dir.getAbsolutePath() + AniwaysServiceUtils.KEYWORDS_DIR);
		}
		//TODO: this currently should never happen because we only use external storage if the app is installed there, so its always available.
		//		if we support using external storage when the app is on internal, need to deal with this on the upper levels that use the results of this method!!!!

		Log.e(true, TAG, "Returning null storage folder!! useExternal: " + useExternal + " isCache: " + isCache);
		this.notAllDirsInitialized = true;
		return null;
	}

	private File getStorageDirRoot(boolean isExternal, boolean isCache){
		if(isExternal){
			return isCache ? mExternalRootCacheDir : mExternalRootFilesDir;
		}
		return isCache ? mInternalRootCacheDir : mInternalRootFilesDir;
	}

	// If targetLocation does not exist, it will be created.
	// Copies sub dirs and files
	private void copyRecursive(File sourceLocation , File targetLocation) throws Throwable {

		if(!sourceLocation.exists()){
			throw new FileNotFoundException(sourceLocation.getAbsolutePath());
		}

		if (sourceLocation.isDirectory()) {
			if (!targetLocation.exists() && !targetLocation.mkdirs()) {
				throw new IOException("Cannot create dir " + targetLocation.getAbsolutePath());
			}

			String[] children = sourceLocation.list();
			for (int i=0; i<children.length; i++) {
				copyRecursive(new File(sourceLocation, children[i]), new File(targetLocation, children[i]));
			}
		} else {

			// make sure the directory we plan to store the recording in exists
			File directory = targetLocation.getParentFile();
			if (directory != null && !directory.exists() && !directory.mkdirs()) {
				throw new IOException("Cannot create dir " + directory.getAbsolutePath());
			}

			InputStream in = null;
			OutputStream out = null;

			try{
				in = new FileInputStream(sourceLocation);
				out = new FileOutputStream(targetLocation);

				// Copy the bits from instream to outstream
				byte[] buf = new byte[1024];
				int len;
				while ((len = in.read(buf)) > 0) {
					out.write(buf, 0, len);
				}
				out.flush();
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Error copying file from " + sourceLocation + " to " +  targetLocation, ex);
				throw ex;
			}
			finally{
				try{
					if(in != null){
						in.close();
					}
				}
				catch(Throwable ex){
					Log.e(true, TAG, "Error closing input stream while copying file from " + sourceLocation + " to " +  targetLocation, ex);
				}
				finally{
					try{
						if(out != null){
							out.close();
						}
					}
					catch(Throwable ex){
						Log.e(true, TAG, "Error closing output stream while copying file from " + sourceLocation + " to " +  targetLocation, ex);
					}
				}
			}
		}
	}


	private boolean deleteRecursive(File fileOrDirectory){
		try{
			if(!fileOrDirectory.exists()){
				Log.e(true, TAG, "Source file for delete doesnt exist: " + fileOrDirectory.getAbsolutePath());
				return false;
			}
			return DeleteRecursiveInternal(fileOrDirectory);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Failed deleting from: " + fileOrDirectory.getAbsolutePath());		
			return false;
		}
	}

	private boolean DeleteRecursiveInternal(File fileOrDirectory) {
		boolean result = true;
		if (fileOrDirectory.isDirectory()){
			for (File child : fileOrDirectory.listFiles()){
				result &= DeleteRecursiveInternal(child);
			}
		}

		boolean result2 = fileOrDirectory.delete();
		if(!result2){
			Log.e(true, TAG, "Error deleting file or dir: " + fileOrDirectory.getAbsolutePath());
		}
		result &= result2;
		return result;
	}

	private boolean moveRecursive(File sourceLocation, File destLocation, boolean deleteSourceIfFail){
		boolean successCopy = true;
		boolean successDelete = true;
		try{
			copyRecursive(sourceLocation, destLocation);
		}
		catch(Throwable ex){
			successCopy = false;
			Log.e(true, TAG, "Failed copying from: " + sourceLocation.getAbsolutePath() + " to: " + destLocation.getAbsolutePath() + (deleteSourceIfFail ? ". Deleting source" : ""));
		}
		if(successCopy || deleteSourceIfFail){
			successDelete = deleteRecursive(sourceLocation);
			if(!successDelete){
				Log.e(true, TAG, "Failed deleting from: " + sourceLocation.getAbsolutePath() + " after " + (deleteSourceIfFail ? "failed" : "succeeded" + " copying to: " + destLocation.getAbsolutePath()));		
			}
		}

		return successCopy & successDelete;
	}

	@SuppressLint({ "InlinedApi", "SdCardPath" })
	private static boolean isAppInstalledOnSdCard() {
		// check for API level 8 and higher
		if (Utils.isAndroidVersionAtLeast(8)) {
			PackageManager pm = sContext.getPackageManager();
			try {
				PackageInfo pi = pm.getPackageInfo(sContext.getPackageName(), 0);
				ApplicationInfo ai = pi.applicationInfo;
				return (ai.flags & ApplicationInfo.FLAG_EXTERNAL_STORAGE) == ApplicationInfo.FLAG_EXTERNAL_STORAGE;
			} catch (NameNotFoundException e) {
				// ignore
			}
		}

		// check for API level 7 - check files dir
		try {
			String filesDir = sContext.getFilesDir().getAbsolutePath();
			if (filesDir.startsWith("/data/")) {
				return false;
			} else if (filesDir.contains("/mnt/") || filesDir.contains("/sdcard/") ||  filesDir.contains(Environment.getExternalStorageDirectory().getPath()) ) {
				return true;
			}
		} catch (Throwable e) {
			// ignore
		}

		return false;
	}

	/**
	 * @return
	 */
	@SuppressLint("NewApi")
	private File getExternalFilesDir(boolean isCache) {
		if(Utils.isAndroidVersionAtLeast(8)){
			return (isCache ? mContext.getExternalCacheDir() : mContext.getExternalFilesDir(null));
		}
		return new File (Environment.getExternalStorageDirectory()+ "/Android/data/" + sPackageName + "/files" + (isCache ? "/cache" : ""));
	}

	private File getInternalFilesDir(boolean isCache) {
		return (isCache ? mContext.getCacheDir() : mContext.getFilesDir());
	}

	private void updateExternalStorageState() { 
		String state = Environment.getExternalStorageState(); 
		if (Environment.MEDIA_MOUNTED.equals(state)) { 
			mExternalStorageReadable = mExternalStorageWriteable = true; 
		} else if (Environment.MEDIA_MOUNTED_READ_ONLY.equals(state)) { 
			mExternalStorageReadable = true; 
			mExternalStorageWriteable = false; 
		} else { 
			mExternalStorageReadable = mExternalStorageWriteable = false; 
		} 

		Log.i(TAG, "External stogare readable: " + mExternalStorageReadable + ". Writable: " + mExternalStorageWriteable); 
	} 

	private File createNonMediaDirectoryIfDoesntExist(File dir) throws IOException{
		if (!dir.exists()) {
			if(!dir.mkdirs()){
				throw new IOException("Could not create directory: " + dir.getAbsolutePath());
			}
			Log.i(TAG, "Created non media directory:" + dir.getAbsolutePath());
		}

		addNonMediaFileToDirIfDoesntExist(dir.getAbsolutePath());

		return dir;
	}

	private void addNonMediaFileToDirIfDoesntExist(String dirName) throws IOException{
		File nomedia = new File(dirName, NO_MEDIA);
		if(!nomedia.exists()){
			nomedia.createNewFile();
		}
	}

	private boolean isExternalStorageReadable() 
	{ 
		updateExternalStorageState(); 
		return mExternalStorageReadable; 
	} 

	/*
	private boolean isExternalStorageWritable() 
	{ 
		updateExternalStorageState(); 
		return sExternalStorageWriteable; 
	}
	 */

	private static Boolean getAreEmoticonsStoredOnExternalStorage() {
		// open the shared preferences
		SharedPreferences prefs = 
				sContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());

		// return the values that stored there, in case that no value
		// is stored return null
		if(prefs.contains(AniwaysServiceUtils.KEY_ARE_EMOTICONS_STORED_ON_EXTERNAL_STORAGE)){
			return prefs.getBoolean(AniwaysServiceUtils.KEY_ARE_EMOTICONS_STORED_ON_EXTERNAL_STORAGE, false);
		}
		return null;
	}

	//TODO: use this when finish download
	private void setAreEmoticonsStoredOnExternalStorage(boolean areThey) {
		// open the shared preferences
		SharedPreferences prefs = 
				sContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// write the new value
		edit.putBoolean(AniwaysServiceUtils.KEY_ARE_EMOTICONS_STORED_ON_EXTERNAL_STORAGE, areThey);
		edit.commit();
	}

	//TODO: use this when finish download
	private synchronized static void removeAreEmoticonsStoredOnExternalStorageKey() {
		// open the shared preferences
		SharedPreferences prefs = 
				sContext.getSharedPreferences(AniwaysServiceUtils.SHARED_PREFERENCES, Utils.getSharedPreferencesFlags());
		Editor edit = prefs.edit();
		// write the new value
		edit.remove(AniwaysServiceUtils.KEY_ARE_EMOTICONS_STORED_ON_EXTERNAL_STORAGE);
		edit.commit();
	} 

	/**
	 * This is to support the move to adding an 'Aniways' folder for aniways assets.
	 * Users with old names used internal storage only, so its safe to use the File.renameTo() method.
	 */
	private static void resetStorageIfNamesFromOldVersionExist() {
		File oldAssetsDir = new File(sContext.getFilesDir() + AniwaysServiceUtils.OLD_ASSETS_DIR);
		File oldKeywordsFile = new File(sContext.getFilesDir() + "/" + AniwaysServiceUtils.KEYWORDS_FILE);
		if(oldAssetsDir.exists() || oldKeywordsFile.exists()){
			if(oldAssetsDir.exists()){
				deleteDirectory(oldAssetsDir);
			}
			// Set back the keywords and assets version to 0.0 and the last successful update time to 0 so they will be downloaded again
			resetKeywordsVersions();
		}
	}

	private static boolean deleteDirectory(File path) {
		if( path.exists() ) {
			File[] files = path.listFiles();
			for(int i=0; i<files.length; i++) {
				if(files[i].isDirectory()) {
					deleteDirectory(files[i]);
				}
				else {
					files[i].delete();
				}
			}
		}
		return( path.delete() );
	}

	public void recordLocationOfNewKeywords() {
		this.setAreEmoticonsStoredOnExternalStorage(this.mUsingExternal);
	}

	/**
	 * This is static because it is used by the init of the private config, b4 the storage manager is initialized.
	 * For this reason, it uses the internal storage only.
	 * @param context
	 * @return
	 */
	public static String getConfigFilePath(Context context) {
		return getConfigDirPath(context) + "/" + CNOFIG_FILE;
	}

	public static String getConfigDirPath(Context context) {
		return context.getFilesDir().getAbsolutePath() + CNOFIG_DIR;
	}

	public static String getConfigCacheDirPath(Context context) {
		return context.getCacheDir().getAbsolutePath() + CNOFIG_DIR;
	}

	public static String getConfigFileName(){
		return CNOFIG_FILE;
	}
}
