/**
 * 
 */
package com.aniways.data;

import java.lang.Thread.UncaughtExceptionHandler;

import com.aniways.AniwaysNotInitializedException;
import com.aniways.Log;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.blur.BlurRenderer;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.emoticons.button.AniwaysRecentIconsManager;
import com.aniways.service.manager.UpdateManager;
import com.aniways.volley.toolbox.Volley;

import android.content.Context;

/**
 * @author Shai
 *
 */
public class AniwaysStatics {
	private static final String TAG = "AniwaysStatics";

	private static boolean sIsApplicationInit;
	private static boolean sIsServiceInit;
	private static boolean sIsInitializing;
    private static Context sApplicationContext;

	public static void makeSureAniwaysIsInitialized(boolean isBackendService){
		if(!isInit(isBackendService)){
			throw new AniwaysNotInitializedException();
		}
	}

	public static void makeSureBackendSyncThreadIsRunning() {
		// Check if the Backend Sync thread is running, and if not then start it..
		// Need this check in case the thread has stopped for some reason..
		AniwaysBackendSyncChecker.makeSureRunning();
	}

	public static synchronized void init(Context applicationContext, Context context, boolean isBackendService) {
		long startTime = System.currentTimeMillis();

		if(applicationContext == null){
			throw new IllegalArgumentException("context is null");
		}

		// If we are already initialized, then no need to init again..
		// If the app is initialized, then no need to init the service,
		// If the service is initialized, then if it is the app, init only the delta
		if(isInit(isBackendService)){
			android.util.Log.i(TAG, "Not initializing Aniways statics, because it is already initialized. Is service: " + isBackendService);
			return;
		}

		sIsInitializing = true;
        sApplicationContext = applicationContext.getApplicationContext();

		if(!sIsApplicationInit && !sIsServiceInit){

			// Set an Exception handler to report all uncaught Exceptions to analytics
			// TODO: what if the application code sets this default handler to something else (before or later). 
			// If before then there is no problem cause this one chains the calls, 
			// but after, we depend on the new handler to chain the call to us..
			// TODO: what about other threads??..
			UncaughtExceptionHandler existingHandler = Thread.getDefaultUncaughtExceptionHandler();
			// No need to add our exception handler if it already exists here
			if (!(existingHandler instanceof AniwaysCustomExceptionHandler)){
				Thread.setDefaultUncaughtExceptionHandler(new AniwaysCustomExceptionHandler());
			}

			// Init the config
			AniwaysPrivateConfig.forceInit(applicationContext);

			// Init the logs
			Log.forceInit(applicationContext);

			// Init Analytics
			AnalyticsReporter.forceInit(applicationContext, context);

			// Init Storage
			AniwaysStorageManager.forceInit(applicationContext);
		}

		// Starts the Aniways Backend Service
		if (!sIsApplicationInit && !isBackendService){
			// Init the phrase to icons mapping 
			// Its important that the phrase replacement data is init before the backend client because the backend client uses it
			// This will also start the process of parsing the json.
			AniwaysPhraseReplacementData.forceInit(applicationContext);

			// Init the backend sync checker which periodically checks if new config or keywords versions need to be parsed
			AniwaysBackendSyncChecker.forceInit(applicationContext);

			// Renderscript init
			try{
				BlurRenderer.initRenderScript(applicationContext);
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Could not init renderscript", ex);
			}

			// Manage recently used icons
			AniwaysRecentIconsManager.forceInit(applicationContext);

			// Manage the credits store
			AniwaysStoreManager.forceInit(applicationContext);

			// Starts the backend sync service 
			UpdateManager.startUpdates(applicationContext);

			// Start the N/W state checker
			AniwaysNetworkStateChecker.forceInit(applicationContext);

			// Init Volley
			Volley.forceInit(applicationContext);
		}

		setInit(isBackendService);

		AnalyticsReporter.ReportTiming(Verbosity.Info, startTime, "Performance", "Initialization. Service:" + isBackendService, "", TAG, "");
	}

	/**
	 * @return the sContext
	 */
	public static boolean isInit(boolean isBackendService) {
		// Application and service may or may not be in the same process
		// If we are already initialized, then no need to init again..
		// If the app is initialized, then we are initialized, even if we are in the service,
		// If the service is initialized, then if it is the app, we are not fully initialized
		if (isBackendService){
			return sIsServiceInit;
		}
		else{
			return sIsApplicationInit;
		}
	}

    private static void setInit(boolean isBackendService) {
		sIsServiceInit = true;
		if(!isBackendService){
			sIsApplicationInit = true;
		}
		sIsInitializing = false;
	}

	public static boolean isInitializing() {
		return sIsInitializing;
	}

    public static Context getApplicationContext(){
        return sApplicationContext;
    }
}

class AniwaysCustomExceptionHandler implements UncaughtExceptionHandler{
	// TODO: consider using the Google analytics exception handller as described here: http://dandar3.blogspot.co.il/2013/03/google-analytics-easytracker-detailed.html

	private UncaughtExceptionHandler mDefaultUncaughtExceptionHandler;

	public AniwaysCustomExceptionHandler(){
		mDefaultUncaughtExceptionHandler = Thread.getDefaultUncaughtExceptionHandler();
	}

	@Override
	public void uncaughtException(Thread t, Throwable e) { 

		Log.e(true, "UE", "Uncaught error!!", e);

		// Send to reporting service
		if (AnalyticsReporter.isInitialized()){
			AnalyticsReporter.reportError(Verbosity.Error, "", "Uncaught Exception in thread: " + t.getName(), e, false);
		}

		// Pass along to the default handler
		mDefaultUncaughtExceptionHandler.uncaughtException(t,e);
	}
}
