package com.aniways.data;

import java.util.UUID;

import com.aniways.IconData;
import com.aniways.Log;
import com.aniways.R;
import com.aniways.Utils;
import com.aniways.analytics.AnalyticsReporter;
import com.aniways.analytics.AnalyticsReporter.StoreEventAction;
import com.aniways.analytics.GoogleAnalyticsReporter;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.billing.AniwaysCreditsStoreActivity;
import com.aniways.data.AniwaysConfiguration.Verbosity;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.Intent;
import android.graphics.Rect;
import android.media.MediaPlayer;
import android.media.MediaPlayer.OnCompletionListener;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnLongClickListener;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;
import android.view.animation.AnimationSet;
import android.view.animation.DecelerateInterpolator;
import android.view.animation.Animation.AnimationListener;
import android.view.inputmethod.InputMethodManager;
import android.widget.ImageView;
import android.widget.PopupWindow;
import android.widget.TextView;

/**
 * Contains common functionality to handle presentation of locked icon in the suggestions popup and the emoticons button and elsewhere,
 * and handles launching the credits store if necessary and receiving results from actions in it and acting upon them.
 * Use an instance of this to handle the lifetime of a suggestion popup or an emoticons button popup. This instance is tied to an instance
 * of an JsonParser to preserve cohesion on all action, even if new data is received from the backend in the middle
 * @author Shai
 *
 */
public class AniwaysLockedIconHelper {
	private static final int CREDITS_CHANGE_ANIMATION_TIME = 600;
	private static final String TAG = "AniwaysLockedIconHelper";
	protected static final long CHECK_RETURN_FROM_STORE_INTERVAL = 100;

	private JsonParser mParser;
	private boolean launchedStoreFromTryingToUnlockIcon = false;
	private boolean launchedStoreFromButton = false;
	private int verticalLocationOfAnchorOnScreenBeforeStoreLaunched = 0;
	private CompleteIconUnlockSequenceData lastCompleteIconUnlockSequenceData = null;
	private OnItemContainingLockedIconClickListener mOnItemContainingLockedIconClickListener = null;
	private PopupWindow mPopup;
	private TextView mCreditsBalanceButton;
	private OnStoreAboutToLaunchListener mStoreLaunchListener;
	private Context mContext;
	private String mSourceScreen;
	// The view to which the popup is attached
	private View mAnchorView;
	private NonThrowingRunnable mCheckReturnFromStoreRunnable;


	public AniwaysLockedIconHelper(JsonParser parser, OnItemContainingLockedIconClickListener clickListener, OnStoreAboutToLaunchListener storeLauchListener,
                                   TextView creditsBalanceButton, PopupWindow popup, View anchorView, boolean showCreditsIfStoreIsEnabled, Context context, String sourceScreen){
		//validateConstructorParam(parser, "parser"); // no need to validate cause when displaying icon info it is null
		validateConstructorParam(clickListener, "clickListener");
		validateConstructorParam(popup, "popup");
		validateConstructorParam(creditsBalanceButton, "creditsBalanceButton");

		mParser = parser;
		mOnItemContainingLockedIconClickListener = clickListener;
		mPopup = popup;
		mCreditsBalanceButton = creditsBalanceButton;
		mStoreLaunchListener = storeLauchListener;
		mContext = context;
		mSourceScreen = sourceScreen;
		mAnchorView = anchorView;

		setupCreditsBalanceButton(showCreditsIfStoreIsEnabled);
		setupCheckReturnedFromStore();
	}

	@SuppressLint("NewApi")
	public void setIconAndLockVisibility(ImageView img, ImageView lock, IconData icon) {
		if (mParser!=null && icon != null && AniwaysStoreManager.isIconLocked(icon)){
			lock.setVisibility(View.VISIBLE);
			if(Utils.isAndroidVersionAtLeast(11)){
				img.setAlpha(AniwaysPrivateConfig.getInstance().lockedIconAlpha);
			}
			else{
				img.setAlpha((int) (AniwaysPrivateConfig.getInstance().lockedIconAlpha*255));
			}
		}
		else{
			lock.setVisibility(View.GONE);
		}
	}

    public void onIconClicked(ImageView img, ImageView lock, IconData icon, Object iconClickContext, OnItemContainingLockedIconClickListener iconClickListener) {
        this.mOnItemContainingLockedIconClickListener = iconClickListener;

        onIconClicked(img,lock,icon,iconClickContext);
    }

	public void onIconClicked(ImageView img, ImageView lock, IconData icon, Object iconClickContext) {
		// Check if icon is locked. If it isn't then just perform the click
		// If it is then: if the  user has credits to unlock then unlock, if not the launch the credits store
		if(mParser!=null && icon != null && AniwaysStoreManager.isIconLocked(icon)){

			CompleteIconUnlockSequenceData completeUnlockData = new CompleteIconUnlockSequenceData(icon, lock, img, iconClickContext); 

			// Check if user has enough credits to purchase the icon. Start the store if not
			if(AniwaysStoreManager.canUserBuyIcon(icon)){
				resetLaunchedStoreParams();
				completeUnlockIconSequence(completeUnlockData);
			}
			else{
				try{
					this.launchCreditsStore(completeUnlockData);
				}
				catch(Throwable ex){
					Log.e(true, TAG, "Error starting Store activity folowing trying to unlock icon", ex);
				}
			}
		}
		else{ // Icon is not locked, so just perform the click..
			callOnItemClickedListener(iconClickContext);
		}

	}

	public JsonParser getDataParser() {
		return this.mParser;
	}

	public Context getContext(){
		return mContext;
	}

	private boolean mDetectedStoreOpened = false;

	private void setupCheckReturnedFromStore() {

		// Will continue deleting as long as the 'delete' button is pressed
		mCheckReturnFromStoreRunnable = new NonThrowingRunnable(TAG, "while checking if returned from store", ""){
			@Override
			public void innerRun() {
				// As long as the store is still running then just post another check for the next interval
				if (AniwaysCreditsStoreActivity.sIsRunning){
					mDetectedStoreOpened = true;
					mAnchorView.postDelayed(mCheckReturnFromStoreRunnable, CHECK_RETURN_FROM_STORE_INTERVAL); 
					return;
				}
				else if (!mDetectedStoreOpened){
					mAnchorView.postDelayed(mCheckReturnFromStoreRunnable, CHECK_RETURN_FROM_STORE_INTERVAL);
					return;
				}

				// Perform the 'return from store' flow
				mDetectedStoreOpened = false;
				onReturnFromStore();
			}
		};
	}

	// Try to figure out if the store activity is no longer showing and credits balance has changed, so need to unlock icon that 
	// prompted the store launch, or at the very least, update the credits balance..
	private void onReturnFromStore() {

		try {
			Log.i(TAG, "Returning from store..");

			// Open the keyboard if it was previously opened.
			if(verticalLocationOfAnchorOnScreenBeforeStoreLaunched != getAnchorViewLocationOnScreen()){
				InputMethodManager inputMethodManager=(InputMethodManager)mContext.getSystemService(Context.INPUT_METHOD_SERVICE);
				if(inputMethodManager != null){
					inputMethodManager.toggleSoftInputFromWindow(mAnchorView.getApplicationWindowToken(), InputMethodManager.SHOW_IMPLICIT, 0);
				}
			}

			// Call store launch listener
			if(mStoreLaunchListener != null){
				mStoreLaunchListener.onStoreClosing();
			}

			// Reset the opened store related vars
			final boolean wasLaunchedStoreFromTryingToUnlockIcon = launchedStoreFromTryingToUnlockIcon;
			final CompleteIconUnlockSequenceData completeIconUnlockSequenceDataLocal = lastCompleteIconUnlockSequenceData;
			resetLaunchedStoreParams();

			// See if the credits amount has changed after the store activity was launched, and if it did 
			// then continue the icon purchase flow.
			if(wasLaunchedStoreFromTryingToUnlockIcon || launchedStoreFromButton){
				int lastCreditsBalance = 0;
				try {
					lastCreditsBalance = Integer.parseInt(mCreditsBalanceButton.getText().toString());
				} catch (NumberFormatException e) {
					Log.e(true, TAG, "Could not convert the credits balance in the popup to an integer", e);
				}
				if(lastCreditsBalance != AniwaysStoreManager.getUserCreditsLeft()){
					mAnchorView.post(new NonThrowingRunnable(TAG, "changing credits value", ""){

						@Override
						public void innerRun() {

							// Animate the credits change
							if (!mCreditsBalanceButton.isShown()){
								//Log.e(true, TAG, "The credits balance view is not showing althogh credits sum was changed");
							}
							AnimationSet creditsChangeAnimation = getCreditsChangeAnimation();
							int creditsChangeDuration = CREDITS_CHANGE_ANIMATION_TIME;
							mCreditsBalanceButton.startAnimation(creditsChangeAnimation);

							// If the store was launched as a result of trying to unlock an icon then start the sequece after the credits change animation
							if(wasLaunchedStoreFromTryingToUnlockIcon){
								if(AniwaysStoreManager.canUserBuyIcon(completeIconUnlockSequenceDataLocal.icon)){
									mAnchorView.postDelayed(new NonThrowingRunnable(TAG, "completing unlock sequence", ""){
										@Override
										public void innerRun() {
											completeUnlockIconSequence(completeIconUnlockSequenceDataLocal);
										}
									}, creditsChangeDuration);
								}
								else{
									Log.w(true, TAG, "Returning from store, credits amount changed, but still cannot unlock icon: " + completeIconUnlockSequenceDataLocal.icon.getFileName() + " . Available credits: " + AniwaysStoreManager.getUserCreditsLeft());
								}
							}

						}
					});
				}
			}
		} catch (Exception e) {
			Log.e(true, TAG, "Error in global layout change", e);
		}

	}

	private void setupCreditsBalanceButton(boolean showCreditsIfStoreIsEnabled) {		
		if (showCreditsIfStoreIsEnabled && AniwaysStoreManager.isStoreEnabled()){
			mCreditsBalanceButton.setVisibility(View.VISIBLE);
			mCreditsBalanceButton.setText(Integer.toString(AniwaysStoreManager.getUserCreditsLeft()));
			mCreditsBalanceButton.setOnClickListener(new OnClickListener(){
				@Override
				public void onClick(View v) {
					try{
						if(AniwaysPrivateConfig.getInstance().disableCreditStoreIcon){
							Log.v(TAG, "Credit store balance button is disabled");
							return;
						}
						launchCreditsStore(null);
					}
					catch(Throwable ex){
						Log.e(true, TAG, "Error starting Store activity following a click on the store balance", ex);
					}
				}
			});
			mCreditsBalanceButton.setOnLongClickListener(new OnLongClickListener(){
				@Override
				public boolean onLongClick(View v) {
					try{
						if(AniwaysPrivateConfig.getInstance().disableCreditStoreIcon){
							launchCreditsStore(null);
							return true;
						}
						
					}
					catch(Throwable ex){
						Log.e(true, TAG, "Error starting Store activity following a long click on the store balance", ex);
					}
					return false;
				}
			});
		}
		else{
			mCreditsBalanceButton.setVisibility(View.GONE);
		}

	}

	/*
	 * @throws IllegalStateException
	 */
	@SuppressLint("NewApi")
	private void completeUnlockIconSequence(final CompleteIconUnlockSequenceData completeIconUnlockSequenceData) throws IllegalStateException {
		if(mPopup != null && !mPopup.isShowing()){
			Log.w(true, TAG, "Received request to complete unlock sequence, but the popup is no longer showing");
			return;
		}

		// Unlock the icon and decrease the credits
		AniwaysStoreManager.unlockIconAndDecreaseCredits(this.mSourceScreen, completeIconUnlockSequenceData.icon);

		// Change the text of the credits balance
		if (!mCreditsBalanceButton.isShown()){
			//Log.e(true, TAG, "The credits balance view is not showing althogh credits sum was changed");
		}
		AnimationSet creditsChangeAnimation = getCreditsChangeAnimation();


		// Need to set the end value of the alpha here, cause otherwise the values of the animation are from the setAlpha that was before 
		// (if it was 0.4 then alpha 1.0 will be 1.0*0.4 - so not fully opaque)
		if(Utils.isAndroidVersionAtLeast(11)){
			completeIconUnlockSequenceData.img.setAlpha(1.0f);
		}
		else{
			completeIconUnlockSequenceData.img.setAlpha(255);
		}
		AlphaAnimation fadeInAnimation = new AlphaAnimation(AniwaysPrivateConfig.getInstance().lockedIconAlpha, 1.0f);
		fadeInAnimation.setDuration(1200);
		fadeInAnimation.setStartOffset(0);
		fadeInAnimation.setInterpolator(new AccelerateInterpolator());					
		AlphaAnimation fadeOutAnimation = new AlphaAnimation(1.0f, 0.0f);
		fadeOutAnimation.setDuration(1200);
		fadeOutAnimation.setStartOffset(0);
		fadeOutAnimation.setInterpolator(new DecelerateInterpolator());

		// Start Animations
		completeIconUnlockSequenceData.img.startAnimation(fadeInAnimation);
		completeIconUnlockSequenceData.lock.startAnimation(fadeOutAnimation);
		mCreditsBalanceButton.startAnimation(creditsChangeAnimation);

		// Start the unlock icon sound
		try{
			final MediaPlayer mp = MediaPlayer.create(completeIconUnlockSequenceData.img.getContext(), R.raw.aniways_unlock_icon_sound);
			if(mp == null){
				Log.e(true, TAG, "Failed to create media player for the unlock icon");
			}
			else{
				mp.start();
				mp.setOnCompletionListener(new OnCompletionListener(){
					@Override
					public void onCompletion(MediaPlayer mp) {
						try{
							mp.release();
						}
						catch(Throwable ex){
							Log.e(true, TAG, "Caught Exception while completing playing a sound");
						}
					}
				});
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught exception while creating media player for the unlock icon");
		}

		// Complete the sequence after the animations finish
		completeIconUnlockSequenceData.img.postDelayed(new NonThrowingRunnable(TAG, "conmpleting unlock icon sequence after animations finish", "") {
			@Override
			public void innerRun() {
				completeIconUnlockSequenceData.lock.setVisibility(View.GONE);
				callOnItemClickedListener(completeIconUnlockSequenceData.iconClickedContext);
			}
		}, 1200);
	}

	/**
	 * @return
	 */
	private AnimationSet getCreditsChangeAnimation() {
		AnimationSet s = new AnimationSet(false);
		AlphaAnimation creditsBalancefadeInAnimation = new AlphaAnimation(0.0f, 1.0f);
		creditsBalancefadeInAnimation.setDuration(CREDITS_CHANGE_ANIMATION_TIME / 2);
		creditsBalancefadeInAnimation.setStartOffset(0);
		creditsBalancefadeInAnimation.setInterpolator(new AccelerateInterpolator());
		AlphaAnimation creditsBalanceFadeOutAnimation = new AlphaAnimation(1.0f, 0.0f);
		creditsBalanceFadeOutAnimation.setAnimationListener(new AnimationListener(){

			@Override
			public void onAnimationStart(Animation animation) {
				// TODO Auto-generated method stub
				try{

				}
				catch(Throwable ex){
					Log.e(true, TAG, "Caught an Exception in onAnimationStart", ex);
				}

			}

			@Override
			public void onAnimationEnd(Animation animation) {
				try{
					mCreditsBalanceButton.setText(Integer.toString(AniwaysStoreManager.getUserCreditsLeft()));
				}
				catch(Throwable ex){
					Log.e(true, TAG, "Caught an Exception in onAnimationEnd", ex);
				}
			}

			@Override
			public void onAnimationRepeat(Animation animation) {
				// TODO Auto-generated method stub

				try{

				}
				catch(Throwable ex){
					Log.e(true, TAG, "Caught an Exception in onAnimationRepeat", ex);
				}
			}

		});
		creditsBalanceFadeOutAnimation.setDuration(CREDITS_CHANGE_ANIMATION_TIME / 2);
		creditsBalanceFadeOutAnimation.setStartOffset(0);
		creditsBalanceFadeOutAnimation.setInterpolator(new DecelerateInterpolator());
		s.addAnimation(creditsBalanceFadeOutAnimation);
		s.addAnimation(creditsBalancefadeInAnimation);
		s.setRepeatCount(0);
		return s;
	}

	private void callOnItemClickedListener(Object iconClickContext){
		if(this.mOnItemContainingLockedIconClickListener != null){
			mOnItemContainingLockedIconClickListener.onItemClick(iconClickContext);
		}

	}

	private void launchCreditsStore(CompleteIconUnlockSequenceData completeUnlockData) {
		boolean followingClickOnStoreBalance = completeUnlockData == null;

		Intent creditsStoreIntent = new Intent(mCreditsBalanceButton.getContext(), AniwaysCreditsStoreActivity.class);
		creditsStoreIntent.addFlags(Intent.FLAG_ACTIVITY_SINGLE_TOP);
		creditsStoreIntent.putExtra(AniwaysCreditsStoreActivity.KEY_GENERATE_BY_TRYIMG_TO_UNLOCK_ICON, !followingClickOnStoreBalance);
		String iconPath = null;
		if(!followingClickOnStoreBalance){
			iconPath = completeUnlockData.icon.getFileName();
			creditsStoreIntent.putExtra(AniwaysCreditsStoreActivity.KEY_ICON_TRYING_TO_UNLOCK, iconPath);
		}
		String id = UUID.randomUUID().toString();
		creditsStoreIntent.putExtra(AniwaysCreditsStoreActivity.KEY_ID, id);
		creditsStoreIntent.putExtra(AniwaysCreditsStoreActivity.KEY_SOURCE_SCREEN, mSourceScreen);
		int creditsBalance = AniwaysStoreManager.getUserCreditsLeft();
		GoogleAnalyticsReporter.reportEvent(Verbosity.Info, AniwaysCreditsStoreActivity.CREDITS_PURCHASE_CATEGORY, "Open Store", followingClickOnStoreBalance ? "clicked on store balance" : iconPath, creditsBalance);
		AnalyticsReporter.reportStoreEvent(StoreEventAction.launchingStore, id, creditsBalance, 0, 0, !followingClickOnStoreBalance, !followingClickOnStoreBalance ? iconPath : null, mSourceScreen, null, null, null);

		// Call store launch listener
		if(mStoreLaunchListener != null){
			mStoreLaunchListener.onStoreAboutToLaunch();
		}

		// Set params to help us continue after the activity finishes
		setLaunchedStoreParams(completeUnlockData);

		// Close the keyboard (for consistency - because sometimes it will close on its own)
		InputMethodManager inputMethodManager=(InputMethodManager)mContext.getSystemService(Context.INPUT_METHOD_SERVICE);
		if(inputMethodManager != null){
			inputMethodManager.hideSoftInputFromWindow(mAnchorView.getApplicationWindowToken(), 0);
		}

		// Start the activity
		Context context = followingClickOnStoreBalance ? this.mCreditsBalanceButton.getContext() : completeUnlockData.img.getContext();
		context.startActivity(creditsStoreIntent);

		// Start check if the activity has returned
		try{
			mAnchorView.post(mCheckReturnFromStoreRunnable);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in posting mCheckReturnFromStoreRunnable", ex);
		}


		Log.i(TAG, "Starting the credits store " + (followingClickOnStoreBalance ? "following a click on the store balance" : ""));
	}

	private void setLaunchedStoreParams(CompleteIconUnlockSequenceData completeUnlockData) {
		boolean followingClickOnStoreBalance = completeUnlockData == null;

		launchedStoreFromTryingToUnlockIcon = !followingClickOnStoreBalance;
		launchedStoreFromButton = followingClickOnStoreBalance;
		lastCompleteIconUnlockSequenceData = completeUnlockData;
		verticalLocationOfAnchorOnScreenBeforeStoreLaunched = getAnchorViewLocationOnScreen();
	}

	private void resetLaunchedStoreParams() {
		launchedStoreFromTryingToUnlockIcon = false;
		launchedStoreFromButton = false;
		lastCompleteIconUnlockSequenceData = null;
	}

	private void validateConstructorParam(Object param, String paramName){
		if(param == null){
			throw new IllegalArgumentException("Null param: " + paramName);
		}
	}

	private int getAnchorViewLocationOnScreen() {
		Rect r = new Rect();
		mAnchorView.getWindowVisibleDisplayFrame(r);

		int[] loc = new int[2];
		mAnchorView.getLocationOnScreen(loc);
		return loc[1] + r.bottom;
	}

	/**
	 * Listener for item click
	 *
	 */
	public interface OnItemContainingLockedIconClickListener {
		public abstract void onItemClick(Object iconClickContext);
	}

	public interface OnStoreAboutToLaunchListener {
		public abstract void onStoreAboutToLaunch();

		public abstract void onStoreClosing();
	}

	private class CompleteIconUnlockSequenceData{
		final ImageView lock;
		final ImageView img;
		final Object iconClickedContext;
		final IconData icon;

		public CompleteIconUnlockSequenceData (IconData icon, ImageView lock, ImageView img, Object iconClickedContext){
			this.iconClickedContext = iconClickedContext;
			this.img = img;
			this.lock = lock;
			this.icon = icon;
		}
	}

}
