/**
 * 
 */
package com.aniways.data;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.InputStreamReader;
import com.aniways.Log;
import com.aniways.Utils;
import com.aniways.VersionComparisonResult;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.data.JsonParser.InputStreamProvider;
import com.aniways.service.AniwaysAlarmListener;
import com.aniways.service.utils.AniwaysServiceUtils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Resources.NotFoundException;
import android.os.Handler;
import android.os.Process;
import aniways.com.google.gson.stream.JsonReader;

/**
 * @author Shai
 *
 */
public class AniwaysBackendSyncChecker extends NonThrowingRunnable {

	private static final String TAG = "AniwaysBackendSyncChecker";
	private static final String THREAD_NAME = "AniwaysBackendSyncChecker";
	static final String BACKEND_SYNC_CHECKER_KEY = "AniwaysBackendSyncChecker";

	private static AniwaysBackendSyncChecker sInstance;

	private Thread mThread;

	private static volatile long sLastBackendSyncRequest = 0;
	private static volatile long sLastBackendSyncThreadRunningCheck = 0;
	private static Context sContext;

	private static Handler sHandler;

	public static void forceInit(Context applicationContext) {
		if(sInstance != null){
			Log.e(true, TAG, "Calling init of Backend Sync Checker more than once");
		}

		sContext = applicationContext;
		sHandler = new Handler(sContext.getMainLooper());
		AniwaysBackendSyncChecker instance = new AniwaysBackendSyncChecker(applicationContext);
		Thread thread = new Thread(instance);
		thread.setName(THREAD_NAME);
		instance.setThread(thread);

		if(sInstance != null){
			Log.i(TAG, "There is already a backend sync thread running, so interrupting it");
			Thread oldThread = sInstance.getThread();
			oldThread.interrupt();
		}

		Log.i(TAG, "Starting the backend sync thread..");
		thread.start();
		sInstance = instance;
	}

	private Thread getThread() {
		return this.mThread;

	}

	private void setThread(Thread thread) {
		this.mThread = thread;

	}

	/**
	 *
	 */
	public AniwaysBackendSyncChecker(Context context) {
		super(TAG, "run", "");
	}

	/**
	 * 
	 */
	@Override
	public void innerRun() {
		try{
			Process.setThreadPriority(Process.THREAD_PRIORITY_BACKGROUND + Process.THREAD_PRIORITY_MORE_FAVORABLE);
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Could not set thead priority");
		}
		while(true){
			Log.i(TAG, "Backend sync calling parseConfingAndKeywordsIfNecessary..");
			parseConfingAndKeywordsIfNecessary(sContext, false, false);

			// Wait between each cycle
			try {
				boolean keywordsVersionZero = AniwaysPhraseReplacementData.getDataParser().isEmpty();
				long sleepTime;
				if(keywordsVersionZero){
					sleepTime = AniwaysPrivateConfig.getInstance().shortIntervalBetweenBackendSyncMillis;
				} else {
					sleepTime = AniwaysPrivateConfig.getInstance().longIntervalBetweenBackendSyncMillis;
				}
				Thread.sleep(sleepTime);
			} catch (InterruptedException e1) {
				Log.w(true, TAG, "Thread interrupted while waiting for next backend sync cycle", e1);
				break;
			}
		}
	}

	static void makeSureRunning() {
		Log.d(TAG, "Checking if the backend sync thread is running");
		Thread thread = sInstance.getThread();
		if(!thread.isAlive()){
			Log.w(true, TAG, "Backend Sync thread was not running. Restarting it..");
			thread = new Thread(sInstance);
			thread.setName(THREAD_NAME);
			thread.start();
		}
	}

	/**
	 * Normally, the service requests sync automatically, but upon user interaction it is good in some cases
	 * to use this method to check that everyting is OK and maybe request a sync ahead of time
	 * (for example, if the keywoprds version is 0 because of an error in the first sync and the user is active and we would like
	 *  to get icons for him faster since he is active in the app)
	 */
	public static void requestSyncWithServerIfNecessary(){
		// Check when the last successful sync with the Aniways server was performed.
		// If it was more than the alarm interval then it means that probably there was
		// an error somewhere (no network connection for instance). So we request to 
		// perform a sync now when the user is active in the application and the network 
		// connection probably exists.
		long now = System.currentTimeMillis();
		boolean keywordsVersionIsZero = AniwaysPhraseReplacementData.getDataParser().getKeywordsVersion().equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION);

		// The last time we requested the sync was not too recent (don't want to tax the server)), unless the keywords version is 0.0
		// This 'if' is not together with the ones below because it doesn't require the shared perfs or
		// the service manager, so it can save some processing to put it here and improve performance.
		if((now - sLastBackendSyncRequest) > AniwaysPrivateConfig.getInstance().backendSyncRequestInterval || keywordsVersionIsZero){
			Context context = sContext.getApplicationContext();
			// If there is no value then get 'now' - it will prompt an update request if keywords version is 0.0 only. If it is not 0.0 then there should be a value in the shared perfs anyway..
			AniwaysAlarmListener alarmListener = new AniwaysAlarmListener();
			alarmListener.sendWakefulWork(context, false);
			sLastBackendSyncRequest = now;
		}
		// Make sure that the backend server is running
		if((now - sLastBackendSyncThreadRunningCheck ) > 300000){
			AniwaysStatics.makeSureBackendSyncThreadIsRunning();
			sLastBackendSyncThreadRunningCheck = now;
		}
	}

	public enum JsonSupplier {
		existing,
		server,
		preInstalled
	}

	public synchronized static void parseConfingAndKeywordsIfNecessary(Context context, boolean isConfigInit, boolean isConfigRefresh) {
		try{
			sContext = context;
			Log.i(TAG, "parseConfingAndKeywordsIfNecessary called. Is config init: " + isConfigInit + " . Is config refresh: " + isConfigRefresh);
			// Parse config
			parseIfNecessaryInner(true, (isConfigInit || isConfigRefresh), isConfigRefresh);

			// Parse keywords
			if(!isConfigInit && !isConfigRefresh){
				parseIfNecessaryInner(false, false, false);
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught Exception in parseConfingAndKeywordsIfNecessary", ex);
		}	
	}

	private synchronized static void parseIfNecessaryInner(boolean isConfig, boolean setNewVersionOnCallingThread, boolean forceConfigUpdate) {
		try{
			String jsonSupplier = isConfig ? "config" : "keywords";
			Log.i(TAG, "parsing " + jsonSupplier + " if necessary. setNewVersionOnCallingThreadt: " + setNewVersionOnCallingThread + ". forceConfigUpdate: " + forceConfigUpdate);

			String serverVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
			String preInstalledVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;

			InputStreamProvider preInstalledFileStream = null;
			InputStreamProvider serverFileStream = null;

			// Get pre-installed version
			try{
				String packageName = sContext.getPackageName();
				if(packageName == null){
					Log.e(true, TAG, "package Name is null");
				}
				else{
					final int id = sContext.getResources().getIdentifier("aniways_" + jsonSupplier, "raw", packageName);
					if(id <= 0){
						Log.w(false, TAG, jsonSupplier + " file doesn't exist in raw assets");
					}
					else{
						try{
							preInstalledFileStream = new InputStreamProvider(){

								@Override
								public InputStream getStream() {
									return sContext.getResources().openRawResource(id);
								}
								
							};
							InputStream tempStream = preInstalledFileStream.getStream();

							if (preInstalledFileStream != null) {
								preInstalledVersion = getVersionFromStream(tempStream);
								if (!Utils.isVersionLegal(preInstalledVersion)){
									Log.e(true, TAG, "Illegel pre-installed keywords version: " + preInstalledVersion);
									preInstalledVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
								}
							}
						}
						catch(NotFoundException ex){
							Log.w(true, TAG, jsonSupplier + " file doesn't exist in raw assets");
						}
					}
				}
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught Exception while trying to get pre-installed " + jsonSupplier + " version", ex);
				preInstalledVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
			}

			Log.i(TAG, String.format("Pre-installed %s version: %s", jsonSupplier, preInstalledVersion));

			// Get server version
			try{
				File serverFile = null;
				if(isConfig){
					serverFile = new File(AniwaysStorageManager.getConfigFilePath(sContext));
				}
				else{
					AniwaysStorageManager storageManager = AniwaysStorageManager.getInstance(sContext);
					serverFile = new File(storageManager.getKeywordsDir().getAbsolutePath() + "/" + AniwaysServiceUtils.KEYWORDS_FILE);
				}
				
				final File serverFileFinal = serverFile;
				
				if(!serverFile.exists()){
					Log.w(false, TAG, "Server " + jsonSupplier + " file could not be found in: " + serverFile.getAbsolutePath());
				}
				else{
					try {
						serverFileStream = new InputStreamProvider(){

							@Override
							public InputStream getStream() throws FileNotFoundException {
								return new FileInputStream(serverFileFinal);
							}
							
						};
						InputStream tempStream = serverFileStream.getStream();
						serverVersion = getVersionFromStream(tempStream);
					} catch (FileNotFoundException e) {
						Log.e(true, TAG, "Caught FileNotFoundException while reading " + jsonSupplier + " server Json", e);
						serverVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
					} 
				}
			}
			catch(Throwable ex){
				Log.e(true, TAG, "Caught Exception while trying to get server " + jsonSupplier + " version", ex);
				serverVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
			}

			Log.i(TAG, String.format("Server %s version: %s", jsonSupplier, serverVersion));

			// Get pre-existing version
			String existingVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
			if(isConfig){
				existingVersion = AniwaysPrivateConfig.getInstance().version;
			}
			else{
				existingVersion = AniwaysPhraseReplacementData.getDataParser().getKeywordsVersion();
			}

			Log.i(TAG, String.format("Pre-existing %s version: %s", jsonSupplier, existingVersion));

			// parse if necessary
			//
			if(isConfig && forceConfigUpdate){
				// We make the existing version to be 0 in order for any other version to be taken
				existingVersion = AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION;
			}
			JsonSupplier supplier = whichOneToTake(existingVersion, preInstalledVersion, serverVersion, jsonSupplier);
			Log.i(TAG, "Going with config supplier: " + supplier);
			if(supplier == JsonSupplier.existing){
				// do nothing
			}
			else{
				String expectedVersion = null;
				InputStreamProvider stream = null;
				// Parse the config file on a different thread if necessary
				if(supplier == JsonSupplier.preInstalled){
					expectedVersion = preInstalledVersion;
					stream = preInstalledFileStream;
				}
				else if(supplier == JsonSupplier.server){
					expectedVersion = serverVersion;
					stream = serverFileStream;
				}
				else{
					Log.e(true, TAG, "Unknown json supplier: " + supplier);
				}

				if(isConfig){
					final AniwaysPrivateConfig newConfig = AniwaysPrivateConfig.createNewInstance(stream.getStream());

					if(setNewVersionOnCallingThread){
						AniwaysPrivateConfig.setNewInstance(newConfig);
					}
					else{
						sHandler.post(new NonThrowingRunnable(TAG, "parse if necessary", jsonSupplier) {
							@SuppressLint("NewApi")
							@Override
							public void innerRun(){
								AniwaysPrivateConfig.setNewInstance(newConfig);
							}
						});
					}
				}
				else{
					final String expectedKeywordsVersion = expectedVersion;
					final JsonParser newParser = AniwaysPhraseReplacementData.parseData(stream);
					sHandler.post(new NonThrowingRunnable(TAG, "parse if necessary", jsonSupplier) {
						@SuppressLint("NewApi")
						@Override
						public void innerRun(){
							AniwaysPhraseReplacementData.setNewParser(newParser, expectedKeywordsVersion);
						}
					});
				}
			}
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught an Exception in parseIfNecessaryInner", ex);
		}
	}

	private static JsonSupplier whichOneToTake(String existingVersion, String preInstalledVersion, String serverVersion, String jsonName){
		VersionComparisonResult compareResult = null;

		// First, compare pre-installed with existing version
		try{
			if(!Utils.isVersionLegal(preInstalledVersion) || !Utils.isVersionLegal(existingVersion)){
				if(Utils.isVersionLegal(preInstalledVersion)){
					compareResult = new VersionComparisonResult(1);
				}
				else if(Utils.isVersionLegal(existingVersion)){
					compareResult = new VersionComparisonResult(-1);
				}
				else{
					compareResult = null;
				}
			}
			else{
				compareResult = Utils.compareVersionStrings(preInstalledVersion, existingVersion);
			}
		}
		catch(IllegalArgumentException e){
			Log.e(true, TAG, "Error comparing versions of " + jsonName + " version. pre-installed verison:" + preInstalledVersion + ". Esisting: " + existingVersion);
		}

		// If there was error then return server version if it legal or existing if it isn't because it is already parsed..
		// Otherwise, if there was no error, then compare the server version with the bigger version
		if(compareResult == null){
			if(Utils.isVersionLegal(serverVersion)){
				return JsonSupplier.server;
			}
			else{
				return JsonSupplier.existing;
			}
		}
		else if(!Utils.isVersionLegal(serverVersion)){
			if(compareResult.result == 1){
				return JsonSupplier.preInstalled;
			}
			else{ // == -1 or 0
				return JsonSupplier.existing;
			}
		}
		else{
			// Server is legal, and also the one that was bigger out of existing and pre-installed
			// So need to compare the bigger one and the server
			if(compareResult.result == 1){
				// Compare with pre-installed
				try{
					compareResult = Utils.compareVersionStrings(preInstalledVersion, serverVersion);
				}
				catch(IllegalArgumentException e){
					Log.e(true, TAG, "Error comparing versions of " + jsonName + " version. pre-installed verison:" + preInstalledVersion + ". Server: " + serverVersion);
					// Return pre-installed because it passed the comparison b4
					return JsonSupplier.preInstalled;
				}
				if(compareResult.result >= 0){
					return JsonSupplier.preInstalled;
				}
				else{ // == -1 or 0
					return JsonSupplier.server;
				}
			}
			else{ // == -1 or 0
				// Compare with existing
				try{
					compareResult = Utils.compareVersionStrings(existingVersion, serverVersion);
				}
				catch(IllegalArgumentException e){
					Log.e(true, TAG, "Error comparing version of " + jsonName + " version. existing verison:" + existingVersion + ". Server: " + serverVersion);
					// Return existing because it passed the comparison b4
					return JsonSupplier.existing;
				}
				if(compareResult.result >= 0){
					return JsonSupplier.existing;
				}
				else{ // == -1 or 0
					return JsonSupplier.server;
				}
			}
		}
	}

	static String getVersionFromStream(InputStream stream){
		String result = null;

		try{
			JsonReader jsonReader = new JsonReader(new InputStreamReader(stream, "UTF-8"));
			jsonReader.beginObject();
			//stay in loop as long as there are more data elements
			while (jsonReader.hasNext()) {
				//get the element name
				String name = jsonReader.nextName();

				if (name.equals("version")) {
					result = jsonReader.nextString();
					Log.d(TAG, "Discovered version: " + result);
					break;
				}
				else {
	                jsonReader.skipValue();
	            }
			}
			//end reader and close the stream
			//jsonReader.endObject();
			jsonReader.close();
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Caught exception while trying to get version from stream", ex);
		}
		try{
			stream.close();
		}
		catch(Throwable tr){
			Log.w(false, TAG, "Caught Exception while closing stream");
		}
		return result;
	}
}
