package com.aniways.bugsnag.android;

import java.io.File;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import com.aniways.analytics.service.AniwaysAnalyticsService;
import com.aniways.bugsnag.Error;
import com.aniways.bugsnag.MetaData;
import com.aniways.bugsnag.Notification;
import com.aniways.bugsnag.android.utils.Async;

public class Client extends com.aniways.bugsnag.Client {
    private static final String NOTIFIER_NAME = "Android Bugsnag Notifier";
    private static final String NOTIFIER_VERSION = "2.2.0";

    private Logger logger;
    private Context applicationContext;
    private String cachePath;

    public Client(Context androidContext, String apiKey, boolean enableMetrics) {
        super(apiKey);

        // Create a logger
        logger = new Logger();
        setLogger(logger);

        // Get the application context, many things need this
        applicationContext = androidContext.getApplicationContext();

        this.diagnostics = new Diagnostics(config, applicationContext, this);

        // Set notifier info
        setNotifierName(NOTIFIER_NAME);
        setNotifierVersion(NOTIFIER_VERSION);

        // Send metrics data (DAU/MAU etc) if enabled
//        if(enableMetrics) {
//            //TODO:SM We should prevent this sending on rotate
//            makeMetricsRequest();
//        }

        // Flush any queued exceptions
        flushErrors();

        logger.info("Bugsnag is loaded and ready to handle exceptions");
    }

    public void notify(Throwable e, String severity, MetaData overrides) {
        try {
            if(!config.shouldNotify()) return;
            if(config.shouldIgnore(e.getClass().getName())) return;

            // Create the error object to send
            final Error error = createError(e, severity, overrides);
            
            

            // Send the error
            Async.safeAsync(new Runnable() {
                @Override
                public void run() {
					Notification notif = createNotification(error);
					StringBuilder deliver = notif.getPayload();
					
					queuePayload(deliver);
                }

            });
        } catch(Throwable t) {
            logger.warn("Error notifying Bugsnag", t);
        }
    }
    
	private void queuePayload(StringBuilder deliver) {
		Intent intent = new Intent(applicationContext, AniwaysAnalyticsService.class);
		intent.setAction(AniwaysAnalyticsService.ACTION_ENQUEUE_ERROR);
		intent.putExtra(AniwaysAnalyticsService.SERIALIZED_EVENT_KEY, deliver.toString());

		applicationContext.startService(intent);
	}


    private void flushErrors() {
        if(cachePath == null) return;

        Async.safeAsync(new Runnable() {
            @Override
            public void run() {
                // Look up all saved error files
                File exceptionDir = new File(cachePath);
                if(exceptionDir.exists() && exceptionDir.isDirectory()) {
                    Notification notif = null;

                    for(File errorFile : exceptionDir.listFiles()) {
                        try {
                            if(notif == null) notif = createNotification();
                            notif.setError(errorFile);
                            StringBuilder deliver = notif.getPayload();
                            
                            queuePayload(deliver);

                            logger.debug("Deleting sent error file " + errorFile.getName());
                            errorFile.delete();
                        } catch (Throwable t) {
                            logger.warn("Problem sending unsent error from disk", t);
                            errorFile.delete();
                        }
                    }
                }
            }
        });
    }

    public void setContext(Activity context) {
        String contextString = ActivityStack.getContextName(context);
        setContext(contextString);
    }

    public void setLogger(Logger logger) {
        super.setLogger(logger);
        Async.logger = logger;
    }
}
