package com.aniways.blur;

import java.util.Locale;
import com.aniways.Log;
import com.aniways.R;
import com.aniways.ViewUtils;
import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Color;
import android.graphics.LinearGradient;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.RectF;
import android.graphics.Shader;
import android.graphics.Paint.Style;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.PathShape;
import android.util.AttributeSet;


public class ContextualBubbleView extends BlurLinearLayout
{
	private static final int DEFAULT_POINTER_HEIGHT = 15;
	private static final int DEFAULT_POINTER_WIDTH = 15;
	private static final int DEFAULT_CORNER_RADIUS = 12;
	private static final int DEFAULT_BUBBLE_COLOR = 0x66000000;
	private static final int DEFAULT_BUBBLE_OUTLINE_COLOR = Color.BLACK;
	private static final int DEFAULT_SHADOW_COLOR = Color.GRAY;
	private static final int DEFAULT_SHADOW_RADIUS = 0;
	private static final int DEFAULT_SHADOW_OFFSET_X = 0;
	private static final int DEFAULT_SHADOW_OFFSET_Y = 0;
	private static final int DEFAULT_BUBBLE_GRADIENT_TOP_COLOR = 0x66000000;
	private static final int DEFAULT_BUBBLE_GRADIENT_BOTTOM_COLOR = 0x66000000;
	private static final int DEFAULT_BUBBLE_OUTLINE_WIDTH = 1;
	private static final String TAG = "AniwaysContextualBubbleView";

	private Path bubblePath;
	int pointerHeight = DEFAULT_POINTER_HEIGHT;
	private int pointerWidth = DEFAULT_POINTER_WIDTH;
	private int cornerRadius = DEFAULT_CORNER_RADIUS;
	private int bubbleColor = DEFAULT_BUBBLE_COLOR;
	private int bubbleOutlineColor = DEFAULT_BUBBLE_OUTLINE_COLOR;
	private int shadowColor = DEFAULT_SHADOW_COLOR;
	private int bubbleGradientTopColor = DEFAULT_BUBBLE_GRADIENT_TOP_COLOR;
	private int bubbleGradientBottomColor = DEFAULT_BUBBLE_GRADIENT_BOTTOM_COLOR;
	private int shadowRadius = DEFAULT_SHADOW_RADIUS;
	private int shadowOffsetX = DEFAULT_SHADOW_OFFSET_X;
	private int shadowOffsetY = DEFAULT_SHADOW_OFFSET_Y;
	private int bubbleOutlineWidth = DEFAULT_BUBBLE_OUTLINE_WIDTH;
	private PathShape pathShape;
	private ShapeDrawable fillLayer;
	private ShapeDrawable outlineLayer;
	private float pointerCenter;
	private boolean forceLayout = false;
	private boolean arrowDown = true;

	public ContextualBubbleView(Context context)
	{
		super(context);
	}

	public ContextualBubbleView(Context context, AttributeSet attrs)
	{
		super(context, attrs);
		initAttributes(context, attrs);
	}

	public void setArrowDown(float center)
	{
		if(!arrowDown){
			this.setPadding(this.getPaddingLeft(), this.getPaddingTop() - pointerHeight, this.getPaddingRight(), this.getPaddingBottom() + pointerHeight);
		}
		
		arrowDown = true;
		setArrowLocation(center);
	}

	public void setArrowUp(float center) {
		if(arrowDown){
			this.setPadding(this.getPaddingLeft(), this.getPaddingTop() + pointerHeight, this.getPaddingRight(), this.getPaddingBottom() - pointerHeight);
		}
		
		arrowDown = false;
		setArrowLocation(center);
	}

	private void setArrowLocation(float center){
		pointerCenter = center;
		forceLayout  = true;
		this.requestLayout();
	}

	@SuppressLint("DrawAllocation") @Override
	protected void onLayout (boolean changed, int l, int t, int r, int b)
	{
		super.onLayout(changed, l, t, r, b);

		if(!changed && !forceLayout){
			return;
		}

		float inset = bubbleOutlineWidth * 0.5f;
		
		// This is not the t,l,b,r because these values are in comparison to the parent view.
		// Since the shape is drawn in relation to itself the coordinates need to start at 0,0
		// and the r,b params are calculated by the width and height and inset
		calculateBubblePath(0 + inset, 0 + inset, r - l - inset, b - t - inset);

		int w = r-l;
		int h = b-t;

		fillLayer.setIntrinsicWidth(w);
		fillLayer.setIntrinsicHeight(h);
		fillLayer.setBounds(l, t, r, b);

		outlineLayer.setIntrinsicWidth(w);
		outlineLayer.setIntrinsicHeight(h);
		outlineLayer.setBounds(l, t, r, b);

		((LayerDrawable)this.getBackground()).setBounds(l, t, r, b);

		// Must re-create with the new width and height, otherwise it uses scaling
		pathShape = new PathShape(bubblePath, (float) w, (float) h);

		fillLayer.setShape(pathShape);
		outlineLayer.setShape(pathShape);

		forceLayout = false;
	}

	private void initAttributes(Context context, AttributeSet attrs)
	{
		TypedArray a = context.obtainStyledAttributes(attrs, R.styleable.AniwaysContextualBubbleView);
		this.pointerHeight = a.getDimensionPixelSize(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_pointerHeight, DEFAULT_POINTER_HEIGHT);
		this.pointerWidth = a.getDimensionPixelSize(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_pointerWidth, DEFAULT_POINTER_WIDTH);
		this.cornerRadius = a.getDimensionPixelSize(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_cornerRadius, DEFAULT_CORNER_RADIUS);
		this.bubbleColor = a.getColor(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_color, DEFAULT_BUBBLE_COLOR);
		this.bubbleOutlineColor = a.getColor(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_outlineColor, DEFAULT_BUBBLE_OUTLINE_COLOR);
		this.bubbleOutlineWidth = a.getDimensionPixelSize(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_outlineWidth, DEFAULT_BUBBLE_OUTLINE_WIDTH);
		this.shadowColor = a.getColor(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_shadowColor, DEFAULT_SHADOW_COLOR);
		this.shadowRadius = a.getDimensionPixelSize(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_shadowRadius, DEFAULT_SHADOW_RADIUS);
		this.shadowOffsetX = a.getDimensionPixelOffset(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_shadowOffsetX, DEFAULT_SHADOW_OFFSET_X);
		this.shadowOffsetY = a.getDimensionPixelOffset(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_shadowOffsetY, DEFAULT_SHADOW_OFFSET_Y);
		this.bubbleGradientTopColor = a.getColor(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_gradientTopColor, DEFAULT_BUBBLE_GRADIENT_TOP_COLOR);
		this.bubbleGradientBottomColor = a.getColor(R.styleable.AniwaysContextualBubbleView_aniways_contextual_bubble_gradientBottomColor, DEFAULT_BUBBLE_GRADIENT_BOTTOM_COLOR);
		a.recycle();

		int minWidth = (cornerRadius * 2) + pointerWidth + bubbleOutlineWidth;
		int minHeight = (2*cornerRadius) + pointerHeight + bubbleOutlineWidth;
		this.setMinimumWidth(minWidth);
		this.setMinimumHeight(minHeight);
		//this.setMinimumWidth(600);
		//this.setMinimumHeight(200);
		this.pointerCenter = minWidth / 2.0f;
		this.arrowDown = true;

		bubblePath = new Path();
		pathShape = new PathShape(bubblePath, (float) minWidth, (float) minHeight);

		fillLayer = new ShapeDrawable(pathShape);
		fillLayer.getPaint().setStyle(Style.FILL);
		fillLayer.getPaint().setColor(bubbleColor);
		fillLayer.getPaint().setAntiAlias(true);

		outlineLayer = new ShapeDrawable(pathShape);
		outlineLayer.getPaint().setStyle(Style.STROKE);
		outlineLayer.getPaint().setColor(bubbleOutlineColor);
		outlineLayer.getPaint().setStrokeWidth(bubbleOutlineWidth);
		outlineLayer.getPaint().setAntiAlias(true);
		LayerDrawable ld = new LayerDrawable(new Drawable[] {fillLayer, outlineLayer});
		
		// They disappear after setting the background
		int pLeft = getPaddingLeft();
		int pTop = getPaddingTop();
		int pRight = getPaddingRight();
		int pBottom = getPaddingBottom();
		
		this.setBackgroundDrawable(ld);

		this.setPadding(pLeft, pTop + cornerRadius, pRight, pBottom + cornerRadius + pointerHeight);
	}

	private void calculateBubblePath(float l, float t, float r, float b)
	{
		bubblePath.reset();

		float viewBottom = b;
		
		float viewWidth = r - l;
		float viewHeight = b - t;
		//float viewCenterX = l + (viewWidth / 2.0f);
		float halfPointerWidth = pointerWidth / 2.0f;
		float minPointerCenter = l + cornerRadius + halfPointerWidth; 
		float maxPointerCenter = r - cornerRadius - halfPointerWidth;
		float cornerDiameter = 2*cornerRadius;
		float pointerCenterTemp = pointerCenter - ViewUtils.getRelativeLeftToViewRoot(this);
		
		//TODO: add checks for height
		
		// Make sure that there is enough room for the pointer
		if(viewWidth - cornerDiameter < pointerWidth){
			// do not show pointer
			Log.w(true, TAG, String.format(Locale.US, "Not showing pointer cause there is not enough room for it. L:%s, T:%s, R:%s, B:%s, Pointer width:%s", l,t,r,b,pointerWidth));
			pointerWidth = -1;
			r = l + cornerDiameter;
		}

		if(viewHeight < cornerDiameter){
			Log.w(true, TAG, String.format(Locale.US, "Setting path height to cornerDiameter as it is less then corner diameter. L:%s, T:%s, R:%s, B:%s, cornerDiameter:%s", l,t,r,b,cornerDiameter));
			pointerWidth = -1;
			b = t + cornerDiameter;
		}
		// If the pointer is too much to the right or left, then bring it closer to the center
		else if(minPointerCenter > pointerCenterTemp){
			pointerCenterTemp = minPointerCenter;
		}
		else if(maxPointerCenter < pointerCenterTemp){
			pointerCenterTemp = maxPointerCenter;
		}

		RectF cornerRect = new RectF(0, 0, cornerDiameter, cornerDiameter);

		// If there is a practical pointer, then draw the bubble and pointer.
		if ((pointerHeight > 0f) && (pointerWidth > 0f))
		{
			// TODO: these 2 seperate paths can become one (with 2 'ifs' inside)
			if(arrowDown){
				float bubbleBottom = viewBottom - pointerHeight;
				// Start at the point of the triangle at the bottom and draw the line up to the bottom of the round rect.
				bubblePath.moveTo(pointerCenterTemp, viewBottom);
				bubblePath.lineTo(pointerCenterTemp + halfPointerWidth, bubbleBottom);

				// Move to the bottom right corner of the bubble rect and draw the corner.
				bubblePath.lineTo(r - cornerRadius, bubbleBottom);
				cornerRect.offsetTo(r - cornerDiameter, bubbleBottom - cornerDiameter);
				bubblePath.arcTo(cornerRect, 90f, -90f);

				// Move up to the top right corner and draw the corner.
				bubblePath.lineTo(r, t + cornerRadius);
				cornerRect.offsetTo(r - cornerDiameter, t);
				bubblePath.arcTo(cornerRect, 0f, -90f);

				// Move up to the top left corner and draw the corner.
				bubblePath.lineTo(l + cornerRadius, t);
				cornerRect.offsetTo(l, t);
				bubblePath.arcTo(cornerRect, 270f, -90f);

				// Move up to the bottom left corner and draw the corner.
				bubblePath.lineTo(l, bubbleBottom - cornerRadius);
				cornerRect.offsetTo(l, bubbleBottom - cornerDiameter);
				bubblePath.arcTo(cornerRect, 180f, -90f);

				// Finish the bottom and the left hand side of the pointer.
				bubblePath.lineTo(pointerCenterTemp - halfPointerWidth, bubbleBottom);
				bubblePath.lineTo(pointerCenterTemp, viewBottom);
				bubblePath.close();
			}
			else{
				float bubbleTop = t + pointerHeight;
				
				// Start at the point of the triangle at the top and draw the line down to the top of the round rect.
				bubblePath.moveTo(pointerCenterTemp, t);
				bubblePath.lineTo(pointerCenterTemp - halfPointerWidth, bubbleTop);

				// Move up to the top left corner and draw the corner.
				bubblePath.lineTo(l + cornerRadius, bubbleTop);
				cornerRect.offsetTo(l, bubbleTop);
				bubblePath.arcTo(cornerRect, 270f, -90f);

				// Move up to the bottom left corner and draw the corner.
				bubblePath.lineTo(l, b - cornerRadius);
				cornerRect.offsetTo(l, b - cornerDiameter);
				bubblePath.arcTo(cornerRect, 180f, -90f);
				
				// Move to the bottom right corner of the bubble rect and draw the corner.
				bubblePath.lineTo(r - cornerRadius, b);
				cornerRect.offsetTo(r - cornerDiameter, b - cornerDiameter);
				bubblePath.arcTo(cornerRect, 90f, -90f);

				// Move up to the top right corner and draw the corner.
				bubblePath.lineTo(r, bubbleTop + cornerRadius);
				cornerRect.offsetTo(r - cornerDiameter, bubbleTop);
				bubblePath.arcTo(cornerRect, 0f, -90f);

				// Finish the top and the right hand side of the pointer.
				bubblePath.lineTo(pointerCenterTemp + halfPointerWidth, bubbleTop);
				bubblePath.lineTo(pointerCenterTemp, t);
				bubblePath.close();
			}
		}

		// No pointer.  Just make a round rect for the bubble.
		else
		{
			bubblePath.addRoundRect(new RectF(l, t, r, b), cornerRadius, cornerRadius, Path.Direction.CW);
		}
	}

	private LinearGradient calculateGradient(int l, int t, int r, int b)
	{
		return new LinearGradient(0, 0, 0, b - t, bubbleGradientTopColor, bubbleGradientBottomColor, Shader.TileMode.CLAMP);
	}
}
