package com.aniways.blur;

import com.aniways.Log;
import com.aniways.ViewUtils;
import com.aniways.data.AniwaysPrivateConfig;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.PorterDuff.Mode;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.drawable.ColorDrawable;
import android.graphics.drawable.Drawable;
import android.support.v8.renderscript.Allocation;
import android.support.v8.renderscript.Element;
import android.support.v8.renderscript.RenderScript;
import android.support.v8.renderscript.ScriptIntrinsicBlur;
import android.view.View;
import android.view.ViewTreeObserver;
import android.view.WindowManager.LayoutParams;

/**
 * Class for handling RenderScript based blur
 */
public class BlurRenderer {

	// Constant used for scaling the off-screen bitmap
	private static final float BITMAP_SCALE_FACTOR = AniwaysPrivateConfig.getInstance().blurBitmapScaleFactor;
	private static final String TAG = "AniwaysBlurRenderer";
	private static boolean sDoNotInitRS = false;

	private static ScriptIntrinsicBlur sScript;
	private static Context sContext;
	private static RenderScript sRS;

	private final View mView;
	private final Canvas mCanvas;
	private final Matrix mMatrixScale;

	private Bitmap mBitmap;
	private Allocation mAllocationBitmap;
	private boolean mPerformBlur = false;
	private float mRadius = 1f;
	private Bitmap mask;

	public static void initRenderScript(Context context){
		if(!AniwaysPrivateConfig.getInstance().getUseBlurEffect()){
			return;
		}
		
		try{
			sContext = context.getApplicationContext();
			sRS = RenderScript.create(context.getApplicationContext());
			sScript = ScriptIntrinsicBlur.create(sRS, Element.U8_4(sRS));	
		}
		catch(Throwable ex){
			sDoNotInitRS = true;
			Log.e(true, TAG, "Could not init renderscript", ex);
		}
	}

	public static RenderScript getRS(){
		if (sRS == null){
			if(sDoNotInitRS){
				// do nothing - we will return null
			}
			else{
				initRenderScript(sContext);
			}
		}
		if(sRS == null){
			Log.w(false, TAG, "Returning null RS");
		}
		return sRS;
	}

	/**
	 * Default constructor
	 */
	public BlurRenderer(View view) {
		mView = view;

		sContext = view.getContext().getApplicationContext();

		// Prepare the Canvas
		mCanvas = new Canvas();

		// Prepare matrices for scaling up/down the off-screen bitmap
		mMatrixScale = new Matrix();

		// RenderScript related variables
		mPerformBlur = AniwaysPrivateConfig.getInstance().getUseBlurEffect(); 
		if(mView.isInEditMode()){
			mPerformBlur = false;
		}
		
		if(sRS == null){
			this.mPerformBlur = false;
			Log.w(false, TAG, "RS null in cTor");
		}
	}

	/**
	 * Must be called from owning View.onAttachedToWindow
	 */
	public void onAttachedToWindow() {
		// Start listening to onDraw calls
		mView.getViewTreeObserver().addOnPreDrawListener(onPreDrawListener);
	}

	/**
	 * Must be called from owning View.onDetachedFromWindow
	 */
	@SuppressLint("MissingSuperCall")
	public void onDetachedFromWindow() {
		// Remove listener
		mView.getViewTreeObserver().removeOnPreDrawListener(onPreDrawListener);
	}

	/**
	 * Returns true if this draw call originates from this class and is meant to
	 * be an off-screen drawing pass.
	 */
	public boolean isOffscreenCanvas(Canvas canvas) {
		return canvas == mCanvas;
	}

	/**
	 * Applies blur to current off-screen bitmap
	 */
	public void applyBlur() {

		if(!this.mPerformBlur){
			return;
		}
		
		RenderScript rs = getRS();
		if(rs == null){
			this.mPerformBlur = false;
			Log.w(false, TAG, "Not applyig blur cause couldnt create renderscript");
			return;
		}

		mAllocationBitmap.copyFrom(mBitmap);
		sScript.setRadius(mRadius);
		sScript.setInput(mAllocationBitmap);
		sScript.forEach(mAllocationBitmap);
		mAllocationBitmap.copyTo(mBitmap);
	}

	/**
	 * Draws off-screen bitmap into current canvas
	 */
	public void drawToCanvas(Canvas canvas) {
		
		if(getRS() == null){
			this.mPerformBlur = false;
			Log.w(false, TAG, "RS null in drawToCanvas");
		}
		
		if(!mPerformBlur || mBitmap == null){
		    Drawable background = mView.getBackground();
            if(background != null){
                background.setBounds(0, 0, mView.getWidth(), mView.getHeight());
                background.draw(canvas);
            }
			return;
		}
		
		if(mask != null && !AniwaysPrivateConfig.getInstance().performLiveBlur){
			canvas.drawBitmap(mask, new Matrix(), null);
            return;
		}
		
		
		// This will be our mask, we draw the background in full color, and then we will later 
		// use the PorterDuff paint to only show the part of the blurred bitmap that is covered by the background 
		// drawable
		mask = drawableToBitmap (mView.getBackground(), mView.getWidth(), mView.getHeight());

		if(mask == null){
			canvas.drawBitmap(mBitmap, new Matrix(), null);
			return;
		}

		// We're going to apply this paint eventually using a porter-duff xfer mode.
		// This will allow us to only overwrite certain pixels.
		Paint xferPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
		xferPaint.setXfermode(new PorterDuffXfermode(Mode.SRC_IN));

		// Now we draw the blurred bitmap, and then use the mask with the PorterDuff paint to only 
		// leave out the parts we want
		Canvas resultCanvas = new Canvas(mask);
		resultCanvas.save();
		resultCanvas.scale(1/BITMAP_SCALE_FACTOR, 1/BITMAP_SCALE_FACTOR);
		resultCanvas.drawBitmap(mBitmap, 0, 0, xferPaint);

		// Draw the background
		Drawable background = mView.getBackground();
		if(background != null){
			background.setBounds(0, 0, mask.getWidth(), mask.getHeight());
			resultCanvas.restore();
			background.draw(resultCanvas);
		}

		// Draw the result to the canvas
		canvas.drawBitmap(mask, new Matrix(), null);
	}

	private Bitmap drawableToBitmap(Drawable drawable, int width, int height) {
		if(drawable == null){
			return null;
		}
        boolean isColorDrawable = false;
        if(drawable instanceof ColorDrawable){
            isColorDrawable = true;
        }
		
		// Used to make the colors completely opaque, so the PorterDuff paint would show only the parts of
		// the blurred image that are covered by the background
		// TODO: Make sure this is the correct mode also when there is alpha involved
		if(!isColorDrawable) {
            drawable.setColorFilter(Color.BLACK, Mode.DST_OVER);
        }

		Bitmap bitmap;

        // Taking size from bitmap, and not view, cause it needs to fit it perfectly, and sometimes because of scaling, rounding to 4pix etc. the bitmap is not
		try {
			bitmap = Bitmap.createBitmap(width, height, Config.ARGB_8888);
			Canvas canvas = new Canvas(bitmap);
			drawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
			if(isColorDrawable){
                canvas.drawColor(Color.BLACK);
            } else {
                drawable.draw(canvas);
            }
		} catch (Throwable e) {
			Log.e(true, TAG, "Caught Exception in drawableToBitmap", e);
			bitmap = null;
		}

        if(!isColorDrawable) {
            // Clear the filter
            drawable.clearColorFilter();
        }

		return bitmap;
	}

	/**
	 * Private method for grabbing a "screenshot" of screen content
	 */
	@SuppressLint("NewApi") 
	private void drawOffscreenBitmap() {
		if(!this.mPerformBlur){
			return;
		}
		
		if(getRS() == null){
			this.mPerformBlur = false;
			Log.w(false, TAG, "RS null in drawOffscreenBitmap");
			return;
		}

		if(mBitmap != null && !AniwaysPrivateConfig.getInstance().performLiveBlur){
			return;
		}

		// Calculate scaled off-screen bitmap width and height
		// We use ceiling here , and after, before we make the last 2 bits to 0, we add 4, in order to
		// make sure that the blured image would be bigger than what we need. We later cut it using the mask
		// to the desired size
		int width = (int) Math.ceil((float)mView.getWidth() * BITMAP_SCALE_FACTOR);
		int height = (int) Math.ceil((float)mView.getHeight() * BITMAP_SCALE_FACTOR);

		// This is added due to RenderScript limitations I faced.
		// If bitmap width is not multiple of 4 - in RenderScript
		// index = y * width
		// does not calculate correct index for line start index.
		width = (width + 4) & ~0x03;

		// Width and height must be > 0
		width = Math.max(width, 1);
		height = Math.max(height, 1);

		mBitmap = Bitmap.createBitmap(width, height, Config.ARGB_8888);

		mAllocationBitmap = Allocation.createFromBitmap(sRS, mBitmap);
		Allocation.createSized(sRS, Element.U8_3(sRS), width * height);

		// Due to adjusting width into multiple of 4 calculate scale matrix
		// only here
		mMatrixScale.setScale((float) BITMAP_SCALE_FACTOR,(float) BITMAP_SCALE_FACTOR);

		// Restore canvas to its original state
		mCanvas.restoreToCount(1);
		mCanvas.setBitmap(mBitmap);
		// Using scale matrix will make draw call to match
		// resized off-screen bitmap size
		mCanvas.setMatrix(mMatrixScale);

		int b4 = mCanvas.save();
		
		int x = ViewUtils.getRelativeLeftToViewRoot(mView);
		int y = ViewUtils.getRelativeTopToViewRoot(mView);
		
		mCanvas.translate(-x, -y);

		// Start drawing from the root view
		View viewBehindPopup = ((IBlurLayout)mView).getViewBehindPopup();
		
		// This logic is especially for popups - take the root from the view
		// from behind the popup, since they are not in the same view tree
		if(viewBehindPopup != null){
			int saveBeforeRootBehindPopupDraw = mCanvas.save();
			View rootView = mView.getRootView();
			LayoutParams params = (LayoutParams) rootView.getLayoutParams();
			// Translate relative to the popup location on screen
			View behindPopupRootView = viewBehindPopup.getRootView();
			
			int keyboardHeight = viewInLandscape(rootView) ? getKeyboardHeight(behindPopupRootView) : 0;
			
			mCanvas.translate(-params.x, -(params.y + keyboardHeight));
			behindPopupRootView.draw(mCanvas);
            mCanvas.restoreToCount(saveBeforeRootBehindPopupDraw);
		}
		
		mView.getRootView().draw(mCanvas);

		mCanvas.restoreToCount(b4);

		this.applyBlur();
	}

	private boolean viewInLandscape(View view) {
		int orientation = view.getResources().getConfiguration().orientation;
		return orientation == Configuration.ORIENTATION_LANDSCAPE;
	}

	private int getKeyboardHeight(View rootView){
		Rect r = new Rect();
        rootView.getWindowVisibleDisplayFrame(r);

        int screenHeight = rootView.getRootView().getHeight();
        int keyboardHeight = screenHeight - r.bottom;
		return keyboardHeight > 100 ? keyboardHeight : 0;
	}

	/**
	 * Listener for receiving onPreDraw calls from underlying ui
	 */
	private final ViewTreeObserver.OnPreDrawListener onPreDrawListener = new ViewTreeObserver.OnPreDrawListener() {
		@Override
		public boolean onPreDraw() {
			try{
				// Only care if View we are doing work for is visible
				if (mView.getVisibility() == View.VISIBLE) {
					drawOffscreenBitmap();
				}
			}
			catch (Throwable e) {
				Log.e(true, TAG, "Caught Exception in onPreDraw", e);
			}
			return true;
		}
	};

	/**
	 * Set blur radius in screen pixels. Value is mapped in range [1, 254].
	 */
	public void setBlurRadius(float radius) {

		if(mView.isInEditMode()){
			mPerformBlur = false;
			return;
		}
		
		if(getRS() == null){
			this.mPerformBlur = false;
			Log.w(false, TAG, "RS null in setBlurRadius");
			return;
		}

		if(radius <= 0){
			this.mPerformBlur = false;
			return;
		}
		// Check if need different radiuses for different screen densities..

		// Map radius into scaled down off-screen bitmap size
		mRadius = radius * BITMAP_SCALE_FACTOR;

        // Legal range for blur radius is 1,25 inclusive
        if(mRadius > 25){
            mRadius = 25;
        }
        else if(mRadius < 1){
            mRadius = 1;
        }
		
		mBitmap = null;
		mask = null;

		mPerformBlur = AniwaysPrivateConfig.getInstance().getUseBlurEffect(); 
	}

}
