package com.aniways.analytics.service;

import java.util.List;
import java.util.concurrent.CountDownLatch;

import android.content.Intent;

import com.aniways.Log;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.analytics.db.IPayloadDatabaseLayer.EnqueueCallback;
import com.aniways.analytics.db.PayloadDatabase;
import com.aniways.analytics.db.PayloadDatabaseThread;
import com.aniways.analytics.flush.FlushThread;
import com.aniways.analytics.flush.FlushThread.BatchFactory;
import com.aniways.analytics.flush.FlushThread.FlushCallback;
import com.aniways.analytics.flush.FlushTimer;
import com.aniways.analytics.info.InfoManager;
import com.aniways.analytics.models.BasePayload;
import com.aniways.analytics.models.Batch;
import com.aniways.analytics.models.Context;
import com.aniways.analytics.request.BasicRequester;
import com.aniways.analytics.stats.AnalyticsStatistics;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysStatics;
import com.aniways.service.utils.AniwaysServiceUtils;

public class AniwaysAnalyticsService extends ContinuosIntentService {
	public static final String ACTION_ENQUEUE = "Enqueue";
	public static final String ACTION_ENQUEUE_ERROR = "Enqueue_Error";
	public static final String ACTION_FLUSH = "Flush";
	public static final String SERIALIZED_EVENT_KEY = "event";

	private static final String TAG = "AniwaysAnalyticsService";

	private FlushTimer mFlushTimer;
	private PayloadDatabase mDatabase;
	private PayloadDatabaseThread mDatabaseLayer;
	private FlushThread mFlushLayer;
	private AnalyticsStatistics mStatistics;
	private boolean mInitialized;
	private InfoManager mInfoManager;
	private volatile boolean mIsFlushingToAniwaysServer;
	private volatile boolean mIsFlushingErrorHandler;
	//Flushes on a clock timer
	private NonThrowingRunnable mFlushClock;
	//Factory that creates batches from payloads. Inserts system information into global batches
	private BatchFactory mBatchFactory;
	private volatile long mFlushingToAniwaysServerStartTime;
	private volatile long mFlushingToErrorHandlerStartTime;

	public AniwaysAnalyticsService() {
		super("AniwaysAnalyticsService");
	}

	public AniwaysAnalyticsService(String name) {
		super(name);
	}

	@Override
	public void onCreate(){
		super.onCreate();
		// TODO: this is a temp hack to prevent duplicate events, but it means we might lose events.
		//		 we need to solve this correctly and make sure the intent is not redelivered if the 
		//		event is in the DB, but not to kill the service right afetr (maybe seperate the flushing mechanism to a
		//		different service)
		this.setIntentRedelivery(false);

		try{
			// Init Aniways
			AniwaysStatics.init(this.getApplicationContext(), this,  true);

			// Init the Analytics service threads
			if (mInitialized) return;

			mFlushClock = new NonThrowingRunnable(TAG, "flush runnable", "", true) {
				private int mNumberOfEmptyDbFlushes = 0;

				@Override
				public void innerRun() {
					if (mDatabase.getRowCount() == 0 && mDatabase.getErrorsRowCount() == 0){
						mNumberOfEmptyDbFlushes++;
					}
					else{
						mNumberOfEmptyDbFlushes = 0;
					}
					if(mNumberOfEmptyDbFlushes > AniwaysPrivateConfig.getInstance().emptyAnalyticsFlushIntervalsBeforeKillingService){
						Log.v(TAG, "Stopping Analytics Service due to lack of activity");
						stopSelf();
						return;
					}
					if(mNumberOfEmptyDbFlushes == 0){
						Log.i(TAG, "Flushing in flush clock");
						flush();
					}
				}
			};

			mBatchFactory = new BatchFactory() {

				@Override
				public Batch create(List<BasePayload> payloads) {

					Batch batch = new Batch(payloads); 

					// add global batch settings from system information
					// We are creating them anew on every batch, and not once in on-create,
					// because we want the creation to be as close as possible to the sending.
					// The reason for this is that some of the building requires Internet connection, and if it
					// doesn't exist then we lack data. By building it b4 sending we increase the chance that
					// if there is no Internet connection at the time of the building and there is missing data
					// then the context will not be sent, because the batch itself will not be sent and viseversa.
					batch.setContext(new Context(mInfoManager.build(AniwaysAnalyticsService.this)));

					return batch;
				}
			};

			mIsFlushingToAniwaysServer = false;
			mIsFlushingErrorHandler = false;

			mStatistics = new AnalyticsStatistics();

			mInfoManager = new InfoManager();

			// create the database using the activity context
			mDatabase = PayloadDatabase.getInstance(this);

			BasicRequester requester = new BasicRequester(this);

			// now we need to create our singleton thread-safe database thread
			mDatabaseLayer = new PayloadDatabaseThread(mDatabase);
			mDatabaseLayer.start();

			// start the flush thread
			mFlushLayer = new FlushThread(requester, mBatchFactory, mDatabaseLayer);

			mFlushTimer = new FlushTimer(mFlushClock);

			mInitialized = true;

			// start the other threads
			mFlushTimer.start();
			mFlushLayer.start();
			
			// We do not want a single event to cause a flush (unless it exceeded our threshold..)
			//flush();
		}
		catch(Throwable e) {
			// TODO: Maybe let the Exception crash the process..
			Log.eToGaOnly(true, TAG, "Caught Exception While creating Analytics service", e);
		}
	}

	/**
	 * Stops the analytics client threads, and resets the client
	 */
	@Override
	public void onDestroy(){
		super.onDestroy();

		try{
			// stops the looper on the timer, flush, and database thread
			mFlushTimer.quit();
			mFlushLayer.quit();
			mDatabaseLayer.quit();

			// closes the database
			mDatabase.close();

			mInitialized = false;
		}
		catch(Throwable ex){
			Log.eToGaOnly(true, TAG, "Caught Exception while destroying the Analytics service", ex);
		}
	}

	@Override
	protected void onHandleIntent(Intent intent) {
		try{
			String action = intent.getAction();
			if(action == null){
				return;
			}
			if(action.equalsIgnoreCase(ACTION_ENQUEUE)){
				String jsonText = intent.getStringExtra(SERIALIZED_EVENT_KEY);
				enqueue(jsonText);
			}
			else if(action.equalsIgnoreCase(ACTION_ENQUEUE_ERROR)){
				String payload = intent.getStringExtra(SERIALIZED_EVENT_KEY);
				enqueueErrorPayload(payload);
			}
			else if(action.equalsIgnoreCase(ACTION_FLUSH)){
				Log.i(TAG, "Received request to flush, so flushing..");
				flush();
			}
			else{
				Log.e(true, TAG, "Unknown Action: " + intent.getAction());
			}
		}
		catch(Throwable ex){
			Log.eToGaOnly(true, TAG, "Caught Exception in onHandleIntent", ex);
		}
	}

	/**
	 * Enqueues an {@link com.aniways.analytics.models.Identify}, {@link com.aniways.analytics.models.Track}, {@link com.aniways.analytics.models.Alias},
	 * or any action of type {@link com.aniways.analytics.models.BasePayload}
	 * @param payload
	 */
	private void enqueue(final String payload) {

		mStatistics.updateInsertAttempts(1);

		final long start = System.currentTimeMillis(); 

		mDatabaseLayer.enqueue(payload, new EnqueueCallback() {

			@Override
			public void onEnqueue(boolean success, long rowCount) {

				long duration = System.currentTimeMillis() - start;
				mStatistics.updateInsertTime(duration);

				if (rowCount >= AniwaysPrivateConfig.getInstance().analyticsMaxEventsBeforeFlush) {
					Log.i(TAG, "Flushing because row count >= max defined. Count: " + rowCount + ". Max defined: " + AniwaysPrivateConfig.getInstance().analyticsMaxEventsBeforeFlush);
					flush();
				}
			}
		});
	}

	private void enqueueErrorPayload(final String payload) {

		mStatistics.updateInsertAttempts(1);

		final long start = System.currentTimeMillis(); 

		mDatabaseLayer.enqueueErrorPayload(payload, new EnqueueCallback() {

			@Override
			public void onEnqueue(boolean success, long rowCount) {

				long duration = System.currentTimeMillis() - start;
				mStatistics.updateInsertTime(duration);

				if (rowCount >= AniwaysPrivateConfig.getInstance().analyticsMaxEventsBeforeFlush) {
					Log.i(TAG, "Flushing to error handler because row count >= max defined. Count: " + rowCount + ". Max defined: " + AniwaysPrivateConfig.getInstance().analyticsMaxEventsBeforeFlush);
					flush();
				}
			}
		});
	}

	/**
	 * Blocks until the queue is flushed
	 */
	private synchronized void flush() {
		
		if(!AniwaysServiceUtils.shouldPerformFlush(this)){
			return;
		}
		
		if(mIsFlushingToAniwaysServer && ((System.currentTimeMillis() - mFlushingToAniwaysServerStartTime) < AniwaysPrivateConfig.getInstance().flushTimeout)){
			Log.v(TAG, "Not flushing to Aniways server since a previous flush is already running");
		}
		else{
			if(mIsFlushingToAniwaysServer){
				Log.w(true, TAG, "Previous flush took too long, so starting a new one: " + (System.currentTimeMillis() - mFlushingToAniwaysServerStartTime));
			}

			mIsFlushingToAniwaysServer = true;
			mFlushingToAniwaysServerStartTime = System.currentTimeMillis();
			mStatistics.updateFlushAttempts(1);

			final long start = System.currentTimeMillis(); 

			final CountDownLatch latch = new CountDownLatch(1);

			mFlushLayer.flushAniwaysEvents(new FlushCallback() {

				@Override
				public void onFlushCompleted(boolean success) {
					mIsFlushingToAniwaysServer = false;
					
					latch.countDown();

					if (success) {
						long currentTimeMillis = System.currentTimeMillis();
						AniwaysServiceUtils.setLastSuccessAnalyticsFlush(AniwaysAnalyticsService.this, currentTimeMillis);
						long duration = currentTimeMillis - start;
						mStatistics.updateFlushTime(duration);
						
						// If we need to stop the service as soon as a flush ends (assuming there are
						// no more events in the db) then do it now (if there is no other flush going on)
						if (mDatabase.getRowCount() == 0 && mDatabase.getErrorsRowCount() == 0){
							if(mIsFlushingErrorHandler == false && AniwaysPrivateConfig.getInstance().emptyAnalyticsFlushIntervalsBeforeKillingService == 0){
								Log.v(TAG, "Stopping Analytics Service after successful analytics flush");
								stopSelf();
							}
						}
					}
				}
			});
		}

		if(mIsFlushingErrorHandler && ((System.currentTimeMillis() - mFlushingToErrorHandlerStartTime) < AniwaysPrivateConfig.getInstance().flushTimeout)){
			Log.v(TAG, "Not flushing to error handler since a previous flush is already running");
		}
		else{
			if(mIsFlushingErrorHandler){
				Log.w(true, TAG, "Previous flush to error handler took too long, so starting a new one: " + (System.currentTimeMillis() - mFlushingToErrorHandlerStartTime));
			}

			mIsFlushingErrorHandler = true;
			mFlushingToErrorHandlerStartTime = System.currentTimeMillis();
			mStatistics.updateFlushAttempts(1);

			final long start = System.currentTimeMillis(); 

			final CountDownLatch latch = new CountDownLatch(1);

			mFlushLayer.flushErrorEvents(new FlushCallback() {

				@Override
				public void onFlushCompleted(boolean success) {
					mIsFlushingErrorHandler = false;

					latch.countDown();

					if (success) {
						long duration = System.currentTimeMillis() - start;
						mStatistics.updateFlushTime(duration);
						
						// If we need to stop the service as soon as a flush ends (assuming there are
						// no more events in the db) then do it now (if there is no other flush going on)
						if (mDatabase.getRowCount() == 0 && mDatabase.getErrorsRowCount() == 0){
							if(mIsFlushingToAniwaysServer == false && AniwaysPrivateConfig.getInstance().emptyAnalyticsFlushIntervalsBeforeKillingService == 0){
								Log.v(TAG, "Stopping Analytics Service after successful error handler flush");
								stopSelf();
							}
						}
					}
				}

			});
		}
	}
}
