package com.aniways.analytics.request.azure;

import java.io.UnsupportedEncodingException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;

/**
 * Represents a container for a storage key.
 */
 final class StorageKey {
    /**
     * Computes a signature for the specified string using the HMAC-SHA256 algorithm.
     * 
     * @param storageKey
     *            A <code>StorageKey</code> object that represents the storage key to use.
     * @param stringToSign
     *            The UTF-8-encoded string to sign.
     * 
     * @return A <code>String</code> that contains the HMAC-SHA256-encoded signature.
     * 
     * @throws IllegalArgumentException
     *             If the string to sign is not a valid Base64-encoded string.
     * @throws java.security.InvalidKeyException
     *             If the key is not a valid storage key.
     */
     static synchronized String computeMacSha256(final StorageKey storageKey, final String stringToSign)
            throws InvalidKeyException {
        if (storageKey.hmacSha256 == null) {
            storageKey.initHmacSha256();
        }

        byte[] utf8Bytes = null;
        try {
            utf8Bytes = stringToSign.getBytes(Constants.UTF8_CHARSET);
        }
        catch (final UnsupportedEncodingException e) {
            throw new IllegalArgumentException(e);
        }

        return Base64.encode(storageKey.hmacSha256.doFinal(utf8Bytes));
    }

    /**
     * Stores a reference to the hmacsha256 Mac.
     */
    private Mac hmacSha256;

    /**
     * Stores a reference to the hmacsha256 SecretKey.
     */
    private SecretKey key256;

    /**
     * Stores the key.
     */
    private byte[] key;

    /**
     * Creates an instance of the <code>StorageKey</code> class.
     * 
     * @param key
     *            An array of bytes that represent the storage key.
     */
     StorageKey(final byte[] key) {
    	this.key = key;
        this.hmacSha256 = null;
        this.key256 = null;
    }

    /**
     * Initializes the HMAC-SHA256 Mac and SecretKey.
     * 
     * @throws java.security.InvalidKeyException
     *             If the key is not a valid SecretKey according to specification.
     */
    private void initHmacSha256() throws InvalidKeyException {
        this.key256 = new SecretKeySpec(this.key, "HmacSHA256");
        try {
            this.hmacSha256 = Mac.getInstance("HmacSHA256");
        }
        catch (final NoSuchAlgorithmException e) {
            throw new IllegalArgumentException();
        }
        this.hmacSha256.init(this.key256);
    }
}
