package com.aniways.analytics.flush;

import com.aniways.Log;
import com.aniways.analytics.AniwaysAnalyticsReporter;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.analytics.db.IPayloadDatabaseLayer.ErrorPayloadCallback;
import com.aniways.analytics.db.IPayloadDatabaseLayer.PayloadCallback;
import com.aniways.analytics.db.IPayloadDatabaseLayer.RemoveCallback;
import com.aniways.analytics.db.PayloadDatabaseThread;
import com.aniways.analytics.models.BasePayload;
import com.aniways.analytics.models.Batch;
import com.aniways.analytics.request.BasicRequester;
import com.aniways.analytics.stats.AnalyticsStatistics;
import com.aniways.analytics.utils.LooperThreadWithHandler;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysPrivateConfig.AnalyticsCloud;

import java.util.List;

/**
 * A Looper/Handler backed flushing thread
 *
 */
public class FlushThread extends LooperThreadWithHandler {
	private static final String TAG = "AniwaysAnalyticsFlushThread";	

	/**
	 * A factory to create a batch around a list of payload actions
	 */
	public interface BatchFactory {
		public Batch create(List<BasePayload> payloads);
	}

	private BasicRequester requester;
	private PayloadDatabaseThread databaseLayer;
	private BatchFactory batchFactory;

	/**
	 * Callback for the #flush method
	 *
	 */
	public interface FlushCallback {
		/**
		 * Called when all messages have been flushed from the queue
		 * @param success True for successful flush, false for not. 
		 */
		public void onFlushCompleted(boolean success);
	}

	public FlushThread(BasicRequester requester, BatchFactory batchFactory, PayloadDatabaseThread databaseLayer) {
		super(TAG);

		this.requester = requester;
		this.batchFactory = batchFactory;
		this.databaseLayer = databaseLayer;
	}


	/**
	 * Gets messages from the database, makes a request to the server,
	 * deletes the sent messages from the database, and continues flushing
	 * if there's messages left in the queue.
	 * @param callback
	 */
	public void flushAniwaysEvents(final FlushCallback callback) {
		// ask the database for the next payload batch
		databaseLayer.nextPayload(new PayloadCallback() {

			@Override
			public void onPayload(
					final long minId, 
					final long maxId,
					final List<BasePayload> payloads) {

				// we are currently executing on the database
				// thread so we're still technically locking the 
				// database

				if (payloads.size() > 0) {
					// we have things to flush

					final long start = System.currentTimeMillis();

					// let's create the batch frame that we're gonna flush
					Batch batch = batchFactory.create(payloads);

					// now let's make a request on the flushing thread
					performRequest(batch.toString(), false, new RequestCallback() {

						@Override
						public void onRequestCompleted(boolean success) {
							// we are now executing in the context of the
							// flushing thread

							if (!success) {
								// if we failed at flushing (connectivity issues)
								// return
								if (callback != null) callback.onFlushCompleted(false);

							} else {

								long duration = System.currentTimeMillis() - start;
								AniwaysAnalyticsReporter.getStatistics().updateRequestTime(duration);

								// if we were successful, we need to 
								// first delete the old items from the
								// database, and then continue flushing

								databaseLayer.removePayloads(minId, maxId, new RemoveCallback() {

									@Override
									public void onRemoved(int removed) {
										// we are again executing in the context of the database
										// thread

										AnalyticsStatistics statistics = AniwaysAnalyticsReporter.getStatistics();

										if (removed == -1) {

											for (int i = 0; i < removed; i += 1)
												statistics.updateFailed(1);

											Log.eToGaOnly(true, TAG, "Failed to remove payload from the database. MinId-MaxId" + minId + "-" + maxId, null);

											if (callback != null) callback.onFlushCompleted(false);

										} else if (removed == 0) {

											for (int i = 0; i < removed; i += 1)
												statistics.updateFailed(1);

											Log.eToGaOnly(true, TAG, "Didn't end up removing anything from the database. MinId-MaxId" + minId + "-" + maxId, null);

											if (callback != null) callback.onFlushCompleted(false);

										} else {

											for (int i = 0; i < removed; i += 1){
												statistics.updateSuccessful(1);
											}

											// Not continuing flush for simplicity, and also not to create small files..
											// So if there are many many events, they will wait to the next flush
											if (callback != null) callback.onFlushCompleted(true);
										}
									}
								});
							}
						}
					});

				} else {
					// there is nothing to flush, we're done
					if (callback != null) callback.onFlushCompleted(true);
				}
			}

		});
	}

	public void flushErrorEvents(final FlushCallback callback) {
		// ask the database for the next payload batch
		databaseLayer.nextErrorPayload(new ErrorPayloadCallback() {

			@Override
			public void onPayload(
					final long minId, 
					final long maxId,
					final String payload) {

				// we are currently executing on the database
				// thread so we're still technically locking the 
				// database

				if (payload != null) {
					// we have things to flush

					final long start = System.currentTimeMillis();

					// now let's make a request on the flushing thread
					performRequest(payload, true, new RequestCallback() {

						@Override
						public void onRequestCompleted(boolean success) {
							// we are now executing in the context of the
							// flushing thread

							if (!success) {
								// if we failed at flushing (connectivity issues)
								// return
								if (callback != null) callback.onFlushCompleted(false);

							} else {

								long duration = System.currentTimeMillis() - start;
								AniwaysAnalyticsReporter.getStatistics().updateRequestTime(duration);

								// if we were successful, we need to 
								// first delete the old items from the
								// database, and then continue flushing

								databaseLayer.removeErrorPayloads(minId, maxId, new RemoveCallback() {

									@Override
									public void onRemoved(int removed) {
										// we are again executing in the context of the database
										// thread

										AnalyticsStatistics statistics = AniwaysAnalyticsReporter.getStatistics();

										if (removed == -1) {

											for (int i = 0; i < removed; i += 1)
												statistics.updateFailed(1);

											Log.eToGaOnly(true, TAG, "Failed to remove payload from the error database. MinId-MaxId" + minId + "-" + maxId, null);

											if (callback != null) callback.onFlushCompleted(false);

										} else if (removed == 0) {

											for (int i = 0; i < removed; i += 1)
												statistics.updateFailed(1);

											Log.eToGaOnly(true, TAG, "Didn't end up removing anything from the error database. MinId-MaxId" + minId + "-" + maxId, null);

											if (callback != null) callback.onFlushCompleted(false);

										} else {

											for (int i = 0; i < removed; i += 1){
												statistics.updateSuccessful(1);
											}

											// now we can initiate another flush to make
											// sure that there's nothing left 
											// in the database before we say we're fully flushed
											flushErrorEvents(callback);
										}
									}
								});
							}
						}
					});

				} else {
					// there is nothing to flush, we're done
					if (callback != null) callback.onFlushCompleted(true);
				}
			}

		});
	}


	/**
	 * Callback for when a request to the server completes
	 *
	 */
	public interface RequestCallback {
		public void onRequestCompleted(boolean success);
	}

	/**
	 * Performs the request on the 
	 * @param batch
	 * @param callback
	 */
	private void performRequest(final String content, final boolean isErrorHandler, final RequestCallback callback) {

		handler().post(new NonThrowingRunnable(TAG, "report result of request", "ErrorHandler: " + isErrorHandler, true) {

			@Override
			public void innerRun() {

				boolean success = false;
				AnalyticsCloud analyticsCloud = null;
				try{
					analyticsCloud = AniwaysPrivateConfig.getInstance().analyticsCloud; 
					if(isErrorHandler){
						success = requester.sendToErrorHandler(content);
					}
					else{
						success = requester.sendAnalytics(content, analyticsCloud);
					}

					if(success){
						Log.v(TAG, "Successfully sent a batch to the server" + ". ErrorHandler: " + isErrorHandler + ". Analytics cloud: " + analyticsCloud);
					}
					else{
						Log.w(false, TAG, "Could not send batch to the server" + ". ErrorHandler: " + isErrorHandler + ". Analytics cloud: " + analyticsCloud);
					}
				}
				catch(Throwable ex){
					Log.eToGaOnly(true, TAG, "Caught Exception in perform request on the flush thread" + ". ErrorHandler: " + isErrorHandler + ". Analytics cloud:" + analyticsCloud, ex);
				}
				finally{
					try{
						if (callback != null) callback.onRequestCompleted(success);
					}
					catch(Throwable ex){
						Log.eToGaOnly(true, TAG, "Caught Exception while calling the request completed callback following call" + ". ErrorHandler: " + isErrorHandler, ex);
					}
				}
			}
		});
	}
}