package com.aniways;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Locale;
import java.util.regex.Pattern;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.Editable;
import android.text.Spannable;
import android.text.TextUtils;
import android.util.Pair;

import com.aniways.analytics.AnalyticsReporter;
import com.aniways.data.AniwaysConfiguration.Verbosity;
import com.aniways.data.AniwaysPhraseReplacementData;
import com.aniways.data.AniwaysPhraseReplacementData.IOnNonEmptyParserSetListener;
import com.aniways.data.AniwaysPrivateConfig;
import com.aniways.data.AniwaysPrivateConfig.IconEncodingMethod;
import com.aniways.data.AniwaysStatics;
import com.aniways.data.JsonParser;
import com.aniways.data.Phrase;
import com.aniways.data.PhraseWithPartToReplace;
import com.aniways.volley.toolbox.Volley;

// Used to insert Anyways icons to text or convert Aniways icon to text (when sending or receiving messages)
public class AniwaysIconConverter {

	private static final String TAG = "AniwaysIconConverter";
	private static final int INDEX_NOT_FOUND = -1;
	private static final String SPACE = " ";


	/**
	 * Encode the message according to the configured method (either code or url
	 * @param textWithIcons the message
	 * @return the encoded text
	 */
	static String encodeMessage(Spannable textWithIcons, Context context, boolean autoreplaceTextWithIcons, IconEncodingMethod enforceIconEncodingMethod, boolean forceDoNotReplacePhraseWithEmoji){
		long startTime = System.currentTimeMillis();
		String result = null;

		Log.d(TAG, "Start encode");

		// Add auto-replace text with icons if needed
		if(autoreplaceTextWithIcons){
			// Do this cause for some reason, with some partners the text may still be hooked to the
			// text watchers of the edittext.
			textWithIcons = Editable.Factory.getInstance().newEditable(textWithIcons);
			if(AniwaysPrivateConfig.getInstance().autoReplaceKeyPhrasesOnEncode){
				Log.d(TAG, "Start autoreplace phrases on encode");
				autoreplaceTextWithIcons(textWithIcons, context, null, false);
				Log.d(TAG, "End autoreplace phrases on encode");
			}
		}

		if(enforceIconEncodingMethod == null){
			enforceIconEncodingMethod = AniwaysPrivateConfig.getInstance().iconEncodingMethod;
		}

		if (enforceIconEncodingMethod == IconEncodingMethod.Invisible){
			result = replaceAniwaysIconsWithTextEncodedAsInvisible(textWithIcons, forceDoNotReplacePhraseWithEmoji);
		} else if (enforceIconEncodingMethod == IconEncodingMethod.ViralLink){
			result = replaceAniwaysIconsWithTextEncodedAsInvisibleAndViralUrl(context, textWithIcons, forceDoNotReplacePhraseWithEmoji);
		}
		else{
			Log.e(true, TAG, "received illegal encoding method: " + AniwaysPrivateConfig.getInstance().iconEncodingMethod);
		}

		// Report timing
		if (AnalyticsReporter.isInitialized()){
			AnalyticsReporter.ReportTiming(Verbosity.Statistical, startTime, "Performance", "Encode Message" + " - " + AniwaysPrivateConfig.getInstance().iconEncodingMethod, String.valueOf(textWithIcons.length()), TAG, "num chars");
		}
		Log.d(TAG, "End encode");
		return result;
	}

	static void autoreplaceTextWithIcons(Spannable textWithIcons, Context context, IIconInfoDisplayer iconInfoDisplayer, boolean useSmallIcons) {
		Pattern autoreplacePattern = AniwaysPrivateConfig.getInstance().autoreplacePattern;
		if(autoreplacePattern != null){
			HashSet<String> ignorePhrases = AniwaysPrivateConfig.getInstance().autoreplaceIgnorePhrases;
			JsonParser parser = AniwaysPhraseReplacementData.getDataParser();
			AniwaysMarkerInserter.addSuggestionMarkersToText(textWithIcons, autoreplacePattern, ignorePhrases, parser);
			AniwaysSuggestionSpanForAutoreplace[] spans = textWithIcons.getSpans(0, textWithIcons.length(), AniwaysSuggestionSpanForAutoreplace.class);
			if(spans != null && spans.length > 0){
				for(AniwaysSuggestionSpanForAutoreplace span : spans){

					int start = textWithIcons.getSpanStart(span);
					Phrase phrase = span.phrase;
					IconData[] potentialIcons = phrase.icons.icons;
					ArrayList<IconData> icons = new ArrayList<IconData>();
					for (IconData ic : potentialIcons){
						if(ic.family.equals(AniwaysPrivateConfig.getInstance().autoReplaceEmoticonsInTextViewDefaultIconsFamily)){
							icons.add(ic);
						}
					}
					if(!icons.isEmpty()){
						IconData icon = icons.get(0);
						Log.i(TAG, "Autoreplacing phrase: " + phrase + " with icon: " + icon.getFileName());
						// Cannot use the method that uses the suggestion span because it is for EditText and not TextView 
						// - it puts the icon info displayer to be null and the suggestion displayer to be not null and thus 
						// the icon will be displayed in the size of the Edit text
						insertAniwaysIconToText(context, textWithIcons, icon, start, phrase, null, iconInfoDisplayer, IAniwaysImageSpan.ImageSpanMetadata.Empty, false, useSmallIcons, false);
						if(iconInfoDisplayer != null){
							textWithIcons.removeSpan(span);
						}
					}
					else{
						if(parser.getKeywordsVersion().equalsIgnoreCase(AniwaysPhraseReplacementData.EMPTY_PARSER_VERSION)){
							Log.w(false, TAG, "Autoreplacement: No icons for replacement of phrase: " + phrase + ". Map version: " + parser.getKeywordsVersion());
						}
						else{
							Log.w(true, TAG, "Autoreplacement: No icons for replacement of phrase: " + phrase + ". Map version: " + parser.getKeywordsVersion());
						}
					}
				}
			}
		}
	}

	/**
	 * Decode an Aniways message
	 * @param text - the editable text object in which the icons will be displayed
	 */
	static void decodeMessage(Context context, final Editable text, final ISuggestionDisplayer suggestionDisplayer, final IIconInfoDisplayer iconInfoDisplayer, final IAniwaysTextContainer textContainer, JsonParser parser, final IconEncodingMethod enforceIconDecodingMethod, final boolean useSmallIcons, final boolean isFromPaste, boolean useEmptyParser, final boolean fromTextWatcher) {
		//long startTime = System.currentTimeMillis();
		if(TextUtils.isEmpty(text)){
			return;
		}

		// Remove unneeded spans
		if(suggestionDisplayer == null){
			removeAllEditTextSpecificSpans(text);
		}
		if(iconInfoDisplayer == null){
			removeAllTextViewSpecificSpans(text);
		}

		AniwaysDecoderResult result = null;

		// If the parser is empty, then do the parsing, after a non empty parser is set..
		if(parser.isEmpty() && !useEmptyParser){
			// The app context is needed in case the decoding is performed after the activity has re-created and the context is
			// not usable
			final Context appContext = context.getApplicationContext();
			AniwaysPhraseReplacementData.addOnNonEmptyParserSetListener(new IOnNonEmptyParserSetListener(){

				@Override
				public void onNonEmptyParserSet(JsonParser newParser) {
					if(textContainer == null){
						Log.e(true, TAG, "Text container is null");
					}
					else{
						// If this is an empty parser, then we decode it when a parser is ready
						textContainer.removeTextWatchers();
					}

					decodeMessage(appContext, text, suggestionDisplayer, iconInfoDisplayer, textContainer, newParser, enforceIconDecodingMethod, useSmallIcons, isFromPaste, true, fromTextWatcher);

					if(textContainer != null){
						textContainer.addBackTheTextWatchers();
					}
				}	
			});
			
			Log.d(TAG, "Parser is empty, will decode again when there is a non empty one");
			
			return;
		}

        if(parser.isEmpty() && useEmptyParser) {
            Log.w(false, TAG, "Using empty parser to decode - might not get full functionality for message");
        }

		if(enforceIconDecodingMethod != null){
			if(enforceIconDecodingMethod == IconEncodingMethod.Invisible || enforceIconDecodingMethod == IconEncodingMethod.ViralLink){
				result = insertAniwaysIconsToTextEncodedAsInvisible(context, text, suggestionDisplayer, iconInfoDisplayer, parser, useSmallIcons, isFromPaste, fromTextWatcher);
			}
			else{
				Log.e(true, TAG, "received illegal encoding method: " + enforceIconDecodingMethod);
			} 
		}
		else{
			if (AniwaysPrivateConfig.getInstance().useOnlyConfiguredEncodingMethodForDecoding){
				if(AniwaysPrivateConfig.getInstance().iconEncodingMethod == IconEncodingMethod.Invisible){
					result = insertAniwaysIconsToTextEncodedAsInvisible(context, text, suggestionDisplayer, iconInfoDisplayer, parser, useSmallIcons, isFromPaste, fromTextWatcher);
				}
				else{
					Log.e(true, TAG, "received illegal encoding method: " + AniwaysPrivateConfig.getInstance().iconEncodingMethod);
				}
			}
			else{
				result = insertAniwaysIconsToTextEncodedAsInvisible(context, text, suggestionDisplayer, iconInfoDisplayer, parser, useSmallIcons, isFromPaste, fromTextWatcher);
			}
		}

		if(result == null || result.getError() != null){
			Log.w(false, TAG, "Could not decode message: " + text + ". Parser version: " + parser.getKeywordsVersion() + ". Because: " + (result != null && result.getError() != null ? result.getError().message : "result is null"));
		}

		// Report timing
		//if (AnalyticsReporter.isInitialized()){
		//AnalyticsReporter.ReportTiming(startTime, "Performance", "Decode Message", String.valueOf(text.length()), TAG, "num chars");
		//}
	}

	/**
	 * This will insert from a suggestion span
	 */
	static void insertAniwaysIconToText(Context context, Spannable spannable, AniwaysSuggestionSpan suggestionSpan, IconData icon, IAniwaysImageSpan.ImageSpanMetadata imageSpanMetadata, boolean useSmallIcons) {
		int[] startEnd = new int[] {spannable.getSpanStart(suggestionSpan), spannable.getSpanEnd(suggestionSpan)};

		int startOfReplacementText = startEnd[0];
		ISuggestionDisplayer suggestionDisplayer = suggestionSpan.suggestionDisplayer;

		// We add from a suggestion span, so request not to create another one..
		insertAniwaysIconToText(context, spannable, icon, startOfReplacementText, suggestionSpan.phrase, suggestionDisplayer, null, imageSpanMetadata, false, useSmallIcons, false);
	}

    //TODO: maybe can get selection origin instead of metadata, since we already have the icon
	static int insertAniwaysIconToText(Context context, Spannable spannable, IconData icon, int startOfReplacementText, Phrase phrase, ISuggestionDisplayer suggestionDisplayer, IIconInfoDisplayer iconInfoDisplayer, IAniwaysImageSpan.ImageSpanMetadata imageSpanMetadata, boolean addSuggestionSpan, boolean useSmallIcons, boolean expectUnicode) {
		//long startTime = System.currentTimeMillis();

		int end = 0;
		int lengthDifference = 0;
		String originalText = null;
		if(expectUnicode){
			end = startOfReplacementText + icon.getUnicodeToReplaceText().length();
			originalText = phrase.getPartToReplace();
		}
		else{
			end = startOfReplacementText + phrase.getLengthOfPartToReplace();
			originalText = spannable.subSequence(startOfReplacementText, end).toString();
		}

		AniwaysSuggestionSpan suggestionSpan = null;
		IAniwaysIconInfoSpan infoSpan = null;

		// Add a suggestion span to where we add the icon, so the icon could be clicked and a suggestion to replace it would appear
		if(suggestionDisplayer != null && addSuggestionSpan){
			suggestionSpan = new AniwaysSuggestionSpan(phrase, suggestionDisplayer, originalText);
		}

		// Add an info displayer span to where we add the icon, so the icon could be clicked and its info would appear
		if(iconInfoDisplayer != null){
			if(icon.hasExternalData()){
				infoSpan = new AniwaysExternalIconInfoSpan(icon, context);
			} else {
				infoSpan = new AniwaysInternalIconInfoSpan(phrase, icon, iconInfoDisplayer, originalText);
			}
		}

		// Remove any the background color and image spans that sit where we place the picture
		// The suggestion span remains in order to be able to change picture in the future
		// The code will not remove adjacent spans whose range includes the start or end points 
		IAniwaysWordMarkerSpan[] bcs = spannable.getSpans(startOfReplacementText, end, IAniwaysWordMarkerSpan.class);
		IAniwaysImageSpan[] iss = spannable.getSpans(startOfReplacementText, end, IAniwaysImageSpan.class);
		if(bcs != null && bcs.length > 0){
			for(IAniwaysWordMarkerSpan span : bcs){
				if(!Utils.isSpanAdjacentToRange(spannable, startOfReplacementText, end, span)){
					spannable.removeSpan(span);
				}
			}
		}

		if(iss != null && iss.length > 0){
			for(IAniwaysImageSpan span : iss){
				if(!Utils.isSpanAdjacentToRange(spannable, startOfReplacementText, end, span)){
					spannable.removeSpan(span);
				}
			}
		}

		AniwaysImageSpan is = null;

		int maxWidth = 0;
		int maxHeight = 0;
		AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
		boolean displayBig = false;
		
		//TODO we assume here that the IsuggestionDisplayer is TextView - but it can also be a custom view, like in Telegram
		if(useSmallIcons || (config.useSmallIcon(icon))){
			maxWidth = AniwaysPrivateConfig.getInstance().smallIconTextWidth; 
			maxHeight = AniwaysPrivateConfig.getInstance().smallIconTextHeight;
		}
		else if (suggestionDisplayer != null){
			maxWidth = config.iconInEditTextWidth; 
			maxHeight = config.iconInEditTextHeight;
		}
		else if (iconInfoDisplayer != null){
			// If they are stand alone then get the stand alone size
			// TODO: Right now, I do not trim for performance, but if the app displays an icon without trimming then it will often have
			//       a space at the end because we add a space when adding an icon
			// TODO: Right now I assume that the message doesn't have a prefix, like there is in Xabber..
			if(startOfReplacementText == 0 && (end == spannable.length() || (end == spannable.length() - 1 && spannable.subSequence(spannable.length() - 1, spannable.length()).toString().equals(SPACE))) && config.makeStandaloneIconsBigger && config.canBeDisplayedBig(icon)){
				maxWidth = config.bigIconWidth;
				maxHeight = config.bigIconHeight;
				displayBig = true;
			}
			else{
				maxWidth = config.iconInTextViewWidth; 
				maxHeight = config.iconInTextViewHeight;
			}
		}
		else{ // they are both null - meaning this is done in the adding of auto replacement icons, or viral link in the encoding
			// We need this temp bitmap since in some cases (devices/apps) the adding of this span triggers a call to getSize
			// and we get a nullpointer exception..
			Bitmap tmpBitmap = BitmapFactory.decodeResource(context.getResources(), android.R.drawable.ic_menu_call);
			is = new AniwaysImageSpan(tmpBitmap, phrase, icon, imageSpanMetadata.iconSelectionOrigin, context, 1, 1);
			lengthDifference = addSpans(spannable, startOfReplacementText, end, suggestionSpan, infoSpan, is, !expectUnicode);
			return lengthDifference;
		}

		// See if the icon is cached and take it from there if it is..
		String url = config.getIconUrl(icon, displayBig, false);

		// Check in-mem cache (and pre-installed) and disk cache
		Object data = Volley.getImageLoader().getCached(url, config.getMaxWidthForCache(icon), config.getMaxHeightForCache(icon), icon.getFileName());
		
		if(data != null){
			Log.v(TAG, "Creating image span from cached image to: " + icon.getFileName());
			is = icon.createImageSpan(data, phrase, imageSpanMetadata.iconSelectionOrigin, context, maxWidth, maxHeight, iconInfoDisplayer);
		    lengthDifference = addSpans(spannable, startOfReplacementText, end, suggestionSpan, infoSpan, is, !expectUnicode);
		}
		else{
			
			Log.v(TAG, "Getting image from network: " + icon.getFileName());
			float textSizeRawPixels = -1;
			
			if(suggestionDisplayer == null && suggestionSpan!= null){
				suggestionDisplayer = suggestionSpan.suggestionDisplayer;
			}
			
			AniwaysEditText editText = null;
			
			if(suggestionDisplayer != null){
				editText = (AniwaysEditText) suggestionDisplayer;
				textSizeRawPixels = editText.getTextSize();
			}
			else if(iconInfoDisplayer != null && iconInfoDisplayer instanceof AniwaysTextView){
				textSizeRawPixels = ((AniwaysTextView)iconInfoDisplayer).getTextSize();
			}
			
			AniwaysLoadingImageSpan lis = new AniwaysLoadingImageSpan(maxWidth, maxHeight, context, phrase, icon, imageSpanMetadata, textSizeRawPixels);
			lis.setImageUrl(url, Volley.getImageLoader(), config.getMaxWidthForCache(icon), config.getMaxHeightForCache(icon));

			// The order here is important to have the span set when the image wants to load
			lengthDifference = addSpans(spannable, startOfReplacementText, end, suggestionSpan, infoSpan, lis, !expectUnicode);

			//TODO: Terrible hack in order to attach the loading image span to the edit text if necessary.
			// Need to make sure that the IAniwaysTextContainer is always passed and then use it..
			if(editText != null){
				IAniwaysTextContainer textContainer = editText.getTextContainer();
				// No need to do this with the icon info displayer cause this happens before, or during setText,
				// after which the TextView calls onSetText and attaches itself to all LoadingImageSpans
				textContainer.getDynamicImageSpansContainer().addDynamicImageSpan(lis);
			}
		}

		//if (AnalyticsReporter.isInitialized()){
		//AnalyticsReporter.ReportTiming(Verbosity.Statistical, startTime, "Performance", "Insert Icons", String.valueOf(spannable.length()), TAG, "num chars");
		//}
		return lengthDifference;
	}

	private static int addSpans(Spannable spannable, int start, int end, AniwaysSuggestionSpan suggestionSpan, 
			IAniwaysIconInfoSpan infoSpan, IAniwaysImageSpan is, boolean replaceTextWithUnicode) {
		// Replace the phrase with a unicode
		int lengthDifference = 0;
		if(replaceTextWithUnicode){
			String replacementUnicode = is.getIcon().getUnicodeToReplaceText();
			((Editable)spannable).replace(start, end, replacementUnicode);
			int origLength = end - start;
			lengthDifference = replacementUnicode.length() - origLength;
			end = start + replacementUnicode.length();
		}

		if(suggestionSpan != null){
			spannable.setSpan(suggestionSpan, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
		}
		if(infoSpan != null){
			spannable.setSpan(infoSpan, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
		}
		spannable.setSpan(is, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
		return lengthDifference;
	}

	private static String replaceAniwaysIconsWithTextEncodedAsInvisible(Spannable spannable, boolean forceDoNotReplacePhraseWithEmoji){
		AniwaysDecoderResult decoderResult = encodeInternal(spannable, true);

		return encodeAniwaysMessageV2(decoderResult, forceDoNotReplacePhraseWithEmoji, false, false);
	}

	private static IAniwaysImageSpan[] getAllAniwaysContentInMessage(Spannable spannable){
		return spannable.getSpans(0, spannable.length(), IAniwaysImageSpan.class);
	}

	private static boolean doesMessageContainAniwaysContent(Spannable spannable){
		IAniwaysImageSpan[] aiss = getAllAniwaysContentInMessage(spannable);
		return aiss != null && aiss.length > 0;
	}

	private static String replaceAniwaysIconsWithTextEncodedAsInvisibleAndViralUrl(Context context,
			Spannable spannable, boolean forceDoNotReplacePhraseWithEmoji) {

		if(!doesMessageContainAniwaysContent(spannable)){
			return replaceAniwaysIconsWithTextEncodedAsInvisible(spannable, forceDoNotReplacePhraseWithEmoji);
		}

		Editable editable = Editable.Factory.getInstance().newEditable(spannable);
		AniwaysPrivateConfig privateConfigInstance = AniwaysPrivateConfig.getInstance();
		IconData iconData = new IconData(privateConfigInstance.viralLinkImageId, null, false, null, null, AssetType.Emoticons, IAniwaysImageSpan.AssetProvider.Aniways, false);
		Phrase p = new Phrase(privateConfigInstance.viralMessage + privateConfigInstance.viralUrl, iconData); 
		iconData.primaryPhrase = p;
		editable.append(privateConfigInstance.viralMessage).append(privateConfigInstance.viralUrl);
		insertAniwaysIconToText(context, editable, iconData, spannable.length(), p, null, null, IAniwaysImageSpan.ImageSpanMetadata.Empty, false, false, false);

		return replaceAniwaysIconsWithTextEncodedAsInvisible(editable, forceDoNotReplacePhraseWithEmoji);
	}

	private static AniwaysDecoderResult encodeInternal(Spannable spannable, boolean expectSuggestionSpans) {
		String origString = spannable.toString();

		AniwaysPrivateConfig cfg = AniwaysPrivateConfig.getInstance();

		AniwaysDecoderResult decoderResult = new AniwaysDecoderResult(Editable.Factory.getInstance().newEditable(origString));

		IAniwaysImageSpan[] imageSpans = getAllAniwaysContentInMessage(spannable);
		if(imageSpans == null || imageSpans.length == 0)
		{
			return decoderResult;
		}

		Arrays.sort(imageSpans, new SpanStartSorter<IAniwaysImageSpan>(spannable));
		for (IAniwaysImageSpan span : imageSpans)
		{	
			int start = spannable.getSpanStart(span);
			int end = spannable.getSpanEnd(span);
			int[] startEnd = {start, end};
			start = startEnd[0];
			end = startEnd[1];
			String phraseName = span.getPhrase().getName().toLowerCase(Locale.US);
			String replacedString = span.getPhrase().getPartToReplace().toLowerCase(Locale.US);

			// Find the suggestion span to get the real original text..
			String originalText = null;
			AniwaysSuggestionSpan[] suggestionSpans = spannable.getSpans(start, end, AniwaysSuggestionSpan.class);
			if(suggestionSpans != null && suggestionSpans.length > 0){
				for(AniwaysSuggestionSpan ss : suggestionSpans){
					int ssStart = spannable.getSpanStart(ss);
					int ssEnd = spannable.getSpanEnd(ss);
					if(ssStart == start && ssEnd == end){
						originalText = ss.originalText;
						break;
					}
				}
			}
			if (span.getIcon().id == cfg.viralLinkImageId){
				originalText = replacedString;
			}
            else if (span.getIcon().id == cfg.animatedGifImageId){
                originalText = span.getIcon().getUnicodeToReplaceText();
            }
			else if(originalText == null){
				if(span.getImageSpanMetadata().iconSelectionOrigin == IAniwaysImageSpan.IconSelectionOrigin.OtherKeyboard){
					originalText = spannable.subSequence(start, end).toString();
					if(cfg.makeNonAniwaysIconsSmart){
						Log.w(true, TAG, "No original text in app icon, although, making non Aniways icons Smart. Phrase: " + span.getPhrase().toString());
					}
				}
				else{
                    if(expectSuggestionSpans) {
                        Log.w(true, TAG, "No original text, using phrase part to replace: " + span.getPhrase().toString());
                    }
					originalText = replacedString;
				}
			}
			else if(!replacedString.equalsIgnoreCase(originalText)){
				Log.w(true, TAG, "Original text != part to replace. Orig: " + originalText + ". Part to replace: " + replacedString);
				originalText = replacedString;
			}

			int indexOfreplacedPart = phraseName.indexOf(replacedString);
			int phraseStartIndex = start - indexOfreplacedPart;
			if(phraseStartIndex < 0){
				// This can happen in copy-paste scenarios, where we have only the picture of the part to replace, but not the whole phrase
				// Create a phrase which is only the part to replace..
				phraseName = replacedString;
				phraseStartIndex = start;
			}
			//TODO: this calculation is not entirely tru, since the end of the phrase might not all be present and yet the length
			// of the spannable is not reached. Need to check if indeed the rest of the phrase exists after the part to replace..
			int phraseNameLengthAfterReplacedPart = phraseName.length() - (indexOfreplacedPart + replacedString.length());
			if(end + phraseNameLengthAfterReplacedPart > spannable.length()){
				// This can happen in copy-paste scenarios, where we have only the picture of the part to replace, but not the whole phrase
				// Create a phrase which is only the part to replace..
				phraseName = replacedString;
				phraseStartIndex = start;
			}
			decoderResult.addIcon(phraseStartIndex, start, originalText, phraseName, span.getIcon(), span.getImageSpanMetadata().iconSelectionOrigin);
		}
		return decoderResult;
	}

	private static AniwaysDecoderResult insertAniwaysIconsToTextEncodedAsInvisible(Context context, Editable editable, ISuggestionDisplayer suggestionDisplayer, IIconInfoDisplayer iconInfoDisplayer, JsonParser parser, boolean useSmallIcons, boolean isFromPaste, boolean fromTextWatcher) {

		AniwaysDecoderResult result = decodeAniwaysMessageV2(editable, null, parser);

		if(result.getError() != null){
			Log.v(TAG, "Could not decode message as invisible because: " + result.getError().message, result.getError().exception);
			return result;
		}

		decodeInternal(context, editable, suggestionDisplayer, iconInfoDisplayer, parser, result, useSmallIcons, isFromPaste, fromTextWatcher);
		return result;
	}

	private static void decodeInternal(Context context, Editable editable, ISuggestionDisplayer suggestionDisplayer, IIconInfoDisplayer iconInfoDisplayer, JsonParser parser, AniwaysDecoderResult result, boolean useSmallIcons, boolean isFromPaste, boolean fromTextWatcher) {

		//The deletes are ordered, the indexes, are post the following deletes.
		// Need this mechanism because of a bug in several android versions where editable.replace() removes the spans in some cases,
		// so can't replace the contents of an editable text with the message in decoder result..
		if(result.getDeletes() != null && !result.getDeletes().isEmpty()){
			for(Pair<Integer,Integer> delete : result.getDeletes()){
				editable.delete(delete.first, delete.second);
			}
		}

		int offset = 0;
		AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
		for (AniwaysDecoderIconData icon : result.getIcons()){

			int startIndex = icon.subphraseStartIndex + offset;
			String replacementText = icon.replacementText;

			int endIndex = startIndex + replacementText.length();

			// Insert the icon..
			Boolean set = true;
			IAniwaysImageSpan[] imageSpans = editable.getSpans(startIndex, endIndex, IAniwaysImageSpan.class);
			for (IAniwaysImageSpan span : imageSpans){
				if (editable.getSpanStart(span) >= startIndex && editable.getSpanEnd(span) <= endIndex){
					editable.removeSpan(span);
				}
				else {
					set = false;
					break;
				}
			}
			if (set) {
				if (icon.icon == null){
					continue;
				}

				// If the icon is a fallback to emoji then we might have an emoji code beneath and 
				// we would like to replace it with the default phrase to make interactive and cover with an
				// image. If the phrase is an actual mapped phrase then we dont do this (TODO: There could be a case
				// where the phrase is not an emoji unicode, but is not mapped in this clients mapping, so we treat this
				// as it it were emoji and we replace with the primary phrase if its 4 chars and less (the max length of emoji unicode). 
				// Fix this somehow - could be problematic since we might not know all emoji unicodes)
				Phrase phrase = parser.getPhraseByName(icon.phraseName);
				if(phrase == null){
					// If the text below the icon is longer than 4 chars then it cannot be an emoji unicode, even
					// if it is not mapped, so we will treat it as an unmapped phrase
					if(icon.icon.hasEmojiFallback() && icon.phraseName != null && icon.phraseName.length() < 5){
						if(icon.icon.primaryPhrase == null || !icon.icon.primaryPhrase.isFullReplacement()){
							Log.w(true, TAG, "Icon with emoji fallback doesn't have a full replacement phrase . Id: " + icon.icon.id);
							continue;
						}
						phrase = icon.icon.primaryPhrase;
						String phraseName = icon.icon.primaryPhrase.getName();
						editable.replace(startIndex, endIndex, phraseName);
						// The insertAniwaysIconToText method brings back the emoji unicode, so no need these offsets
						// because it will later add an offset according to the change in the insert method
						// (would probably do the opposite of these offsets, but still, the code is more general like this
						// an we don't need a condition whether to add offsets after the inserts or not..).
						offset -= (endIndex - startIndex);
						offset += phraseName.length();
					}
					else{
						// This is legal as it could be that the icon is sent from either a previous or earlier version.
						Log.i(TAG, "received phrase name which is not present in the map: " + icon.phraseName + " Map version: " + parser.getKeywordsVersion());

						// Create a temporary phrase which is not in the map..
						phrase = new PhraseWithPartToReplace(icon.phraseName, icon.replacementText, icon.icon);
					}
				}
                IAniwaysImageSpan.ImageSpanMetadata meta = new IAniwaysImageSpan.ImageSpanMetadata(isFromPaste ? IAniwaysImageSpan.IconSelectionOrigin.Paste : IAniwaysImageSpan.IconSelectionOrigin.Unknown, icon.icon.assetType, icon.icon.assetProvider);
				offset += insertAniwaysIconToText(context, editable, icon.icon, startIndex, phrase, suggestionDisplayer, iconInfoDisplayer, meta, config.makeNonAniwaysIconsSmart, useSmallIcons, false);
			}
		}

		// Now find emoji that are not covered by an image and cover them!!
		handleUncoveredEmoji(editable, 0, editable.length(), parser, context, suggestionDisplayer, iconInfoDisplayer, useSmallIcons, isFromPaste, fromTextWatcher);

		// Report timing
		if (AnalyticsReporter.isInitialized()){
			//AnalyticsReporter.ReportTiming(startTime, "Icon Inserter", "Icon Inserter to text as url", String.valueOf(editable.length()), TAG, "num chars");
		}
	}

	/**
	 * This will replace the emoji with its primary phrase, and cover with an image..
	 */
	static boolean handleUncoveredEmoji(Editable text, int start, int end, JsonParser parser, Context context, ISuggestionDisplayer suggestionDisplayer, IIconInfoDisplayer iconInfoDisplayer, boolean useSmallIcons, boolean fromPaste, boolean fromTextWatcher){
		boolean result = false;
		int skip;
		// TODO: what about 4 char emoji?
		// TODO: what if the first 2 chars are the last part of a 4 part emoji?
		if(start > 0 && end > start && Character.isLowSurrogate(text.charAt(start))){
			start --;
		}
		if(end > 0 && end < text.length() && Character.isHighSurrogate(text.charAt(end-1))){
			end++;
		}

		AniwaysPrivateConfig config = AniwaysPrivateConfig.getInstance();
		for (int i = start; i < end; i += skip) {
			skip = 0;
			IconData icon = null;

			// This supports the old iOS encoding
			//char c = text.charAt(i);
			//if (isSoftBankEmoji(c)) {
			//    icon = getSoftbankEmojiResource(c);
			//    skip = icon == 0 ? 0 : 1;
			//}

			if (icon == null) {
				int unicode = Character.codePointAt(text, i);
				skip = Character.charCount(unicode);

				if (unicode > 0xff) {
					icon = parser.getEmoji(unicode);
				}
				
				boolean secondUnicodeExists = false;
				int followUnicode = -1;
				if (icon == null && i + skip < end) {
					followUnicode = Character.codePointAt(text, i + skip);
					icon = parser.getEmoji(new int[] { unicode, followUnicode });
					if(icon != null){
						secondUnicodeExists = true;
						skip += Character.charCount(followUnicode);
					}
				}
				
				// Add the emoji varient selector if needed
				if (icon != null && i + skip < end) {
					int nextUnicode = Character.codePointAt(text, i + skip);
					if(nextUnicode == EmojiWithVarientSelector.EMOJI_VARIENT_SELECTOR){
						IconData oldIcon = icon;
						if(secondUnicodeExists){
							icon = parser.getEmojiWithVarientSelector(new int[] { unicode, followUnicode });
						}
						else{
							icon = parser.getEmojiWithVarientSelector(unicode);
						}
						if(icon == null){
							icon = new EmojiWithVarientSelector(oldIcon);
							parser.addEmojiWithVarientSelector(icon);
						}
						skip += Character.charCount(nextUnicode);
					}
				}
			}

			if (icon != null) {
				// Do nothing if there is already an image span on the emoji..
				boolean isCovering = false;
				IAniwaysImageSpan[] spans = text.getSpans(i, i+skip, IAniwaysImageSpan.class);
				if(spans != null && spans.length > 0){
					for(IAniwaysImageSpan span : spans){
						int startSpan = text.getSpanStart(span);
						int endSpan = text.getSpanEnd(span);
						if(startSpan != i+skip && endSpan != i){
							isCovering = true;
							continue;
						}
					}
				}

				if(isCovering){
					continue;
				}

				// Replacing the emoji unicode with the primary phrase (in order to make the icon interactive)
				if(icon.primaryPhrase != null && icon.primaryPhrase.isFullReplacement()){
					// TODO: We rely here that this is a phrase in which the name==partToRepplace
					//String phraseName = icon.primaryPhrase.getName();
					//text.replace(i, i+skip, phraseName);
					// TODO: addition of the suggestion span makes the icon interactive. Need to
					// define in config if we want this behavior..
                    IAniwaysImageSpan.ImageSpanMetadata meta = new IAniwaysImageSpan.ImageSpanMetadata(fromPaste ? IAniwaysImageSpan.IconSelectionOrigin.Paste : (fromTextWatcher ? IAniwaysImageSpan.IconSelectionOrigin.OtherKeyboard : IAniwaysImageSpan.IconSelectionOrigin.Unknown), icon.assetType, icon.assetProvider);
					insertAniwaysIconToText(context, text, icon, i, icon.primaryPhrase, suggestionDisplayer, iconInfoDisplayer, meta, config.makeNonAniwaysIconsSmart, useSmallIcons, true);
					result = true;
					//skip = icon.getUnicodeRepresentation().length();
					//length = text.length();
				}				
			}
		}
		return result;
	}

	private static String safeSubstring(String text, int start, String expected){
		try{
			String substring =  text.substring(start, start + expected.length());
			if(!substring.equalsIgnoreCase(expected)){
				throw new Exception("Expected != substring. Expected: " + expected + ". Substring: " + substring);
			}
			return substring;
		}
		catch(Throwable ex){
			Log.e(true, TAG, "Error in safe substring. Text: " + text + ". start: " + start + " . Length: " + (expected == null ? "NULL" : expected.length()) + ". Expected: " + expected);
			return expected;
		}
	}

	private static String encodeAniwaysMessageV2(AniwaysDecoderResult decoderResult, boolean forceDoNotReplacePhraseWithEmoji, boolean justBringBackOriginalText, boolean alsoConvertNonContextualIconsToText){

		String result = decoderResult.getMessage().toString();

		AniwaysPrivateConfig cfg = AniwaysPrivateConfig.getInstance();
		Log.d(TAG, "Encoding message: " + result);
		Log.d(TAG, "Encoding message params: cfg.encodeEmojisWithEmojiUnicode: " + cfg.encodeEmojisWithEmojiUnicode + 
				". cfg.encodeIconsWithEmojiReplacementWithEmojiUnicode: " + cfg.encodeIconsWithEmojiReplacementWithEmojiUnicode + 
				". forceDoNotReplacePhraseWithEmoji: " + forceDoNotReplacePhraseWithEmoji + 
				". justBringBackOriginalText: " + justBringBackOriginalText);

		if(decoderResult.getIcons().isEmpty()){
			Log.d(TAG, "No icons to encode. Returning");
			return result;
		}

		int offset = 0;
		for (AniwaysDecoderIconData icon : decoderResult.getIcons())
		{	
			String phraseName = icon.phraseName;
			// TODO: Capitalization here is not correct. Get the original string from the suggestion span somehow..
			String replacedString = icon.replacementText;

			IconData iconData = icon.icon;
			if(iconData == null){
				Log.e(true, TAG, "No icon id found for icon: " + icon);
				continue;
			}

			// If icon is emoji, then just replace the text it covers with the emoji unicode
			// If icon has an emoji fallback then replace the text with the emoji unicode and then also code the icon to use
			int offsetToAdd = 0;

            // If icon is Animated Gif then remove it
            // TODO: When we support encode/decode of these icons then need to remove only on encode for sending a message, and only sometimes even then..
            if(icon.icon.id == cfg.animatedGifImageId){
                Log.d(TAG, "Removing icon for Animated gif: " + icon + ". Offset: " + offset + ". Message: " + result);
                String replacementUnicode = icon.icon.getUnicodeToReplaceText();
                Log.d(TAG, "icon for Animated gif. Unicode: " + replacementUnicode + ". text: " + replacedString +
                        ". Actual string there: " + safeSubstring(result, icon.subphraseStartIndex + offset, replacementUnicode));
                result = replaceAtLocation(result, icon.subphraseStartIndex + offset, replacementUnicode.length(), "");
                offsetToAdd -= replacementUnicode.length();

                // Remove the space before the icon
                if(icon.phraseStartIndex > 0){
                    if(result.subSequence(icon.phraseStartIndex - 1 + offset, icon.phraseStartIndex + offset).toString().equalsIgnoreCase(" ")){
                        result = replaceAtLocation(result, icon.phraseStartIndex - 1 + offset, 1, "");
                        offsetToAdd -= 1;
                    }
                }

                offset += offsetToAdd;
                continue;
            }

            if (justBringBackOriginalText){
                if(IAniwaysImageSpan.IconSelectionOrigin.isFromContextualReplacement(icon.selectionOrigin)) {
                    // Replace the unicode representation back to text
                    String replacementUnicode = icon.icon.getUnicodeToReplaceText();
                    Log.d(TAG, "Replacing emoji unicode back to text. Unicode: " + replacementUnicode + ". text: " + replacedString +
                            ". Actual string there: " + safeSubstring(result, icon.subphraseStartIndex + offset, replacementUnicode));
                    result = replaceAtLocation(result, icon.subphraseStartIndex + offset, replacementUnicode.length(), replacedString);
                    offsetToAdd -= replacementUnicode.length();
                    offsetToAdd += replacedString.length();
                }
                else if(alsoConvertNonContextualIconsToText){
                    // This is for the notifications menu, so we convert icons with emoji fallback to emoji, and icons without fallback to text
                    if(icon.icon.hasEmojiFallback()){
                        //There is already an emoji here, so we dont need to do anything here..
                    }
                    else{
                        // Replace the unicode representation back to text
                        String replacementUnicode = icon.icon.getUnicodeToReplaceText();
                        Log.d(TAG, "Replacing emoji unicode back to text. Unicode: " + replacementUnicode + ". text: " + replacedString +
                                ". Actual string there: " + safeSubstring(result, icon.subphraseStartIndex + offset, replacementUnicode));
                        result = replaceAtLocation(result, icon.subphraseStartIndex + offset, replacementUnicode.length(), replacedString);
                        offsetToAdd -= replacementUnicode.length();
                        offsetToAdd += replacedString.length();
                    }
                }
                else{
                    // If the icon was not added contextually then remove it..
                    String replacementUnicode = icon.icon.getUnicodeToReplaceText();
                    Log.d(TAG, "removing non contextual icon. Unicode: " + replacementUnicode + ". text: " + replacedString +
                            ". Actual string there: " + safeSubstring(result, icon.subphraseStartIndex + offset, replacementUnicode));
                    result = replaceAtLocation(result, icon.subphraseStartIndex + offset, replacementUnicode.length(), "");
                    offsetToAdd -= replacementUnicode.length();
                }
                offset += offsetToAdd;
                continue;
            }

			Log.d(TAG, "Encoding icon: " + icon + ". Offset: " + offset + ". Message: " + result);
            if(icon.icon.assetType == AssetType.Emoji && cfg.encodeEmojisWithEmojiUnicode){
				if(forceDoNotReplacePhraseWithEmoji && !justBringBackOriginalText){
					// Replace the unicode representation back to text
					String replacementUnicode = icon.icon.getUnicodeToReplaceText();
					Log.d(TAG, "Replacing emoji unicode back to text. Unicode: " + replacementUnicode + ". text: " + phraseName + 
							". Actual string there: " + safeSubstring(result, icon.subphraseStartIndex + offset, replacementUnicode));
					result = replaceAtLocation(result, icon.subphraseStartIndex + offset, replacementUnicode.length(), phraseName);
					offsetToAdd -= replacementUnicode.length();
					offsetToAdd += phraseName.length();
				}
				else{
					Log.d(TAG, "Leaving emoji as is: " + safeSubstring(result, icon.subphraseStartIndex + offset, icon.icon.getUnicodeRepresentation()));
                    String emojiUnicode = icon.icon.getUnicodeRepresentation();

                    int subphraseInsidePhraseIndex = icon.subphraseStartIndex - icon.phraseStartIndex;
                    try{
                        phraseName = replaceAtLocation(phraseName, subphraseInsidePhraseIndex, replacedString.length(), emojiUnicode);
                        Log.d(TAG, "Converting subphrase to emoji unicode: Phrase name: " + phraseName + ". Part to replace: " + replacedString + ". Emoji unicode: " + emojiUnicode + ". Emoji unicode there: " + safeSubstring(result, icon.subphraseStartIndex + offset, emojiUnicode));
                    }
                    catch(StringIndexOutOfBoundsException ex){
                        Log.e(true, TAG, "Caught a StringIndexOutOfBoundsException. PN: " + phraseName + ". RS: " + replacedString + ". phrase index: " + icon.phraseStartIndex + ". Subphrase index: " + icon.subphraseStartIndex, ex);
                        continue;
                    }
                    replacedString = emojiUnicode;
				}
			}
			else if(icon.icon.hasEmojiFallback() && cfg.encodeIconsWithEmojiReplacementWithEmojiUnicode && !forceDoNotReplacePhraseWithEmoji){
				String emojiUnicode = icon.icon.getUnicodeRepresentation();

				int subphraseInsidePhraseIndex = icon.subphraseStartIndex - icon.phraseStartIndex;
				try{
					phraseName = replaceAtLocation(phraseName, subphraseInsidePhraseIndex, replacedString.length(), emojiUnicode);
					Log.d(TAG, "Converting subphrase to emoji unicode: Phrase name: " + phraseName + ". Part to replace: " + replacedString + ". Emoji unicode: " + emojiUnicode + ". Emoji unicode there: " + safeSubstring(result, icon.subphraseStartIndex + offset, emojiUnicode));
				}
				catch(StringIndexOutOfBoundsException ex){
					Log.e(true, TAG, "Caught a StringIndexOutOfBoundsException. PN: " + phraseName + ". RS: " + replacedString + ". phrase index: " + icon.phraseStartIndex + ". Subphrase index: " + icon.subphraseStartIndex, ex);
					continue;
				}
				replacedString = emojiUnicode;
			}
			else {
				// Replace the unicode representation back to text
				String replacementUnicode = icon.icon.getUnicodeToReplaceText();
				Log.d(TAG, "Replacing emoji unicode back to text. Unicode: " + replacementUnicode + ". text: " + replacedString + 
						". Actual string there: " + safeSubstring(result, icon.subphraseStartIndex + offset, replacementUnicode));
				result = replaceAtLocation(result, icon.subphraseStartIndex + offset, replacementUnicode.length(), replacedString);
				offsetToAdd -= replacementUnicode.length();
				offsetToAdd += replacedString.length();
			}

			String iconIdString = Integer.toString(icon.icon.id, AniwaysPrivateConfig.getInstance().decoderRadix);

			// Determine length of the sequence - can be a multiple of 6 in default config
			int l = iconIdString.length();
			int baseChunkSize = AniwaysPrivateConfig.getInstance().decoderChunkSize;

			int length = l + 1; // because of the length bit
			int numberOfNeededChunks = (int) Math.ceil((float)length / (float)baseChunkSize);
			int decoderRadix = AniwaysPrivateConfig.getInstance().decoderRadix;
			int maxPossibleChunks = decoderRadix;
			String lengthBit = Integer.toString(numberOfNeededChunks -1, decoderRadix); 
			if(numberOfNeededChunks > maxPossibleChunks){
				Log.e(true, TAG, "Id is too long: " + iconIdString);
			}
			// Add zeroes at the beginning
			int sequenceLength = numberOfNeededChunks * baseChunkSize;
			for(int i = iconIdString.length(); i < sequenceLength - 1; i++){
				iconIdString = '0' + iconIdString;
			}
			iconIdString = lengthBit + iconIdString;
			String[] codepointStrings = AniwaysPrivateConfig.getInstance().decoderCodepointStrings;
			for(int i = 0; i < decoderRadix; i++){			
				iconIdString = iconIdString.replace(Integer.toString(i, decoderRadix), codepointStrings[i]);
			}

			// Add the icon id
			result = insertToString(result, iconIdString, icon.phraseStartIndex + offset);
			offset += iconIdString.length();

			// Add the subphrase start
			String delimiter = AniwaysPrivateConfig.getInstance().delimiterString;
			result = insertToString(result, delimiter, icon.subphraseStartIndex + offset);
			offset += delimiter.length();

			// Add the subphrase end
			result = insertToString(result, delimiter, icon.subphraseStartIndex + offset + replacedString.length());
			offset += delimiter.length();

			// Add the phrase end
			result = insertToString(result, delimiter, icon.phraseStartIndex + offset + phraseName.length());
			offset += delimiter.length();

			// In case there was a fallback to an emoji unicode..
			offset += offsetToAdd;
		}

		Log.d(TAG, "Returning encoded message: " + result);
		return result;
	}

	protected static String replaceAtLocation(String str, int index, int length, String stringToInsert) {
		String beforeRemove = str.substring(0, index);
		String afterRemove = str.substring(index + length);
		str = beforeRemove + stringToInsert + afterRemove;
		return str;
	}

	private static String insertToString(String source, String insert, int position) {
		String before = source.substring(0, position);
		String after = source.substring(position);
		source = before + insert + after;
		return source;
	}
	
	private static AniwaysDecoderResult decodeAniwaysMessageV2(Editable message, AniwaysDecoderResult result, JsonParser parser){
		//long startTime = System.currentTimeMillis();

		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		if(result == null){
			result = new AniwaysDecoderResult(message);
		}

		while (true){
			// Copy the message, so changes we make will not affect the original one..
			// If the result isn't null then, then no need to copy, cause it was copied by the last call (which created the result)
			message = Editable.Factory.getInstance().newEditable(result.getMessage());
			String messageString = message.toString();

			int iconIdSequenceLocation = indexOfAny(messageString, AniwaysPrivateConfig.getInstance().decoderCodepointStrings);
			if (iconIdSequenceLocation < 0){
				break;
			}

			String lengthBitString = messageString.substring(iconIdSequenceLocation, iconIdSequenceLocation + 1);
			int lengthBit = parseUnicodeCharsToInt(result, lengthBitString) + 1;
			int sequenceLength = AniwaysPrivateConfig.getInstance().decoderChunkSize * lengthBit;

			int encodeSequenceLocationEnd = iconIdSequenceLocation + sequenceLength;
			if (messageString.length() < encodeSequenceLocationEnd){
				result.setError("Received editable with sequence end out of the string: " + message.toString());
				break;
			}

			String encodingString = messageString.substring(iconIdSequenceLocation + 1, encodeSequenceLocationEnd);
			int iconId = parseUnicodeCharsToInt(result, encodingString);
			if(result.getError() != null){
				break;
			}
			if (iconId < 0){
				result.setError("Length < 0:" + message);
				break;
			}

			// TODO: replace iconid with path

			int startPhraseIndex = iconIdSequenceLocation;

			// Get the start index of the subphrase
			message = message.delete(iconIdSequenceLocation, encodeSequenceLocationEnd);
			//The deletes are ordered, the indexes, are post the following deletes.
			// Need this mechanism because of a bug in several android versions where editable.replace() removes the spans in some cases,
			// so can't replace the contents of an editable text with the message in decoder result..
			result.addDelete(iconIdSequenceLocation, encodeSequenceLocationEnd);
			String delimiter = AniwaysPrivateConfig.getInstance().delimiterString;
			int startSubphraseIndex = message.toString().indexOf(delimiter);
			if(startSubphraseIndex == -1){
				result.setError("Could not find startSubphrase:" + message);
				break;
			}

			// Get the end index of the subphrase
			message = message.delete(startSubphraseIndex, startSubphraseIndex+1);
			result.addDelete(startSubphraseIndex, startSubphraseIndex + 1);
			int endSubphraseIndex = message.toString().indexOf(delimiter);
			if(endSubphraseIndex == -1){
				result.setError("Could not find endSubphrase:" + message);
				break;
			}

			if(iconId == AniwaysPrivateConfig.getInstance().viralLinkImageId){
				message = message.delete(startSubphraseIndex, endSubphraseIndex);
				result.addDelete(startSubphraseIndex , endSubphraseIndex );
				endSubphraseIndex = message.toString().indexOf(delimiter);
				if(endSubphraseIndex == -1){
					result.setError("Could not find endSubphrase:" + message);
					break;
				}
			}

			// Get the end index of the phrase
			message = message.delete(endSubphraseIndex, endSubphraseIndex+1);
			result.addDelete(endSubphraseIndex, endSubphraseIndex + 1);
			int endPhraseIndex = message.toString().indexOf(delimiter);
			if(endPhraseIndex == -1){
				result.setError("Could not find endPhrase:" + message);
				break;
			}
			message = message.delete(endPhraseIndex, endPhraseIndex+1);
			result.addDelete(endPhraseIndex, endPhraseIndex + 1);

			if(iconId == AniwaysPrivateConfig.getInstance().viralLinkImageId){
				result.setMessage(message);
				continue;
			}

			// Get the replacement text
			String replacementText = message.subSequence(startSubphraseIndex, endSubphraseIndex).toString();

			// Get the phrase name
			String phraseName = message.subSequence(startPhraseIndex, endPhraseIndex).toString();

			// Get the icon name from the id
			IconData icon = parser.getIconById(iconId);
			if(icon == null){
				Log.v(TAG, "Could not find icon and phrase definitions for icon id: " + iconId + ". So, creating new ones. Message: " + message);
				// TODO: handle if it is an emoji
                //TODO: we assume the asset type and provider here, which is not necessarily correct!! also whether animated..
				icon = new IconData(iconId, "Unknown", false, null, null, AssetType.Emoticons, IAniwaysImageSpan.AssetProvider.Aniways, false);
                icon.primaryPhrase = new Phrase(phraseName, icon);
			}

			result.addIcon(startPhraseIndex, startSubphraseIndex, replacementText, phraseName.toLowerCase(Locale.US), icon, IAniwaysImageSpan.IconSelectionOrigin.Unknown);
			result.setMessage(message);
		}

		// Report timing
		if (AnalyticsReporter.isInitialized()){
			//AnalyticsReporter.ReportTiming(startTime, "Icon Inserter", "Icon Inserter to text as url", String.valueOf(editable.length()), TAG, "num chars");
		}

		return result;
	}

	// IndexOfAny strings
	//-----------------------------------------------------------------------
	/**
	 * <p>Find the first index of any of a set of potential substrings.</p>
	 *
	 * <p>A <code>null</code> String will return <code>-1</code>.
	 * A <code>null</code> or zero length search array will return <code>-1</code>.
	 * A <code>null</code> search array entry will be ignored, but a search
	 * array containing "" will return <code>0</code> if <code>str</code> is not
	 * null. This method uses {@link String#indexOf(String)}.</p>
	 *
	 * <pre>
	 * StringUtils.indexOfAny(null, *)                     = -1
	 * StringUtils.indexOfAny(*, null)                     = -1
	 * StringUtils.indexOfAny(*, [])                       = -1
	 * StringUtils.indexOfAny("zzabyycdxx", ["ab","cd"])   = 2
	 * StringUtils.indexOfAny("zzabyycdxx", ["cd","ab"])   = 2
	 * StringUtils.indexOfAny("zzabyycdxx", ["mn","op"])   = -1
	 * StringUtils.indexOfAny("zzabyycdxx", ["zab","aby"]) = 1
	 * StringUtils.indexOfAny("zzabyycdxx", [""])          = 0
	 * StringUtils.indexOfAny("", [""])                    = 0
	 * StringUtils.indexOfAny("", ["a"])                   = -1
	 * </pre>
	 *
	 * @param str  the String to check, may be null
	 * @param searchStrs  the Strings to search for, may be null
	 * @return the first index of any of the searchStrs in str, -1 if no match
	 */
	private static int indexOfAny(String str, String[] searchStrs) {
		if ((str == null) || (searchStrs == null)) {
			return INDEX_NOT_FOUND;
		}
		int sz = searchStrs.length;

		// String's can't have a MAX_VALUEth index.
		int ret = Integer.MAX_VALUE;

		int tmp = 0;
        for (String search : searchStrs) {
            if (search == null) {
                continue;
            }
            tmp = str.indexOf(search);
            if (tmp == INDEX_NOT_FOUND) {
                continue;
            }

            if (tmp < ret) {
                ret = tmp;
            }
        }

		return (ret == Integer.MAX_VALUE) ? INDEX_NOT_FOUND : ret;
	}

	public static boolean isEmpty(final CharSequence cs) {
		return cs == null || cs.length() == 0;
	}

	private static int parseUnicodeCharsToInt(AniwaysDecoderResult decoderResult, String source) {

		int decoderRadix = AniwaysPrivateConfig.getInstance().decoderRadix;
		String[] codepointStrings = AniwaysPrivateConfig.getInstance().decoderCodepointStrings;
		String baseDecoderRadixString = source;
		for(int i = 0; i < decoderRadix; i++){			
			baseDecoderRadixString = baseDecoderRadixString.replace(codepointStrings[i], Integer.toString(i, decoderRadix));
		}

		Integer result = -1;

		if(baseDecoderRadixString.length() > 0){
			try{
				result = Integer.valueOf(baseDecoderRadixString, decoderRadix);
			}
			catch(NumberFormatException ex){
                if(decoderResult != null) {
                    decoderResult.setError("Could not parse chars to int: " + source, ex);
                }
			}
		}

		return result;
	}

	/**
	 * Get the original message string without the Aniways Suffixes 
	 */
	static String getOriginalString(Spannable source, boolean alsoConvertNonContextualIconsToText, boolean expectSuggestionSpans) {
		AniwaysStatics.makeSureAniwaysIsInitialized(false);

		AniwaysDecoderResult decoderResult = encodeInternal(source, expectSuggestionSpans);

		return encodeAniwaysMessageV2(decoderResult, false, true, alsoConvertNonContextualIconsToText);
	}

	private static void removeAllTextViewSpecificSpans(Spannable spannable) {
		IAniwaysIconInfoSpan[] spans = spannable.getSpans(0, spannable.length(), IAniwaysIconInfoSpan.class);
		if (spans == null || spans.length == 0){
			return;
		}
		for (IAniwaysIconInfoSpan span : spans){
			spannable.removeSpan(span);
		}
	}

	private static void removeAllEditTextSpecificSpans(Spannable spannable) {
		IAniwaysWordMarkerSpan[] bcSpans = spannable.getSpans(0, spannable.length(), IAniwaysWordMarkerSpan.class);
		if (bcSpans == null || bcSpans.length == 0){
			return;
		}
		for (IAniwaysWordMarkerSpan span : bcSpans){
			spannable.removeSpan(span);
		}

		AniwaysSuggestionSpan[] sSpans = spannable.getSpans(0, spannable.length(), AniwaysSuggestionSpan.class);
		if (sSpans == null || sSpans.length == 0){
			return;
		}
		for (AniwaysSuggestionSpan span : sSpans){
			spannable.removeSpan(span);
		}
	}

	// TODO: change to Remove all Aniways encodings, and also remove the link
	static void removeAllAniwaysSpans(Spannable spannable) {
		IAniwaysWordMarkerSpan[] bcSpans = spannable.getSpans(0, spannable.length(), IAniwaysWordMarkerSpan.class);
		if (bcSpans != null && bcSpans.length != 0){
			for (IAniwaysWordMarkerSpan span : bcSpans){
				spannable.removeSpan(span);
			}
		}


		AniwaysSuggestionSpan[] sSpans = spannable.getSpans(0, spannable.length(), AniwaysSuggestionSpan.class);
		if (sSpans != null && sSpans.length != 0){
			for (AniwaysSuggestionSpan span : sSpans){
				spannable.removeSpan(span);
			}
		}


		IAniwaysImageSpan[] iSpans = spannable.getSpans(0, spannable.length(), IAniwaysImageSpan.class);
		if (iSpans != null && iSpans.length != 0){
			for (IAniwaysImageSpan span : iSpans){
				spannable.removeSpan(span);
			}
		}
	}

    //TODO: Need to refactor this method to make more clear, efficient and to get a real icon data with the phrase from the parser.
    // to this end, it needs to be more similar to the 'decode' method, and use the editable and deletions
    //TODO: While working on this, I thought of a possible bug in the decoder method (we do not always check the string converted to number that it is legal 1st thing after creation
    public static String getAniwaysStickerUrl(CharSequence text) {
        long startTime = System.currentTimeMillis();
        AniwaysStatics.makeSureAniwaysIsInitialized(false);
        IconData icon = null;

        if(TextUtils.isEmpty(text)){
            return null;
        }
        int offset = 0;
        int length = text.length();
        // Remove all leading spaces
        while(offset < length){
            if(text.charAt(offset) == ' '){
                offset++;
            }
            else{
                break;
            }
        }

        String string = text.subSequence(offset, length).toString();
        if(TextUtils.isEmpty(string)){
            return null;
        }

        int numIconsFound = 0;
        while (true){

            int iconIdSequenceLocation = indexOfAny(string, AniwaysPrivateConfig.getInstance().decoderCodepointStrings);
            if (iconIdSequenceLocation != 0){
                //Check if we have emoji
                if(numIconsFound != 0){
                    return null;
                }
                int skip = 0;
                int end = string.length() >= 6 ? 6 : (string.length() >= 4 ? 4 : (string.length() >= 2 ? 2 : 0));
                JsonParser parser = AniwaysPhraseReplacementData.getDataParser();
                for (int i = 0; i < end; i += skip) {
                    skip = 0;
                    if (icon == null) {
                        int unicode = Character.codePointAt(text, i);
                        skip = Character.charCount(unicode);

                        if (unicode > 0xff) {
                            icon = parser.getEmoji(unicode);
                        }

                        boolean secondUnicodeExists = false;
                        int followUnicode = -1;
                        if (icon == null && i + skip < end) {
                            followUnicode = Character.codePointAt(text, i + skip);
                            icon = parser.getEmoji(new int[] { unicode, followUnicode });
                            if(icon != null){
                                secondUnicodeExists = true;
                                skip += Character.charCount(followUnicode);
                            }
                        }

                        // Add the emoji varient selector if needed
                        if (icon != null && i + skip < end) {
                            int nextUnicode = Character.codePointAt(text, i + skip);
                            if(nextUnicode == EmojiWithVarientSelector.EMOJI_VARIENT_SELECTOR){
                                IconData oldIcon = icon;
                                if(secondUnicodeExists){
                                    icon = parser.getEmojiWithVarientSelector(new int[] { unicode, followUnicode });
                                }
                                else{
                                    icon = parser.getEmojiWithVarientSelector(unicode);
                                }
                                if(icon == null){
                                    icon = new EmojiWithVarientSelector(oldIcon);
                                    parser.addEmojiWithVarientSelector(icon);
                                }
                                skip += Character.charCount(nextUnicode);
                            }
                        }
                        if(icon != null){
                            break;
                        }
                    }
                }
                if(icon == null) {
                    return null;
                }
                else{
                    numIconsFound++;
                    string = string.substring(skip);
                    string = string.trim();
                    if(TextUtils.isEmpty(string)){
                        break;
                    }
                    else{
                        continue;
                    }
                }
            }

            String lengthBitString = string.substring(iconIdSequenceLocation, iconIdSequenceLocation + 1);
            int lengthBit = parseUnicodeCharsToInt(null, lengthBitString) + 1;
            int sequenceLength = AniwaysPrivateConfig.getInstance().decoderChunkSize * lengthBit;

            int encodeSequenceLocationEnd = iconIdSequenceLocation + sequenceLength;
            if (string.length() < encodeSequenceLocationEnd){
                Log.w(true, TAG, "Received string with sequence end out of the string: " + string);
                return null;
            }

            String encodingString = string.substring(iconIdSequenceLocation + 1, encodeSequenceLocationEnd);
            int iconId = parseUnicodeCharsToInt(null, encodingString);
            if (iconId < 0){
                Log.w(true, TAG, "Length < 0: " + string);
                return null;
            }

            // TODO: add more content types here
            if (iconId == AniwaysPrivateConfig.getInstance().animatedGifImageId){
                return null;
            }

            // TODO: replace iconid with path

            int startPhraseIndex = iconIdSequenceLocation;

            // Get the start index of the subphrase
            string = string.substring(encodeSequenceLocationEnd);
            //The deletes are ordered, the indexes, are post the following deletes.
            // Need this mechanism because of a bug in several android versions where editable.replace() removes the spans in some cases,
            // so can't replace the contents of an editable text with the message in decoder result..
            String delimiter = AniwaysPrivateConfig.getInstance().delimiterString;
            int startSubphraseIndex = string.indexOf(delimiter);
            if(startSubphraseIndex == -1){
                Log.w(true, TAG, "Could not find startSubphrase: " + string);
                return null;
            }

            // Get the end index of the subphrase
            string = string.substring(startSubphraseIndex+1);

            int endSubphraseIndex = string.indexOf(delimiter);
            if(endSubphraseIndex == -1){
                Log.w(true, TAG, "Could not find endSubphrase: " + string);
                return null;
            }

            if(iconId == AniwaysPrivateConfig.getInstance().viralLinkImageId){
                string = string.substring(endSubphraseIndex);
                endSubphraseIndex = 0;
            }

            // Get the end index of the phrase
            string = string.substring(endSubphraseIndex+1);
            int endPhraseIndex = string.indexOf(delimiter);
            if(endPhraseIndex == -1){
                Log.w(true, TAG, "Could not find endPhrase: " + string);
                return null;
            }
            string = string.substring(endPhraseIndex+1);

            if(iconId == AniwaysPrivateConfig.getInstance().viralLinkImageId){
                string = string.trim();
                if(TextUtils.isEmpty(string) && numIconsFound == 1){
                    break;
                }
                continue;
            }

            if(numIconsFound > 0){
                return null;
            }

            // Get the replacement text
            //String replacementText = message.subSequence(startSubphraseIndex, endSubphraseIndex).toString();

            // Get the phrase name
            //String phraseName = message.subSequence(startPhraseIndex, endPhraseIndex).toString();

            // Get the icon name from the id
            icon = AniwaysPhraseReplacementData.getDataParser().getIconById(iconId);
            if(icon == null){
                Log.v(TAG, "Could not find icon and phrase definitions for icon id: " + iconId + ". So, creating new ones. Message: " + string);
                // TODO: handle if it is an emoji
                //TODO: we assume the asset type and provider here, which is not necessarily correct!! also whether animated..
                icon = new IconData(iconId, "Unknown", false, null, null, AssetType.Emoticons, IAniwaysImageSpan.AssetProvider.Aniways, false);
                icon.primaryPhrase = new Phrase("unlnown", icon);
            }
            if(icon.assetType != AssetType.Emoticons && icon.assetType != AssetType.Emoji){
                return null;
            }
            numIconsFound++;
            string = string.trim();
            if(TextUtils.isEmpty(string)){
                break;
            }
        }

        // Report timing
        if (AnalyticsReporter.isInitialized()){
            AnalyticsReporter.ReportTiming(Verbosity.Verbose, startTime, "Is Message Aniways Sticker", "Is Message Aniways Sticker", String.valueOf(length), TAG, "num chars");
        }

        return AniwaysPrivateConfig.getInstance().getIconUrl(icon,true, false);
    }
}
