package com.aniways;

import java.util.HashSet;

import android.os.Handler;
import android.os.Looper;
import android.text.Spannable;
import android.text.TextUtils;
import android.view.View;

import com.aniways.IAniwaysDynamicImageSpan.InvalidateListener;
import com.aniways.analytics.NonThrowingRunnable;
import com.aniways.volley.toolbox.IResponseListener;

public class AniwaysDynamicImageSpansContainer {
	private static final String TAG = "AniwaysLoadingImageSpansContainer";
	private HashSet<IAniwaysDynamicImageSpan> mSpans;
	IAniwaysTextContainer mTextContainer;
	InvalidateListener mInvalidateListener;

	public AniwaysDynamicImageSpansContainer(IAniwaysTextContainer textContainer){
		mSpans = new HashSet<IAniwaysDynamicImageSpan>();
		mTextContainer = textContainer;
		mInvalidateListener = new InvalidateListener(){
			
			// This is used to make sure that we are not removing and inserting more than one span in each draw cycle since
			// it takes resources and it is not needed (removing and inserting one will cause all the rest to redraw..)
			boolean invalidateCalledInThisDrawCycle = false;
			
			@Override
			public void invalidate(IAniwaysDynamicImageSpan span) {
				if(invalidateCalledInThisDrawCycle){
					return;
				}
				
				//Log.d(TAG, "Invalidating..");
				Spannable spannable = mTextContainer.getText();
                if(spannable == null){
                    Log.w(true, TAG, "received null spannable from text container");
                    invalidateCalledInThisDrawCycle = false;
                    return;
                }
				int start = spannable.getSpanStart(span);
				int end = spannable.getSpanEnd(span);
				if(start >= 0 && end > 0){
					//Log.d(TAG, "Resetting span..");
					spannable.removeSpan(span);
					spannable.setSpan(span, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
					invalidateCalledInThisDrawCycle = true;
					mTextContainer.getView().invalidate();
				}
				else{
					// We did not actually invalidate..
					invalidateCalledInThisDrawCycle = false;
				}
			}
			
			@Override
			public void onDraw(){
				invalidateCalledInThisDrawCycle = false;
			}
			
		};
	}

	public void addDynamicImageSpan(final IAniwaysDynamicImageSpan span){
		mSpans.add(span);
		if(span.isLoadingImageSpan()){
			final AniwaysLoadingImageSpan loadingImageSpan = ((AniwaysLoadingImageSpan)span);
			loadingImageSpan.registerResponseListener(this, new IResponseListener(){
				
				@Override
				public void onError() {
					removeDynamicImagespan(span, false, false, false);
					mTextContainer.onErrorLoadingImage();
				}
				
				@Override
				public void onSuccess() {
					// Make sure that the span is still in the text that's hooked to this IAniwaysTextContainer
					// Replace this span with an image span
					Spannable spannable = mTextContainer.getText();
					boolean replaced = loadingImageSpan.replaceWithImageSpan(spannable, mTextContainer.getView().getContext(), mTextContainer.getIconInfoDisplayer(), AniwaysDynamicImageSpansContainer.this);
					if(!replaced){
						if(mTextContainer instanceof AniwaysEditTextAndTextViewCommonPart){
							View v = ((AniwaysEditTextAndTextViewCommonPart) mTextContainer).getView();
							if (v instanceof AniwaysEditText){
								Log.v(TAG, "Span not replaced in EditText, probably because text it covered was deleted");
							}
							else{
								Log.e(true, TAG, "Span not replaced. Text container class is: " + v.getClass().getSimpleName());
							}
						}
						else{
							Log.e(true, TAG, "Span not replaced. Text container class is: " + mTextContainer.getClass().getSimpleName());
						}
					}
					Log.d(TAG, "detaching from span after replacing with image. Success: " + replaced);
					removeDynamicImagespan(span, replaced, !replaced, false);
					mTextContainer.onLoadedImageSuccessfuly();
				}
			});
		}
		else if(span instanceof AniwaysAnimatedGifSpan){
			((AniwaysAnimatedGifSpan) span).registerInvalidateListener(this, mInvalidateListener);
		}
		span.onAddedToContainer(AniwaysDynamicImageSpansContainer.this);
	}

	private void removeDynamicImagespan(final IAniwaysDynamicImageSpan span, final boolean success, final boolean error, boolean sync){
		mSpans.remove(span);
		if(sync){
			removeDynamicImageSpanInner(span, success, error);
		}
		
		Handler handler = new Handler(Looper.getMainLooper());
		handler.post(new NonThrowingRunnable(TAG, "removeLoadingImagespan. Success: " + success + ". Error: " + error, "") {
			@Override
			public void innerRun() {
				removeDynamicImageSpanInner(span, success, error);
			}
		});
	}

	private void removeDynamicImageSpanInner(IAniwaysDynamicImageSpan span, boolean success, boolean error) {
		span.onRemovedFromContainer(AniwaysDynamicImageSpansContainer.this);
	}

	public void onDetachFromWindowCalled(){
		for(IAniwaysDynamicImageSpan span : mSpans){
			span.onDetachedFromWindowCalled();
		}
	}

	public void onLayoutCalled(){
		for(IAniwaysDynamicImageSpan span : mSpans){
			span.onLayoutCalled();
		}
	}

	/**
	 * This MUST be called AFTER the aniways message is decoded
	 * @param text
	 */
	public void onSetText(Spannable text, Spannable oldText){
		// Remove linkage from all spans of old text
		if(!TextUtils.isEmpty(oldText) && !oldText.equals(text)){
			IAniwaysDynamicImageSpan[] spans = oldText.getSpans(0, oldText.length(), IAniwaysDynamicImageSpan.class);
			if(spans != null){
				for(IAniwaysDynamicImageSpan span : spans){
					removeDynamicImagespan(span, false, false, true);
				}
			}
		}

		if(TextUtils.isEmpty(text)){
			return;
		}

		// Add any span that is not yet added..
		IAniwaysDynamicImageSpan[] spans = text.getSpans(0, text.length(), IAniwaysDynamicImageSpan.class);
		if(spans != null){
			for(IAniwaysDynamicImageSpan span : spans){
				addDynamicImageSpan(span);
			}
		}
	}

	public void onRemoveSpan(IAniwaysDynamicImageSpan span) {
		removeDynamicImagespan(span, false, false, true);
	}
}
