package com.aniways;

import java.util.HashMap;

import android.annotation.SuppressLint;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.drawable.Drawable;
import android.os.Handler;
import android.os.Looper;

import com.aniways.data.Phrase;
import pl.droidsonroids.gif.GifDrawable;

public class AniwaysAnimatedGifSpan extends AniwaysImageSpan implements IAniwaysDynamicImageSpan, Drawable.Callback{
	private static final String TAG = "AniwaysAnimatedImageSpan";
	private HashMap<AniwaysDynamicImageSpansContainer,InvalidateListener> mInvalidateListeners = new HashMap<AniwaysDynamicImageSpansContainer,InvalidateListener>();
	private Handler mHandler;

	public AniwaysAnimatedGifSpan(byte[] data, Phrase phrase, IconData icon, IconSelectionOrigin selectionOrigin, int maxWidth, int maxHeight) {

		super(IconData.generateAnimatedDrawable(data), phrase, icon, selectionOrigin, maxWidth, maxHeight);
        mHandler = new Handler(Looper.getMainLooper());
        Drawable drawable = this.getDrawable();
		if(drawable != null){
			drawable.setCallback(this);
		}
	}

	@SuppressLint("WrongCall") private void startAnimation() {
		Drawable drawable = this.getDrawable();
		if(drawable == null){
			Log.w(true, TAG, "Trying to start Animation, but drawable is null");
			return;
		}
		if(drawable instanceof GifDrawable){
			GifDrawable gifDrawable = (GifDrawable)drawable;
			if(!gifDrawable.isRunning()){
				for(InvalidateListener listener : mInvalidateListeners.values()){
					// Reset the mechanism which keeps invalidate calls from occurring on the same draw cycle
					listener.onDraw();
				}
				gifDrawable.start();
			}
		}
		else{
			Log.w(true, TAG, "Animation drawable is not GifDrawable");
		}
	}

	private void stopAnimation() {
		Drawable drawable = this.getDrawable();
		if(drawable == null){
			Log.w(true, TAG, "Trying to stop Animation, but drawable is null");
			return;
		}
		if(drawable instanceof GifDrawable){
			GifDrawable gifDrawable = (GifDrawable)drawable;
			if(gifDrawable.isRunning()){
				gifDrawable.stop();
			}
		}
		else{
			Log.w(true, TAG, "Animation drawable is not GifDrawable");
		}
	}

	public void registerInvalidateListener(AniwaysDynamicImageSpansContainer key, InvalidateListener invalidateListener){
		Log.d(TAG, "Registering invalidate listener");
		mInvalidateListeners.put(key, invalidateListener);
	}

	public void unregisterInvalidateListener(AniwaysDynamicImageSpansContainer key){
		Log.d(TAG, "Un-Registering invalidate listener. Before remove: " + mInvalidateListeners.size());
		InvalidateListener removed = mInvalidateListeners.remove(key);
		Log.d(TAG, "After remove: " + mInvalidateListeners.size() + ". Removed: " + (removed != null));
	}

	//
	// IAniwaysDynamicImageSpan methods
	//

	@Override
	public void onDetachedFromWindowCalled() {
		// Stop Animation if not attached to any window
		//TODO: think about it, maybe we can recycle here in some cases
		if(mInvalidateListeners.size() > 1){
			// The span is hooked to more than 1 container, so can't stop the animation
			return;
		}
		stopAnimation();
	}

	@Override
	public void onLayoutCalled() {
		startAnimation();
	}

	//
	// Drawable.Callback methods
	//

	@Override
	public void invalidateDrawable(Drawable who) {
		//Log.d(TAG, "invalidateDrawable");
		for(InvalidateListener listener : mInvalidateListeners.values()){
			//Log.d(TAG, "Calling listener");
			listener.invalidate(this);
		}
	}

	@Override
	public void scheduleDrawable(Drawable who, Runnable what, long when) {
		//Log.d(TAG, "scheduleDrawable");
		mHandler.postAtTime(what, who, when);
	}

	@Override
	public void unscheduleDrawable(Drawable who, Runnable what) {
		//Log.d(TAG, "unscheduleDrawable");
		mHandler.removeCallbacks(what, who);
	}

	@Override
	public void onAddedToContainer(AniwaysDynamicImageSpansContainer container) {
		this.startAnimation();
		
	}

	@Override
	public void onRemovedFromContainer(AniwaysDynamicImageSpansContainer aniwaysDynamicImageSpansContainer) {
		this.unregisterInvalidateListener(aniwaysDynamicImageSpansContainer);
		if(mInvalidateListeners.size() < 1){
			// The span is not hooked to any container, so stop the Animation etc.
			this.onDetachedFromWindowCalled();
		}
	}
	
	@SuppressLint("WrongCall") @Override
    public void draw(Canvas canvas, CharSequence text,
                     int start, int end, float x, 
                     int top, int y, int bottom, Paint paint) {
		
		// We call super.draw() and then we report that there was a draw cycle
        super.draw(canvas, text, start, end, x, top, y, bottom, paint);
        for(InvalidateListener listener : mInvalidateListeners.values()){
			//Log.d(TAG, "Calling listener");
			listener.onDraw();
		}
    }
}