/*
 * Copyright 2010-2015 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 * 
 *  http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
package com.amazonaws.services.config.model;

import java.io.Serializable;

import com.amazonaws.AmazonWebServiceRequest;

/**
 * Container for the parameters to the {@link com.amazonaws.services.config.AmazonConfig#putConfigRule(PutConfigRuleRequest) PutConfigRule operation}.
 * <p>
 * Adds or updates an AWS Config rule for evaluating whether your AWS
 * resources comply with your desired configurations.
 * </p>
 * <p>
 * You can use this action for customer managed Config rules and AWS
 * managed Config rules. A customer managed Config rule is a custom rule
 * that you develop and maintain. An AWS managed Config rule is a
 * customizable, predefined rule that is provided by AWS Config.
 * </p>
 * <p>
 * If you are adding a new customer managed Config rule, you must first
 * create the AWS Lambda function that the rule invokes to evaluate your
 * resources. When you use the <code>PutConfigRule</code> action to add
 * the rule to AWS Config, you must specify the Amazon Resource Name
 * (ARN) that AWS Lambda assigns to the function. Specify the ARN for the
 * <code>SourceIdentifier</code> key. This key is part of the
 * <code>Source</code> object, which is part of the
 * <code>ConfigRule</code> object.
 * </p>
 * <p>
 * If you are adding a new AWS managed Config rule, specify the rule's
 * identifier for the <code>SourceIdentifier</code> key. To reference AWS
 * managed Config rule identifiers, see
 * <a href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config_use-managed-rules.html"> Using AWS Managed Config Rules </a>
 * .
 * </p>
 * <p>
 * For any new rule that you add, specify the <code>ConfigRuleName</code>
 * in the <code>ConfigRule</code> object. Do not specify the
 * <code>ConfigRuleArn</code> or the <code>ConfigRuleId</code> . These
 * values are generated by AWS Config for new rules.
 * </p>
 * <p>
 * If you are updating a rule that you have added previously, specify the
 * rule's <code>ConfigRuleName</code> ,
 * <code>ConfigRuleId</code> ,
 * or <code>ConfigRuleArn</code> in the <code>ConfigRule</code> data
 * type that you use in this request.
 * </p>
 * <p>
 * The maximum number of rules that AWS Config supports is 25.
 * </p>
 * <p>
 * For more information about developing and using AWS Config rules, see
 * <a href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html"> Evaluating AWS Resource Configurations with AWS Config </a>
 * in the <i>AWS Config Developer Guide</i> .
 * </p>
 * <p>
 * </p>
 *
 * @see com.amazonaws.services.config.AmazonConfig#putConfigRule(PutConfigRuleRequest)
 */
public class PutConfigRuleRequest extends AmazonWebServiceRequest implements Serializable, Cloneable {

    /**
     * An AWS Lambda function that evaluates configuration items to assess
     * whether your AWS resources comply with your desired configurations.
     * This function can run when AWS Config detects a configuration change
     * or delivers a configuration snapshot. This function can evaluate any
     * resource in the recording group. To define which of these are
     * evaluated, specify a value for the <code>Scope</code> key. <p>For more
     * information about developing and using AWS Config rules, see <a
     * href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     * AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     * Developer Guide</i>.
     */
    private ConfigRule configRule;

    /**
     * An AWS Lambda function that evaluates configuration items to assess
     * whether your AWS resources comply with your desired configurations.
     * This function can run when AWS Config detects a configuration change
     * or delivers a configuration snapshot. This function can evaluate any
     * resource in the recording group. To define which of these are
     * evaluated, specify a value for the <code>Scope</code> key. <p>For more
     * information about developing and using AWS Config rules, see <a
     * href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     * AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     * Developer Guide</i>.
     *
     * @return An AWS Lambda function that evaluates configuration items to assess
     *         whether your AWS resources comply with your desired configurations.
     *         This function can run when AWS Config detects a configuration change
     *         or delivers a configuration snapshot. This function can evaluate any
     *         resource in the recording group. To define which of these are
     *         evaluated, specify a value for the <code>Scope</code> key. <p>For more
     *         information about developing and using AWS Config rules, see <a
     *         href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     *         AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     *         Developer Guide</i>.
     */
    public ConfigRule getConfigRule() {
        return configRule;
    }
    
    /**
     * An AWS Lambda function that evaluates configuration items to assess
     * whether your AWS resources comply with your desired configurations.
     * This function can run when AWS Config detects a configuration change
     * or delivers a configuration snapshot. This function can evaluate any
     * resource in the recording group. To define which of these are
     * evaluated, specify a value for the <code>Scope</code> key. <p>For more
     * information about developing and using AWS Config rules, see <a
     * href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     * AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     * Developer Guide</i>.
     *
     * @param configRule An AWS Lambda function that evaluates configuration items to assess
     *         whether your AWS resources comply with your desired configurations.
     *         This function can run when AWS Config detects a configuration change
     *         or delivers a configuration snapshot. This function can evaluate any
     *         resource in the recording group. To define which of these are
     *         evaluated, specify a value for the <code>Scope</code> key. <p>For more
     *         information about developing and using AWS Config rules, see <a
     *         href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     *         AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     *         Developer Guide</i>.
     */
    public void setConfigRule(ConfigRule configRule) {
        this.configRule = configRule;
    }
    
    /**
     * An AWS Lambda function that evaluates configuration items to assess
     * whether your AWS resources comply with your desired configurations.
     * This function can run when AWS Config detects a configuration change
     * or delivers a configuration snapshot. This function can evaluate any
     * resource in the recording group. To define which of these are
     * evaluated, specify a value for the <code>Scope</code> key. <p>For more
     * information about developing and using AWS Config rules, see <a
     * href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     * AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     * Developer Guide</i>.
     * <p>
     * Returns a reference to this object so that method calls can be chained together.
     *
     * @param configRule An AWS Lambda function that evaluates configuration items to assess
     *         whether your AWS resources comply with your desired configurations.
     *         This function can run when AWS Config detects a configuration change
     *         or delivers a configuration snapshot. This function can evaluate any
     *         resource in the recording group. To define which of these are
     *         evaluated, specify a value for the <code>Scope</code> key. <p>For more
     *         information about developing and using AWS Config rules, see <a
     *         href="http://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating
     *         AWS Resource Configurations with AWS Config</a> in the <i>AWS Config
     *         Developer Guide</i>.
     *
     * @return A reference to this updated object so that method calls can be chained
     *         together.
     */
    public PutConfigRuleRequest withConfigRule(ConfigRule configRule) {
        this.configRule = configRule;
        return this;
    }

    /**
     * Returns a string representation of this object; useful for testing and
     * debugging.
     *
     * @return A string representation of this object.
     *
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        if (getConfigRule() != null) sb.append("ConfigRule: " + getConfigRule() );
        sb.append("}");
        return sb.toString();
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int hashCode = 1;
        
        hashCode = prime * hashCode + ((getConfigRule() == null) ? 0 : getConfigRule().hashCode()); 
        return hashCode;
    }
    
    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj == null) return false;

        if (obj instanceof PutConfigRuleRequest == false) return false;
        PutConfigRuleRequest other = (PutConfigRuleRequest)obj;
        
        if (other.getConfigRule() == null ^ this.getConfigRule() == null) return false;
        if (other.getConfigRule() != null && other.getConfigRule().equals(this.getConfigRule()) == false) return false; 
        return true;
    }
    
    @Override
    public PutConfigRuleRequest clone() {
        
            return (PutConfigRuleRequest) super.clone();
    }

}
    