/*
 * Decompiled with CFR 0.152.
 */
package com.amazonaws.services.kinesis.leases.impl;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.dynamodbv2.AmazonDynamoDB;
import com.amazonaws.services.dynamodbv2.model.AttributeValueUpdate;
import com.amazonaws.services.dynamodbv2.model.ConditionalCheckFailedException;
import com.amazonaws.services.dynamodbv2.model.CreateTableRequest;
import com.amazonaws.services.dynamodbv2.model.DeleteItemRequest;
import com.amazonaws.services.dynamodbv2.model.DescribeTableRequest;
import com.amazonaws.services.dynamodbv2.model.DescribeTableResult;
import com.amazonaws.services.dynamodbv2.model.GetItemRequest;
import com.amazonaws.services.dynamodbv2.model.GetItemResult;
import com.amazonaws.services.dynamodbv2.model.LimitExceededException;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughput;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughputExceededException;
import com.amazonaws.services.dynamodbv2.model.PutItemRequest;
import com.amazonaws.services.dynamodbv2.model.ResourceInUseException;
import com.amazonaws.services.dynamodbv2.model.ResourceNotFoundException;
import com.amazonaws.services.dynamodbv2.model.ScanRequest;
import com.amazonaws.services.dynamodbv2.model.ScanResult;
import com.amazonaws.services.dynamodbv2.model.TableStatus;
import com.amazonaws.services.dynamodbv2.model.UpdateItemRequest;
import com.amazonaws.services.kinesis.leases.exceptions.DependencyException;
import com.amazonaws.services.kinesis.leases.exceptions.InvalidStateException;
import com.amazonaws.services.kinesis.leases.exceptions.ProvisionedThroughputException;
import com.amazonaws.services.kinesis.leases.impl.Lease;
import com.amazonaws.services.kinesis.leases.interfaces.ILeaseManager;
import com.amazonaws.services.kinesis.leases.interfaces.ILeaseSerializer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class LeaseManager<T extends Lease>
implements ILeaseManager<T> {
    private static final Log LOG = LogFactory.getLog(LeaseManager.class);
    protected String table;
    protected AmazonDynamoDB dynamoDBClient;
    protected ILeaseSerializer<T> serializer;
    protected boolean consistentReads;

    public LeaseManager(String table, AmazonDynamoDB dynamoDBClient, ILeaseSerializer<T> serializer) {
        this(table, dynamoDBClient, serializer, false);
    }

    public LeaseManager(String table, AmazonDynamoDB dynamoDBClient, ILeaseSerializer<T> serializer, boolean consistentReads) {
        this.verifyNotNull(table, "Table name cannot be null");
        this.verifyNotNull(dynamoDBClient, "dynamoDBClient cannot be null");
        this.verifyNotNull(serializer, "ILeaseSerializer cannot be null");
        this.table = table;
        this.dynamoDBClient = dynamoDBClient;
        this.consistentReads = consistentReads;
        this.serializer = serializer;
    }

    @Override
    public boolean createLeaseTableIfNotExists(Long readCapacity, Long writeCapacity) throws ProvisionedThroughputException, DependencyException {
        this.verifyNotNull(readCapacity, "readCapacity cannot be null");
        this.verifyNotNull(writeCapacity, "writeCapacity cannot be null");
        boolean tableDidNotExist = true;
        CreateTableRequest request = new CreateTableRequest();
        request.setTableName(this.table);
        request.setKeySchema(this.serializer.getKeySchema());
        request.setAttributeDefinitions(this.serializer.getAttributeDefinitions());
        ProvisionedThroughput throughput = new ProvisionedThroughput();
        throughput.setReadCapacityUnits(readCapacity);
        throughput.setWriteCapacityUnits(writeCapacity);
        request.setProvisionedThroughput(throughput);
        try {
            this.dynamoDBClient.createTable(request);
        }
        catch (ResourceInUseException e) {
            tableDidNotExist = false;
            LOG.info((Object)("Table " + this.table + " already exists."));
        }
        catch (LimitExceededException e) {
            throw new ProvisionedThroughputException("Capacity exceeded when creating table " + this.table, e);
        }
        catch (AmazonClientException e) {
            throw new DependencyException(e);
        }
        return tableDidNotExist;
    }

    @Override
    public boolean leaseTableExists() throws DependencyException {
        DescribeTableResult result;
        DescribeTableRequest request = new DescribeTableRequest();
        request.setTableName(this.table);
        try {
            result = this.dynamoDBClient.describeTable(request);
        }
        catch (ResourceNotFoundException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)String.format("Got ResourceNotFoundException for table %s in leaseTableExists, returning false.", this.table));
            }
            return false;
        }
        catch (AmazonClientException e) {
            throw new DependencyException(e);
        }
        String tableStatus = result.getTable().getTableStatus();
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Lease table exists and is in status " + tableStatus));
        }
        return TableStatus.ACTIVE.name().equals(tableStatus);
    }

    @Override
    public boolean waitUntilLeaseTableExists(long secondsBetweenPolls, long timeoutSeconds) throws DependencyException {
        long sleepTimeRemaining = TimeUnit.SECONDS.toMillis(timeoutSeconds);
        while (!this.leaseTableExists()) {
            if (sleepTimeRemaining <= 0L) {
                return false;
            }
            long timeToSleepMillis = Math.min(TimeUnit.SECONDS.toMillis(secondsBetweenPolls), sleepTimeRemaining);
            sleepTimeRemaining -= this.sleep(timeToSleepMillis);
        }
        return true;
    }

    long sleep(long timeToSleepMillis) {
        long startTime = System.currentTimeMillis();
        try {
            Thread.sleep(timeToSleepMillis);
        }
        catch (InterruptedException e) {
            LOG.debug((Object)"Interrupted while sleeping");
        }
        return System.currentTimeMillis() - startTime;
    }

    @Override
    public List<T> listLeases() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        return this.list(null);
    }

    List<T> list(Integer limit) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Listing leases from table " + this.table));
        }
        ScanRequest scanRequest = new ScanRequest();
        scanRequest.setTableName(this.table);
        if (limit != null) {
            scanRequest.setLimit(limit);
        }
        try {
            ScanResult scanResult = this.dynamoDBClient.scan(scanRequest);
            ArrayList<T> result = new ArrayList<T>();
            while (scanResult != null) {
                for (Map item : scanResult.getItems()) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug((Object)("Got item " + item.toString() + " from DynamoDB."));
                    }
                    result.add(this.serializer.fromDynamoRecord(item));
                }
                Map lastEvaluatedKey = scanResult.getLastEvaluatedKey();
                if (lastEvaluatedKey == null) {
                    scanResult = null;
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug((Object)"lastEvaluatedKey was null - scan finished.");
                    continue;
                }
                scanRequest.setExclusiveStartKey(lastEvaluatedKey);
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("lastEvaluatedKey was " + lastEvaluatedKey + ", continuing scan."));
                }
                scanResult = this.dynamoDBClient.scan(scanRequest);
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Listed " + result.size() + " leases from table " + this.table));
            }
            return result;
        }
        catch (ResourceNotFoundException e) {
            throw new InvalidStateException("Cannot scan lease table " + this.table + " because it does not exist.", e);
        }
        catch (ProvisionedThroughputExceededException e) {
            throw new ProvisionedThroughputException(e);
        }
        catch (AmazonClientException e) {
            throw new DependencyException(e);
        }
    }

    @Override
    public boolean createLeaseIfNotExists(T lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Creating lease " + lease));
        }
        PutItemRequest request = new PutItemRequest();
        request.setTableName(this.table);
        request.setItem(this.serializer.toDynamoRecord(lease));
        request.setExpected(this.serializer.getDynamoNonexistantExpectation());
        try {
            this.dynamoDBClient.putItem(request);
        }
        catch (ConditionalCheckFailedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Did not create lease " + lease + " because it already existed"));
            }
            return false;
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("create", ((Lease)lease).getLeaseKey(), e);
        }
        return true;
    }

    @Override
    public T getLease(String leaseKey) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(leaseKey, "leaseKey cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Getting lease with key " + leaseKey));
        }
        GetItemRequest request = new GetItemRequest();
        request.setTableName(this.table);
        request.setKey(this.serializer.getDynamoHashKey(leaseKey));
        request.setConsistentRead(Boolean.valueOf(this.consistentReads));
        try {
            GetItemResult result = this.dynamoDBClient.getItem(request);
            Map dynamoRecord = result.getItem();
            if (dynamoRecord == null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("No lease found with key " + leaseKey + ", returning null."));
                }
                return null;
            }
            T lease = this.serializer.fromDynamoRecord(dynamoRecord);
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Got lease " + lease));
            }
            return lease;
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("get", leaseKey, e);
        }
    }

    @Override
    public boolean renewLease(T lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Renewing lease with key " + ((Lease)lease).getLeaseKey()));
        }
        UpdateItemRequest request = new UpdateItemRequest();
        request.setTableName(this.table);
        request.setKey(this.serializer.getDynamoHashKey(lease));
        request.setExpected(this.serializer.getDynamoLeaseCounterExpectation(lease));
        request.setAttributeUpdates(this.serializer.getDynamoLeaseCounterUpdate(lease));
        try {
            this.dynamoDBClient.updateItem(request);
        }
        catch (ConditionalCheckFailedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Lease renewal failed for lease with key " + ((Lease)lease).getLeaseKey() + " because the lease counter was not " + ((Lease)lease).getLeaseCounter()));
            }
            return false;
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("renew", ((Lease)lease).getLeaseKey(), e);
        }
        ((Lease)lease).setLeaseCounter(((Lease)lease).getLeaseCounter() + 1L);
        return true;
    }

    @Override
    public boolean takeLease(T lease, String owner) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        this.verifyNotNull(owner, "owner cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)String.format("Taking lease with leaseKey %s from %s to %s", ((Lease)lease).getLeaseKey(), ((Lease)lease).getLeaseOwner() == null ? "nobody" : ((Lease)lease).getLeaseOwner(), owner));
        }
        UpdateItemRequest request = new UpdateItemRequest();
        request.setTableName(this.table);
        request.setKey(this.serializer.getDynamoHashKey(lease));
        request.setExpected(this.serializer.getDynamoLeaseCounterExpectation(lease));
        Map<String, AttributeValueUpdate> updates = this.serializer.getDynamoLeaseCounterUpdate(lease);
        updates.putAll(this.serializer.getDynamoTakeLeaseUpdate(lease, owner));
        request.setAttributeUpdates(updates);
        try {
            this.dynamoDBClient.updateItem(request);
        }
        catch (ConditionalCheckFailedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Lease renewal failed for lease with key " + ((Lease)lease).getLeaseKey() + " because the lease counter was not " + ((Lease)lease).getLeaseCounter()));
            }
            return false;
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("take", ((Lease)lease).getLeaseKey(), e);
        }
        ((Lease)lease).setLeaseCounter(((Lease)lease).getLeaseCounter() + 1L);
        ((Lease)lease).setLeaseOwner(owner);
        return true;
    }

    @Override
    public boolean evictLease(T lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)String.format("Evicting lease with leaseKey %s owned by %s", ((Lease)lease).getLeaseKey(), ((Lease)lease).getLeaseOwner()));
        }
        UpdateItemRequest request = new UpdateItemRequest();
        request.setTableName(this.table);
        request.setKey(this.serializer.getDynamoHashKey(lease));
        request.setExpected(this.serializer.getDynamoLeaseOwnerExpectation(lease));
        Map<String, AttributeValueUpdate> updates = this.serializer.getDynamoLeaseCounterUpdate(lease);
        updates.putAll(this.serializer.getDynamoEvictLeaseUpdate(lease));
        request.setAttributeUpdates(updates);
        try {
            this.dynamoDBClient.updateItem(request);
        }
        catch (ConditionalCheckFailedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Lease eviction failed for lease with key " + ((Lease)lease).getLeaseKey() + " because the lease owner was not " + ((Lease)lease).getLeaseOwner()));
            }
            return false;
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("evict", ((Lease)lease).getLeaseKey(), e);
        }
        ((Lease)lease).setLeaseOwner(null);
        ((Lease)lease).setLeaseCounter(((Lease)lease).getLeaseCounter() + 1L);
        return true;
    }

    @Override
    public void deleteAll() throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        List<T> allLeases = this.listLeases();
        LOG.warn((Object)("Deleting " + allLeases.size() + " items from table " + this.table));
        for (Lease lease : allLeases) {
            DeleteItemRequest deleteRequest = new DeleteItemRequest();
            deleteRequest.setTableName(this.table);
            deleteRequest.setKey(this.serializer.getDynamoHashKey(lease));
            this.dynamoDBClient.deleteItem(deleteRequest);
        }
    }

    @Override
    public void deleteLease(T lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)String.format("Deleting lease with leaseKey %s", ((Lease)lease).getLeaseKey()));
        }
        DeleteItemRequest deleteRequest = new DeleteItemRequest();
        deleteRequest.setTableName(this.table);
        deleteRequest.setKey(this.serializer.getDynamoHashKey(lease));
        try {
            this.dynamoDBClient.deleteItem(deleteRequest);
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("delete", ((Lease)lease).getLeaseKey(), e);
        }
    }

    @Override
    public boolean updateLease(T lease) throws DependencyException, InvalidStateException, ProvisionedThroughputException {
        this.verifyNotNull(lease, "lease cannot be null");
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)String.format("Updating lease %s", lease));
        }
        UpdateItemRequest request = new UpdateItemRequest();
        request.setTableName(this.table);
        request.setKey(this.serializer.getDynamoHashKey(lease));
        request.setExpected(this.serializer.getDynamoLeaseCounterExpectation(lease));
        Map<String, AttributeValueUpdate> updates = this.serializer.getDynamoLeaseCounterUpdate(lease);
        updates.putAll(this.serializer.getDynamoUpdateLeaseUpdate(lease));
        request.setAttributeUpdates(updates);
        try {
            this.dynamoDBClient.updateItem(request);
        }
        catch (ConditionalCheckFailedException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Lease update failed for lease with key " + ((Lease)lease).getLeaseKey() + " because the lease counter was not " + ((Lease)lease).getLeaseCounter()));
            }
            return false;
        }
        catch (AmazonClientException e) {
            throw this.convertAndRethrowExceptions("update", ((Lease)lease).getLeaseKey(), e);
        }
        ((Lease)lease).setLeaseCounter(((Lease)lease).getLeaseCounter() + 1L);
        return true;
    }

    protected DependencyException convertAndRethrowExceptions(String operation, String leaseKey, AmazonClientException e) throws ProvisionedThroughputException, InvalidStateException {
        if (e instanceof ProvisionedThroughputExceededException) {
            throw new ProvisionedThroughputException(e);
        }
        if (e instanceof ResourceNotFoundException) {
            throw new InvalidStateException(String.format("Cannot %s lease with key %s because table %s does not exist.", operation, leaseKey, this.table), e);
        }
        return new DependencyException(e);
    }

    private void verifyNotNull(Object object, String message) {
        if (object == null) {
            throw new IllegalArgumentException(message);
        }
    }
}

