/*
 * Copyright (C) Jerry Huxtable 1998
 */
package com.alkacon.simapi.filter;

import java.awt.*;
import java.awt.image.*;

/**
 * A Filter which distorts an image by twisting it from the centre out.
 * The twisting is centred at the centre of the image and extends out to the smallest of
 * the width and height. Pixels outside this radius are unaffected.
 */
public class TwirlFilter extends TransformFilter {

	static final long serialVersionUID = 1550445062822803342L;
	
	private float angle = 0;
	private float centreX = 0;
	private float centreY = 0;
	private float radius = 0;
	private float radius2 = 0;

	/**
	 * Construct a TwirlFilter with no distortion.
	 */
	public TwirlFilter() {
	}

	/**
	 * Set the angle of twirl in radians. 0 means no distortion.
	 * @param angle the angle of twirl. This is the angle by which pixels at the nearest edge of the image will move.
	 */
	public void setAngle(float angle) {
		this.angle = angle;
	}
	
	/**
	 * Get the angle of twist.
	 * @return the angle in radians.
	 */
	public float getAngle() {
		return angle;
	}
	
	public void setDimensions(int width, int height) {
		super.setDimensions(width, height);
		centreX = width/2;
		centreY = height/2;
		radius = Math.min(centreX, centreY);
		radius2 = radius*radius;
	}
	
	protected void transformInverse(int x, int y, float[] out) {
		float dx = x-centreX;
		float dy = y-centreY;
		float distance = dx*dx + dy*dy;
		if (distance > radius2) {
			out[0] = x;
			out[1] = y;
		} else {
			distance = (float)Math.sqrt(distance);
			float a = (float)Math.atan2(dy, dx) + angle * (radius-distance) / radius;
			out[0] = centreX + distance*(float)Math.cos(a);
			out[1] = centreY + distance*(float)Math.sin(a);
		}
	}

	public String toString() {
		return "Distort/Twirl...";
	}

}
