/*
 * Copyright (C) Jerry Huxtable 1998
 */
package com.alkacon.simapi.filter;

import java.awt.image.*;

/**
 * A filter which applies Gaussian blur to an image. This is a subclass of ConvolveFilter
 * which simply creates a kernel with a Gaussian distribution for blurring.
 * @author Jerry Huxtable
 */
public class GaussianFilter extends ConvolveFilter {

	static final long serialVersionUID = 5377089073023183684L;

	protected int radius;
	private Kernel[] kernels;
	
	/**
	 * Construct a Gaussian filter
	 */
	public GaussianFilter() {
		this(2);
	}

	/**
	 * Construct a Gaussian filter
	 * @param radius blur radius in pixels
	 */
	public GaussianFilter(int radius) {
		setRadius(radius);
	}

	/**
	 * Set the radius of the kernel, and hence the amount of blur. The bigger the radius, the longer this filter will take.
	 * @param radius the radius of the blur in pixels.
	 */
	public void setRadius(int radius) {
		this.radius = radius;
//		setKernel(makeKernel(radius));
		kernels = separatedKernels(radius);
	}
	
	/**
	 * Get the radius of the kernel.
	 * @return the radius
	 */
	public int getRadius() {
		return radius;
	}

	public void imageComplete(int status) {
		if (status == IMAGEERROR || status == IMAGEABORTED) {
			consumer.imageComplete(status);
			return;
		}

		int width = originalSpace.width;
		int height = originalSpace.height;

		int[] outPixels = new int[width * height];
		convolveH(kernels[0], inPixels, outPixels, width, height, alpha);
		convolveV(kernels[1], outPixels, inPixels, width, height, alpha);

		consumer.setPixels(0, 0, width, height, defaultRGBModel, inPixels, 0, width);
		consumer.imageComplete(status);
		inPixels = null;
	}

	public static Kernel makeKernel(int radius) {
		int rows = radius*2+1;
		int cols = rows;
		float[] matrix = new float[rows*cols];
		float sigma = (float)radius/3;
		float sigma22 = 2*sigma*sigma;
		float sigmaPi2 = 2*ImageMath.PI*sigma;
		float radius2 = radius*radius;
		float total = 0;
		int index = 0;
		for (int row = -radius; row <= radius; row++) {
			for (int col = -radius; col <= radius; col++) {
				float distance = row*row+col*col;
				if (distance > radius2)
					matrix[index] = 0;
				else
					matrix[index] = (float)Math.exp(-(distance)/sigma22) / sigmaPi2;
				total += matrix[index];
				index++;
			}
		}
		for (int i = 0; i < rows*cols; i++)
			matrix[i] /= total;
		Kernel kernel = new Kernel(rows, cols, matrix);
		return kernel;
	}
	
	public static Kernel[] separatedKernels(int radius) {
		int rows = radius*2+1;
		float[] matrix = new float[rows];
		float sigma = (float)radius/3;
		float sigma22 = 2*sigma*sigma;
		float sigmaPi2 = 2*ImageMath.PI*sigma;
		float sqrtSigmaPi2 = (float)Math.sqrt(sigmaPi2);
		float radius2 = radius*radius;
		float total = 0;
		int index = 0;
		for (int row = -radius; row <= radius; row++) {
			float distance = row*row;
			if (distance > radius2)
				matrix[index] = 0;
			else
				matrix[index] = (float)Math.exp(-(distance)/sigma22) / sqrtSigmaPi2;
			total += matrix[index];
			index++;
		}
		for (int i = 0; i < rows; i++)
			matrix[i] /= total;

		return new Kernel[] {
			new Kernel(1, rows, matrix),
			new Kernel(rows, 1, matrix),
		};
	}

	public String toString() {
		return "Blur/Gaussian Blur...";
	}
}
