/*
 * Copyright (C) Jerry Huxtable 1998
 */
package com.alkacon.simapi.filter;

import java.awt.*;
import java.awt.image.*;

/**
 * A convolution kernel.
 * @author Jerry Huxtable
 */
public class Kernel {

	public final static float R2 = (float)Math.sqrt(2);
	
	public final static float[] ROBERTS_V = {
		 0,  0, -1,
		 0,  1,  0,
		 0,  0,  0,
	};
	public final static float[] ROBERTS_H = {
		-1,  0,  0,
		 0,  1,  0,
		 0,  0,  0,
	};
	public final static float[] PREWITT_V = {
		-1,  0,  1,
		-1,  0,  1,
		-1,  0,  1,
	};
	public final static float[] PREWITT_H = {
		-1, -1, -1,
		 0,  0,  0,
		 1,  1,  1,
	};
	public final static float[] SOBEL_V = {
		-1,  0,  1,
		-2,  0,  2,
		-1,  0,  1,
	};
	public static float[] SOBEL_H = {
		-1, -2, -1,
		 0,  0,  0,
		 1,  2,  1,
	};
	public final static float[] FREI_CHEN_V = {
		 -1,  0,  1,
		-R2,  0,  R2,
		 -1,  0,  1,
	};
	public static float[] FREI_CHEN_H = {
		-1, -R2, -1,
		 0,  0,  0,
		 1,  R2,  1,
	};
	public static float[] EDGE_LAPLACE1 = {
		 0, -1,  0,
		-1,  4, -1,
		 0, -1,  0,
	};
	public static float[] EDGE_LAPLACE2 = {
		-1, -1, -1,
		-1,  8, -1,
		-1, -1, -1,
	};

	public float[] matrix = null;
	public int rows, cols;
	public int xCentre, yCentre;

	/**
	 * Construct a null kernel. This is only useful if you're going to change the kernel later on.
	 */
	public Kernel() {
		this(new float[9]);
	}

	/**
	 * Construct a kernel with the given 3x3 matrix.
	 * @param matrix an array of 9 floats containing the kernel
	 */
	public Kernel(float[] matrix) {
		setMatrix(3, 3, 1, 1, matrix);
	}
	
	/**
	 * Construct a kernel with the given matrix.
	 * @param rows	the number of rows in the kernel
	 * @param cols	the number of columns in the kernel
	 * @param matrix	an array of rows*cols floats containing the kernel
	 */
	public Kernel(int rows, int cols, float[] matrix) {
		setMatrix(rows, cols, matrix);
	}
	
	/**
	 * Construct a kernel with the given matrix.
	 * @param rows	the number of rows in the kernel
	 * @param cols	the number of columns in the kernel
	 * @param xCentre the centre of the kernel in the x direction
	 * @param yCentre the centre of the kernel in the y direction
	 * @param matrix	an array of rows*cols floats containing the kernel
	 */
	public Kernel(int rows, int cols, int xCentre, int yCentre, float[] matrix) {
		setMatrix(rows, cols, xCentre, yCentre, matrix);
	}
	
	/**
	 * Set the (3x3) convolution kernel.
	 * @param matrix an array of 9 floats containing the kernel
	 */
	public void setMatrix(float[] matrix) {
		setMatrix(3, 3, 1, 1, matrix);
	}
	
	/**
	 * Set the convolution kernel.
	 * @param rows the number of rows in the kernel
	 * @param cols the number of columns in the kernel
	 * @param matrix an array of rows*cols floats containing the kernel
	 */
	public void setMatrix(int rows, int cols, float[] matrix) {
		setMatrix(rows, cols, rows/2, cols/2, matrix);
	}
	
	/**
	 * Set the convolution kernel.
	 * @param rows the number of rows in the kernel
	 * @param cols the number of columns in the kernel
	 * @param xCentre the centre of the kernel in the x direction
	 * @param yCentre the centre of the kernel in the y direction
	 * @param matrix an array of rows*cols floats containing the kernel
	 */
	public void setMatrix(int rows, int cols, int xCentre, int yCentre, float[] matrix) {
		this.rows = rows;
		this.cols = cols;
		this.xCentre = xCentre;
		this.yCentre = yCentre;
		this.matrix = matrix;
	}
	
	public float[] getMatrix() {
		return matrix;
	}
	
	/**
	 * Normalize a convolution kernel. You can call this to normalize a kernel you've created by hand.
	 * @param matrix the kernel
	 */
	public void normalize() {
		float total = 0;
		for (int i = 0; i < matrix.length; i++)
			total += matrix[i];
		if (total != 0)
			for (int i = 0; i < matrix.length; i++)
				matrix[i] /= total;
	}
	
}
