package com.aliyun.core.utils;

import java.lang.reflect.GenericDeclaration;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public final class TypeUtil {
    private static final Map<Type, Type> SUPER_TYPE_MAP = new ConcurrentHashMap<>();

    public static List<Class<?>> getAllClasses(Class<?> clazz) {
        List<Class<?>> types = new ArrayList<>();
        while (clazz != null) {
            types.add(clazz);
            clazz = clazz.getSuperclass();
        }
        return types;
    }

    public static Type[] getTypeArguments(Type type) {
        if (!(type instanceof ParameterizedType)) {
            return new Type[0];
        }
        return ((ParameterizedType) type).getActualTypeArguments();
    }

    public static Type getTypeArgument(Type type) {
        if (!(type instanceof ParameterizedType)) {
            return null;
        }
        return ((ParameterizedType) type).getActualTypeArguments()[0];
    }

    @SuppressWarnings("unchecked")
    public static Class<?> getRawClass(Type type) {
        if (type instanceof ParameterizedType) {
            return (Class<?>) ((ParameterizedType) type).getRawType();
        } else {
            return (Class<?>) type;
        }
    }

    public static Type getSuperType(final Type type) {
        return SUPER_TYPE_MAP.computeIfAbsent(type, _type -> {
            if (type instanceof ParameterizedType) {
                final ParameterizedType parameterizedType = (ParameterizedType) type;
                final Type genericSuperClass = ((Class<?>) parameterizedType.getRawType()).getGenericSuperclass();

                if (genericSuperClass instanceof ParameterizedType) {
                    final Type[] superTypeArguments = getTypeArguments(genericSuperClass);
                    final Type[] typeParameters =
                            ((GenericDeclaration) parameterizedType.getRawType()).getTypeParameters();
                    int k = 0;

                    for (int i = 0; i != superTypeArguments.length; i++) {
                        for (int j = 0; i < typeParameters.length; j++) {
                            if (typeParameters[j].equals(superTypeArguments[i])) {
                                superTypeArguments[i] = parameterizedType.getActualTypeArguments()[k++];
                                break;
                            }
                        }
                    }
                    return createParameterizedType(((ParameterizedType) genericSuperClass).getRawType(),
                            superTypeArguments);
                } else {
                    return genericSuperClass;
                }
            } else {
                return ((Class<?>) type).getGenericSuperclass();
            }
        });
    }

    public static Type getSuperType(Type subType, Class<?> rawSuperType) {
        while (subType != null && getRawClass(subType) != rawSuperType) {
            subType = getSuperType(subType);
        }
        return subType;
    }

    public static boolean isTypeOrSubTypeOf(Type subType, Type superType) {
        return getRawClass(superType).isAssignableFrom(getRawClass(subType));
    }

    public static ParameterizedType createParameterizedType(Type rawClass, Type... genericTypes) {
        return new ParameterizedType() {
            @Override
            public Type[] getActualTypeArguments() {
                return genericTypes;
            }

            @Override
            public Type getRawType() {
                return rawClass;
            }

            @Override
            public Type getOwnerType() {
                return null;
            }
        };
    }

    public static boolean restResponseTypeExpectsBody(ParameterizedType restResponseReturnType) {
        return getRestResponseBodyType(restResponseReturnType) != Void.class;
    }

    public static Type getRestResponseBodyType(Type restResponseReturnType) {
        final Type[] restResponseTypeArguments = TypeUtil.getTypeArguments(restResponseReturnType);
        if (restResponseTypeArguments != null && restResponseTypeArguments.length > 0) {
            return restResponseTypeArguments[restResponseTypeArguments.length - 1];
        } else {
            return getRestResponseBodyType(TypeUtil.getSuperType(restResponseReturnType));
        }
    }

    // Private Ctr
    private TypeUtil() {
    }
}
