package com.adpdigital.push;

import android.app.ActivityManager;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.net.wifi.WifiManager;
import android.os.AsyncTask;
import android.os.Binder;
import android.os.Build;
import android.os.SystemClock;
import android.util.Log;

import org.eclipse.paho.client.mqttv3.IMqttClient;
import org.eclipse.paho.client.mqttv3.IMqttDeliveryToken;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import de.greenrobot.event.EventBus;

import static com.adpdigital.push.AdpPushClient.PUSH_MSG_RECEIVED_MSG;
import static com.adpdigital.push.AdpPushClient.PUSH_MSG_RECEIVED_TOPIC;


/**
 * @author Behrad
 */
public class PushServiceManager {

    private final static String TAG = PushServiceManager.class.getName();

    private static final int MESSAGE_SIZE = 200;
    private static final short KEEP_ALIVE_DATA = 3 * 60;
    private static final short KEEP_ALIVE_WIFI = 5 * 60;
    private static final long BASE_RETRY = 2 * 1000L;
    private final static long MAX_RETRY = 60 * BASE_RETRY;
    private static final ScheduledExecutorService worker = Executors.newSingleThreadScheduledExecutor();
    private static boolean isScreenOn = true;
    private static boolean energySaverMode = false;
    static private Collection<String> dataCache = new BoundedQueue<>(MESSAGE_SIZE);
    static PushConnection connection;
    // receiver that notifies the Service when the phone gets data connection
    private NetworkConnectionIntentReceiver netConnReceiver;
    private BroadcastReceiver mScreenReceiver;
    private EventBus eventBus = EventBus.getDefault();
    private Collection<String> pendingInAppMsgs = new BoundedQueue<>(MESSAGE_SIZE * 2);
    /**
     * User's ID, this should be set from the acquired user phone number
     */
    private SecureString userId;
    /**
     * Substitute your own application ID here. This is the id of the
     * application you registered in your LoopBack server by calling
     * Application.register().
     */
    private String appId;
    private String installationId;
    private String host;
    private String port;
    private boolean useSecure;
    private String username;
    private String password;
    private WifiManager wifi;
    private AlarmManager alarm;
    private WifiManager.WifiLock wifiWakeLock;
    private long retryInterval;
    private boolean cleaningUp = false;
    private LocalBinder<PushServiceManager> mBinder;
    private Context context;


    public static PushServiceManager instance;


    public static PushServiceManager getInstance(Context ctx) {
        if (instance == null) {
            synchronized (PushServiceManager.class) {
                if (instance == null) {
                    instance = new PushServiceManager(ctx);
                }
            }
        }
        return instance;
    }


    private PushServiceManager(Context context) {
        this.context = context;
        this.retryInterval = BASE_RETRY;
        init();
        Logger.d(TAG, "Creating PushServiceManager for " + userId);
    }

    private void init() {
        cleaningUp = false;
        initPerperties();
    }

    void setupAndStart() {
        init();

        this.wifi = ((WifiManager) getApplicationContext().getSystemService(Context.WIFI_SERVICE));
        this.alarm = ((AlarmManager) getApplicationContext().getSystemService(Context.ALARM_SERVICE));

        handleCrashedService();

//        setUndhandledException(Thread.currentThread());

        Set<String> cache = getSharedPreferences().getStringSet("dataCache", null);
        if (cache != null) {
            dataCache.addAll(cache);
        }

        eventBus.register(this);

        mBinder = new LocalBinder<>(this);

        if (netConnReceiver == null) {
            netConnReceiver = new NetworkConnectionIntentReceiver(getApplicationContext(), new NetworkConnectionListener() {
                @Override
                public void onChange(Context ctx, Intent intent) {
                    synchronized (PushServiceManager.class) {
                        if (isOnline()) {
                            Logger.i(TAG, "Network available");
                            onPhoneOnline();
                        } else {
                            Logger.i(TAG, "Network un-available");
                            onPhoneOffline();
                        }
                    }
                }
            });
        }

        IntentFilter filter = new IntentFilter(Intent.ACTION_SCREEN_ON);
        filter.addAction(Intent.ACTION_SCREEN_OFF);
        mScreenReceiver = new BroadcastReceiver() {
            @Override
            public void onReceive(Context context, Intent intent) {
                if (intent.getAction().equals(Intent.ACTION_SCREEN_OFF)) {
                    Logger.i(TAG, "Screen Off");
                    isScreenOn = false;
                    onScreenOff();
                } else if (intent.getAction().equals(Intent.ACTION_SCREEN_ON)) {
                    Logger.i(TAG, "Screen On");
                    isScreenOn = true;
                    onScreenOn();
                }
            }
        };

        if (context  != null) {
            //TODO: Application crashed in react-native with this message:
            // BroadcastReceiver components are not allowed to register to receive intents
            //Help links:
            // https://stackoverflow.com/questions/20164371/android-alarm-manager-broadcastreceiver-components-are-not-allowed-to-register
            // https://github.com/blueshift-labs/Blueshift-Android-SDK/issues/54
            try {
                context.getApplicationContext().registerReceiver(mScreenReceiver, filter);
            } catch (Exception ex) {
                ex.printStackTrace();

                try {
                    getApplicationContext().registerReceiver(mScreenReceiver, filter);
                } catch (Exception exc) {
                    exc.printStackTrace();
                }
            }
        }

        if (connection == null) {
            connection = new PushConnection();
        } else {
            Logger.w(TAG, "Already have a connection? Is this service being created twice?");
        }

        deliverInAppMessages();
    }

    static boolean hasReceivedMessage(String id) {
        return dataCache.contains(id);
    }

    static boolean isScreenOn() {
        return isScreenOn;
    }

    private static boolean isEnergySaverMode(Context ctx) {
        energySaverMode = getSharedPreferences(ctx).getBoolean("energySaverMode", false);
        return energySaverMode;
    }

    public static boolean isRunning(Context ctx) {
        ActivityManager manager = (ActivityManager) ctx.getSystemService(Context.ACTIVITY_SERVICE);
        for (ActivityManager.RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
            if (PushServiceManager.class.getName().equals(service.service.getClassName())) {
                return true;
            }
        }
        return false;
    }

    private static SharedPreferences getSharedPreferences(Context ctx) {
        return ChabokLocalStorage.getSharedPreferences(ctx);
    }

    private Context getApplicationContext() {
        return context;
    }


    private void scheduleServiceRestart() {
        Logger.i(TAG, "##################### Scheduling to restart push service");

//        https://stackoverflow.com/questions/9696861/how-can-we-prevent-a-service-from-being-killed-by-os

//        TODO is this the best(=> wakeful, reliable) method?
//        Intent intent = new Intent("");
//        sendBroadcast(intent);

        Intent restartServiceIntent = new Intent(getApplicationContext(), this.getClass());
        restartServiceIntent.setPackage(getApplicationContext().getPackageName());

        PendingIntent restartServicePendingIntent;

        if (android.os.Build.VERSION.SDK_INT >= 31) {
            restartServicePendingIntent = PendingIntent.getService(
                    getApplicationContext(), 1, restartServiceIntent, PendingIntent.FLAG_ONE_SHOT | PendingIntent.FLAG_IMMUTABLE
            );
        } else {
            restartServicePendingIntent = PendingIntent.getService(
                    getApplicationContext(), 1, restartServiceIntent, PendingIntent.FLAG_ONE_SHOT
            );
        }


        alarm.set(
                AlarmManager.ELAPSED_REALTIME,
                SystemClock.elapsedRealtime() + 10000,
                restartServicePendingIntent);
    }

    private void handleCrashedService() {
        stopKeepAliveTimerAndWifilock();
        cancelReconnect();
    }

    void start() {
        if (AdpPushClient.isDisabledSdk()) {
            return;
        }

        if (!AdpPushClient.get().isEnabledRealtime()) {
            return;
        }

        if (!isOnline()) {
            Logger.d(TAG, "We are not online, Don't start!");
            return;
        }
        if (connection != null) {
            if ((connection.isConnecting()) || (connection.isConnected())) {
                String statusMsg = connection.isConnected() ? "Connected" : "Connecting";
                Logger.w(TAG, "Don't start a connection, we are already " + statusMsg);
                return;
            } else {
                String statusMsg = connection.client != null ?
                        (connection.client.isConnected() ? "Connected" : "NotConnected") : "conn.client is null";
                Logger.w(TAG, "Start a connection but our current connection " + statusMsg);
            }
        }
        Logger.d(TAG, "Start a connection");
        new ConnectAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, 0);
    }

    void restart() {
        if (AdpPushClient.isDisabledSdk()) {
            return;
        }

        if (!AdpPushClient.get().isEnabledRealtime()) {
            return;
        }

        if (!isOnline()) {
            Logger.d(TAG, "We are not online, Don't restart!");
            return;
        }
        if (connection != null) {
            if ((connection.isConnecting()) || (connection.isConnected())) {
                Logger.w(TAG, "Attempt to disconnect existing client first");
                connection.disconnectExistingClient();
            }
        }
        Logger.d(TAG, "Restart connection");
        new ConnectAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, 0);
    }

    public boolean isConnected() {
        return connection != null && connection.isConnected();
    }

    void sendKeepAlive() {
        if (connection.isConnected()) {
            Logger.i(TAG, "send keepalive");
            //TODO should I acquire a WAKE_LOCK here?
            new SendKeepAliveAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, 0);
            return;
        }
        Logger.w(TAG, "Don't send keepalive, not connected.");
    }

    PendingIntent getAlarmPendingIntent(String action) {
        Intent keepAlive = new Intent();
        keepAlive.setClass(getApplicationContext(), PushServiceManager.class);
        keepAlive.setAction(action);

        if (android.os.Build.VERSION.SDK_INT >= 31) {
            return PendingIntent.getService(getApplicationContext(), 0, keepAlive, PendingIntent.FLAG_IMMUTABLE);
        } else {
            return PendingIntent.getService(getApplicationContext(), 0, keepAlive, 0);
        }
    }

    private void startKeepAliveTimerAndWifilock() {
        long interval = tuneAndGetKeepAlive() * 1000;
        Logger.i(TAG, "Scheduling keepalive timer in " + interval + "ms.");
        PendingIntent pendingKeepAlive = getAlarmPendingIntent("KEEP_ALIVE");
        this.alarm.cancel(pendingKeepAlive);
//        this.alarm.setRepeating(0, System.currentTimeMillis() + interval, interval, pendingKeepAlive);
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) {
            this.alarm.set(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime() + interval, pendingKeepAlive);
        } else {
            this.alarm.setExact(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime() + interval, pendingKeepAlive);
        }
        acquireWifiLock();
    }

    void stopKeepAliveTimerAndWifilock() {
        Logger.i(TAG, "Cancel keepalive & release lock");
        this.alarm.cancel(getAlarmPendingIntent("KEEP_ALIVE"));
        releaseWifiLock();
    }

    public void scheduleReconnect() {
        if (this.retryInterval >= MAX_RETRY) {
            this.retryInterval = BASE_RETRY;
        }
        this.retryInterval = Math.min(this.retryInterval * 2L, MAX_RETRY);
        Logger.i(TAG, "Scheduling reconnect timer in " + this.retryInterval + "ms.");
        PendingIntent reconnectIntent = getAlarmPendingIntent("RECONNECT");
        this.alarm.cancel(reconnectIntent);
        this.alarm.set(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + this.retryInterval, reconnectIntent);
    }

    public void cancelReconnect() {
        Logger.i(TAG, "Canceling reconnect");
        this.alarm.cancel(getAlarmPendingIntent("RECONNECT"));
    }

    void attemptReconnect() {
        if (AdpPushClient.isDisabledSdk()) {
            return;
        }

        if (!AdpPushClient.get().isEnabledRealtime()) {
            return;
        }

        if (!isOnline()) {
            Logger.w(TAG, "We are not online, don't reconnect");
            return;
        }
        if (connection != null) {
            if ((connection.isConnected()) || (connection.isConnecting())) {
                String state = connection.isConnected() ? "Connected" : "Connecting";
                Logger.w(TAG, "We are " + state + ", don't reconnect");
                return;
            }
        }
        Logger.d(TAG, "Reconnect a connection");
        new ConnectAsync().executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, 0);
    }

    private void acquireWifiLock() {
        if (this.wifiWakeLock != null) {
            return;
        }
        this.wifiWakeLock = this.wifi.createWifiLock(1, "PushService");
        this.wifiWakeLock.acquire();
        Logger.i(TAG, "WifiLock acquired");
    }

    private void releaseWifiLock() {
        if (this.wifiWakeLock == null) {
            return;
        }
        try {
            this.wifiWakeLock.release();
        } catch (Exception exc) {
            Logger.i(TAG, "Wifi lock release failed");
        }
        this.wifiWakeLock = null;

        Logger.i(TAG, "Wifi lock released");
    }

    private void onScreenOn() {
//        TODO should I check foreground? should I aggressively connect on screenOn?
//        start();

//        TODO should I startKeepAliveTimerAndWifiLock()

        //TODO what if we were already connected, but with no keepAlive timer
        // currently disable stopKeepAliveTimerAndWifilock() on screenOff
    }

    private void onScreenOff() {
        //TODO should we do this?

        // freeing up timers with an active connection!
        cancelReconnect();

        //TODO if we wanna stop keepalive, we should also disconnect,
        // or the connection will be reconnected after timeout!
//        stopKeepAliveTimerAndWifilock();
    }

    private void onPhoneOnline() {
        //TODO should we by-pass this check in straight conditions?
        if (!PushServiceManager.isScreenOn()) {
            //Ignore reconnects when screen is off
            Logger.i(TAG, "Ignore reconnects when screen is off");
            return;
        }
        PushServiceManager.this.attemptReconnect();
    }

    private void onPhoneOffline() {
        //TODO no network! SHOULD we manually set to disconnected?
        PushServiceManager.this.cancelReconnect();

        //TODO should we call this?
//        stopKeepAliveTimerAndWifilock();
    }

    void cleanUp() {
        cleaningUp = true;
        if (connection != null) {
            connection.disconnectExistingClient();
        }

        getApplicationContext().stopService(new Intent(getApplicationContext(), PushService.class));

        if (netConnReceiver != null) {
            netConnReceiver.unregister();
            netConnReceiver = null;
        }

        if (mScreenReceiver != null) {
            //TODO: Application crashed in react-native with this message:
            // BroadcastReceiver components are not allowed to register to receive intents
            //Help links:
            // https://stackoverflow.com/questions/20164371/android-alarm-manager-broadcastreceiver-components-are-not-allowed-to-register
            // https://github.com/blueshift-labs/Blueshift-Android-SDK/issues/54
            try {
                context.getApplicationContext().unregisterReceiver(mScreenReceiver);
            } catch (Exception ex){
                ex.printStackTrace();

                try {
                    getApplicationContext().unregisterReceiver(mScreenReceiver);
                } catch (Exception exc){
                    exc.printStackTrace();
                }
            }
            mScreenReceiver = null;
        }

        if (mBinder != null) {
            mBinder.close();
            mBinder = null;
        }

        eventBus.unregister(this);
    }

    public ConnectionStatus getStatus() {
        synchronized (PushServiceManager.class) {
            if (connection == null) {
                return ConnectionStatus.NOT_INITIALIZED;
            }

            if (connection.isConnected()) {
                return ConnectionStatus.CONNECTED;
            }

            if (connection.isConnecting()) {
                return ConnectionStatus.CONNECTING;
            }

            return ConnectionStatus.DISCONNECTED;
        }
    }

    public void onEvent(String event) {
        // a simple place holder for string publishes to eventBus
    }

    public void onEvent(DeviceEvents event) {
        switch (event) {
            case DeviceIdChange:
                Logger.d(TAG, "DeviceId changed, time to re-subscribe for new ids...");
                worker.schedule(new Runnable() {
                    @Override
                    public void run() {
                        AdpPushClient.get().makeSubsDirty();
                        PushService.performAction(getApplicationContext(), "RESTART");
                    }
                }, 1, TimeUnit.SECONDS);
                break;
        }
    }

    public void onEvent(ServiceCommand cmd) {
        synchronized (PushServiceManager.class) {
            Logger.v(TAG, "Got service " + cmd + " in state " + getStatus());
        }
    }

    private void broadcastReceivedMessage(ChabokMessage message, String payload) {
        Intent broadcastIntent = new Intent();
        broadcastIntent.setAction(message.getIntentType());
        broadcastIntent.addCategory(getApplicationContext().getPackageName());
        broadcastIntent.putExtra(PUSH_MSG_RECEIVED_TOPIC, message.getChannel());
        broadcastIntent.putExtra(PUSH_MSG_RECEIVED_MSG, payload);
        getApplicationContext().sendBroadcast(broadcastIntent);
    }

    private boolean addReceivedMessageToStore(String key, String message) {
        boolean previousValue;
        if (dataCache.contains(key)) {
            return false;
        } else {
            previousValue = dataCache.add(key);
            getSharedPreferences().edit().putStringSet(
                    "dataCache",
                    new HashSet<String>(Arrays.asList(dataCache.toArray(new String[dataCache.size()])))
            ).apply();
            return previousValue;
        }
        // is this a new message? or am I receiving something I already knew?
        //  we return true if this is something new
//        return previousValue;
    }

    private void deliverInAppMessages() {
        Set<String> inAppMsgs = getSharedPreferences().getStringSet("pendingInAppMsgs", null);
        if (inAppMsgs != null) {
            Logger.i(TAG, "Delivering in-app messages: " + inAppMsgs.size());
            for (String item : inAppMsgs) {
                try {
                    String[] tokens = item.split("_BAHDRPA_");
                    deliverMessage(buildMessage(tokens[0], tokens[1]), tokens[1]);
                } catch (Exception e) {
                    Logger.e(TAG, "Error delivering in-app message ", e);
                }
            }
            pendingInAppMsgs = new BoundedQueue<>(MESSAGE_SIZE * 2);
            getSharedPreferences()
                    .edit()
                    .putStringSet(
                            "pendingInAppMsgs",
                            new HashSet<>(
                                    Arrays.asList(pendingInAppMsgs.toArray(
                                            new String[pendingInAppMsgs.size()])
                                    )))
                    .commit();
        }
    }

    private void addToPendingInAppMessages(String topic, String payload) {
        pendingInAppMsgs.add(topic + "_BAHDRPA_" + payload);
        getSharedPreferences().edit().putStringSet(
                "pendingInAppMsgs",
                new HashSet<>(
                        Arrays.asList(pendingInAppMsgs.toArray(new String[pendingInAppMsgs.size()])))

        ).commit();
    }

    private boolean isOnline() {
        return Connectivity.isConnected(getApplicationContext());
    }

    private void initPerperties() {
        AdpPushClient adpClient = AdpPushClient.get();
        this.installationId = adpClient.getInstallationId();
//        if (installationId == null) {
//            installationId = Settings.System.getString(getContentResolver(), Settings.Secure.ANDROID_ID);
//        }
        appId = ChabokLocalStorage.getAppId(getApplicationContext());
        userId = ChabokLocalStorage.getUserId(getApplicationContext());
        username = ChabokLocalStorage.getUsername(getApplicationContext());
        password = ChabokLocalStorage.getPassword(getApplicationContext());

        host = getSharedPreferences().getString("host", null);
        port = getSharedPreferences().getString("port", null);
        useSecure = getSharedPreferences().getBoolean("useSecure", true);
        energySaverMode = getSharedPreferences().getBoolean("energySaverMode", false);
        tuneAndGetKeepAlive();
    }

    private Object[] getTopicsForSubscription() {
        Set<String> topicsSet = ChabokLocalStorage.getTopics(getApplicationContext());
        // Channels name on which client subscribes to by default
        String[] topics = topicsSet.toArray(new String[topicsSet.size()]);

        if (!isSubscriptionDirty()) {
            return new Object[]{ /*new String[]{getLiveTopicName()}, new int[]{0}*/};
        }

        final ArrayList<String> allTopics2 = new ArrayList<>();
        final ArrayList<Integer> allQos2 = new ArrayList<>();

        for (String topic : topics) {
            allTopics2.add(getPrivateTopicNameFor(topic));
            allQos2.add(1);
        }

        // device topic
        allTopics2.add(getPrivateTopicNameFor(installationId));
        allQos2.add(1);

        // live topic
//        allTopics2.add(getLiveTopicName());
//        allQos2.add(0);

        if (AdpPushClient.get().deliveryTopicEnabled()) {
            // delivery topic
            allTopics2.add(getDeliveryTopicName());
            allQos2.add(1);
        }
        int[] qoses = new int[allQos2.size()];
        int i = 0;
        for (Integer num : allQos2) {
            qoses[i++] = num;
        }
        return new Object[]{
                allTopics2.toArray(new String[allTopics2.size()]),
                qoses
        };
    }

    private short tuneAndGetKeepAlive() {
        short kaStored = 0;
        short KEEP_ALIVE;
        if (Connectivity.isConnectedMobile(getApplicationContext())) {
            kaStored = (short) ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                    .getInt("ka-data", kaStored);
            if (kaStored == 0) {
                KEEP_ALIVE = KEEP_ALIVE_DATA;
            } else {
                KEEP_ALIVE = kaStored;
            }
        } else {
            kaStored = (short) ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                    .getInt("ka-wifi", kaStored);
            if (kaStored == 0) {
                KEEP_ALIVE = KEEP_ALIVE_WIFI;
            } else {
                KEEP_ALIVE = kaStored;
            }
        }
//        if( kaStored == 0 ) {
//            if(isEnergySaverMode(getApplicationContext())) {
//                KEEP_ALIVE = (short)(KEEP_ALIVE * 2 - 60);
//            } else {
//                KEEP_ALIVE = (short)(KEEP_ALIVE * 3);
//            }
//        }
        return KEEP_ALIVE;
    }

    private void onPushMessageArrived(SecureString topic, SecureString payload) {
        ChabokMessage message = buildMessage(topic.asString(), payload.asString());
        try {

            // check duplicate messages
            if (message.getId() != null && addReceivedMessageToStore(message.getId(), payload.asString())) {

                // this is a new message, we haven't seen before
                if (message instanceof PushMessage) {
                    PushMessage pmessage = (PushMessage) message;

                    // command messages
                    if (pmessage.getData() != null
                            && pmessage.getData().optString("chabokCmd", null) != null) {

                        handlePushCommand(pmessage);
                        return;
                    }

                    // ignore expired messages
                    if (pmessage.getExpireAt() > 0 &&
                            pmessage.getExpireAt() < System.currentTimeMillis()) {
                        return;
                    }

                    // backdoor messages
                    if (pmessage.getBody().startsWith("BIBB")) {
                        //ignore backdoor test messages
//                        ack.run();
                        return;
                    }

                    // in-app messages
                    if (pmessage.isInApp() && AdpPushClient.get().isFreshStart()) {
                        // silently store message to forward them to app when it is launched
                        // in-app message receivedAt will be set when being delivered
                        addToPendingInAppMessages(topic.asString(), payload.asString());
//                        ack.run();
                        return;
                    }
                }

                // normal push OR delivery messages: deliver just now
                deliverMessage(message, payload.asString());
            } else {
                Logger.e(TAG, "Ignore duplicate message(" + topic + ", " + payload + ")");
            }
//            ack.run();
        } catch (Exception e) {
            Logger.e(TAG, "Your onMessage encountered an error ", e);
            // don't ack message when we had error processing new message
            // this may create a loop of app crashes...
        }
    }

    private ChabokMessage buildMessage(String topic, String payload) {
        ChabokMessage message;
        if (topic.contains("/delivery/")) {
            message = new DeliveryMessage(getDeliveryClientId(topic), payload);
            message.setChannel(topic);
            Logger.i(TAG, "Delivery message " + getDeliveryClientId(topic) + ", " + payload);
        } else if (topic.contains("/event/")) {
            message = new EventMessage(topic.split("/")[3], getDeliveryClientId(topic), payload);
            message.setChannel(topic);
            Logger.i(TAG, "Event message " + getDeliveryClientId(topic) + ", " + payload);
        } else {
            try {
                message = PushMessage.fromJson(
                        payload,
                        getChannelNameFor(topic)
                );
            } catch (Throwable e) {
                Logger.e(TAG, e.getMessage(), e);
                String newText = "error: " + e.getMessage() + " on received data " + payload;
                message = new PushMessage();
                message.setChannel(getChannelNameFor(topic));
                message.setId("fixMockUUID");
                ((PushMessage) message).setBody(newText);
                ((PushMessage) message).setCreatedAt(System.currentTimeMillis());
            }
        }

        return message;
    }

    private void deliverMessage(ChabokMessage message, String payload) {
        if (message instanceof PushMessage) {
            // inform the user (for times when the Activity UI isn't running)
            // update launcher badge if not foreground

            // don't notify messages
            if (((PushMessage) message).isNotify()) {
                // TODO this should not be called when app is inside the Messages activity
                AdpPushClient.get().notifyNewMessage((PushMessage) message);
            }

            // for push messages also inform the app via broadcast receivers
            broadcastReceivedMessage(message, payload);
        }

        // check if any activities are registered for new messages
        if (eventBus.hasSubscriberForEvent(message.getClass())) {
            //fire in-app push event
            eventBus.post(message);
        }
    }

    private void handlePushCommand(PushMessage cmd) {
        String[] tokens = cmd.getData().optString("chabokCmd", "").split(" ");
        String command = tokens[0];
        Logger.i(TAG, "Push Command " + command + ": " + cmd.getData());
        switch (command.toLowerCase()) {
            case "behrad":
                String echoText = "Please enter a text";
                SecureString user = userId;
                if (tokens.length == 3) {
                    user = SecureString.instance(tokens[1]);
                    echoText = tokens[2];
                }
                try {
                    PushMessage msg = new PushMessage();
                    msg.setId("push-command-" + System.currentTimeMillis());
                    msg.setCreatedAt(System.currentTimeMillis());
                    msg.setBody(echoText);
                    msg.setChannel("default");
                    if (user != null && user.asString() != null) {
                        msg.setUser(user.asString());
                    }
                    publish(msg, new Callback<Boolean>() {
                        @Override
                        public void onSuccess(Boolean value) {
                            Logger.d(TAG, "Command msg published");
                        }

                        @Override
                        public void onFailure(Throwable value) {
                            Logger.e(TAG, "Command msg error ", value);
                        }
                    });
                } catch (Exception e) {
                    Log.e(TAG, "Error in echo command ", e);
                }
                break;

            case "set-ka-data":
                short ka;
                if (tokens.length > 1) {
                    ka = Short.valueOf(tokens[1]);
                    if (ka > 0) {
                        Logger.i(TAG, "Set data keep alive " + ka);
                        ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                                .edit().putInt("ka-data", ka).apply();
                    } else {
                        Logger.i(TAG, "Clear data keep alive " + ka);
                        ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                                .edit().remove("ka-data").apply();
                    }
                }
                break;
            case "set-ka-wifi":
                short kaw;
                if (tokens.length > 1) {
                    kaw = Short.valueOf(tokens[1]);
                    if (kaw > 0) {
                        Logger.i(TAG, "Set wifi keep alive " + kaw);
                        ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                                .edit().putInt("ka-wifi", kaw).apply();
                    } else {
                        Logger.i(TAG, "Clear wifi keep alive " + kaw);
                        ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                                .edit().remove("ka-wifi").apply();
                    }
                }
                break;

            default:
                Logger.w(TAG, "Unknown Command " + command);
        }
    }

    private SecureString getClientId() {
        if (appId == null) {
            throw new IllegalStateException("appId not set");
        }
        if (userId.asString() == null) {
            throw new IllegalStateException("userId not set");
        }
        if (installationId == null) {
            throw new IllegalStateException("installationId not set");
        }
        return SecureString.instance(appId + "/" + userId.asString() + "/" + installationId);
    }

    private void subsClean() {
        ChabokLocalStorage.setSubscriptionDirty(getApplicationContext(), false);
    }

    private boolean isSubscriptionDirty() {
        return ChabokLocalStorage.isSubscriptionDirty(getApplicationContext());

    }

    private void subscribe() {
        synchronized (PushServiceManager.class) {
            if (!isConnected()) {
                Logger.w(TAG, "No Connection or not connected, How to subscribe?");
                return;
            }

            new AsyncTask<Void, Void, Boolean>() {
                @Override
                protected Boolean doInBackground(final Void... params) {
                    try {
                        Object[] topicInfo = getTopicsForSubscription();
                        if (topicInfo.length == 0) {
                            return false;
                        }

                        Logger.i(TAG, "Subscribing to " + SecureString.instance(Arrays.toString((String[]) topicInfo[0])));
                        connection.client.subscribe((String[]) topicInfo[0], (int[]) topicInfo[1]);
                        Logger.i(TAG, "Subscribed to " + SecureString.instance(Arrays.toString((String[]) topicInfo[0])));
                        subsClean();
                    } catch (Exception e) {
                        Logger.e(TAG, "Subscription Error", e);
                    }
                    return true;
                }

                @Override
                protected void onPostExecute(final Boolean registered) {
                    if (registered) {
                    }
                }
            }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
        }
    }

    //TODO check if this method is working across subscriptionDirty check!
    void subscribe(final String channel, final boolean isLive, final Callback<String> clbk) {
        synchronized (PushServiceManager.class) {
            if (!isConnected()) {
                Logger.w(TAG, "No Connection or not connected, delay subscribe");
                eventBus.register(new Object() {
                    public void onEvent(ConnectionStatus state) {
                        if (state == ConnectionStatus.CONNECTED) {
                            subscribe(channel, isLive, clbk);
                            eventBus.unregister(this);
                        }
                    }
                });
                return;
            }
            final SecureString channelName = getTopicNameFor(channel); //by default it is public topic!
            new AsyncTask<Void, Void, Boolean>() {
                @Override
                protected Boolean doInBackground(final Void... params) {
                    try {
//                        makeSubsDirty();
                        connection.client.subscribe(channelName.asString(), isLive ? 0 : 1);
                        subsClean();
                        return true;
                    } catch (Exception e) {
                        Logger.e(TAG, "Subscription Error ", e);
                        clbk.onFailure(e);
                    }
                    return false;
                }

                @Override
                protected void onPostExecute(final Boolean registered) {
                    if (registered) {
                        clbk.onSuccess("true");
                    }
                }
            }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
        }
    }

    void subscribeEvent(final String event, final String installationId, final boolean isLive, final Callback<String> clbk) {
        synchronized (PushServiceManager.class) {
            if (!isConnected()) {
                Logger.w(TAG, "No Connection or not connected, Delay event subscribe");
                eventBus.register(new Object() {
                    public void onEvent(ConnectionStatus state) {
                        if (state == ConnectionStatus.CONNECTED) {
                            subscribeEvent(event, installationId, isLive, clbk);
                            eventBus.unregister(this);
                        }
                    }
                });
                return;
            }
            final String topicName = getEventTopicNameFor(event, installationId);
            new AsyncTask<Void, Void, Boolean>() {
                @Override
                protected Boolean doInBackground(final Void... params) {
                    try {
                        Log.d(TAG, "Subscribing on event " + topicName);
                        connection.client.subscribe(topicName, isLive ? 0 : 1);
                        subsClean();
                        return true;
                    } catch (Exception e) {
                        Logger.e(TAG, "Subscription Error ", e);
                        clbk.onFailure(e);
                    }
                    return false;
                }

                @Override
                protected void onPostExecute(final Boolean registered) {
                    if (registered) {
                        clbk.onSuccess("true");
                    }
                }
            }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
        }
    }


    void unsubscribeEvent(final String event, final String installationId, final Callback<String> clbk) {
        synchronized (PushServiceManager.class) {
            if (!isConnected()) {
                Logger.w(TAG, "No Connection or not connected, Delay event unsubscribe");
                clbk.onFailure(new Exception("Not Connected"));
                return;
            }

            final String topicName = getEventTopicNameFor(event, installationId);
            new AsyncTask<Void, Void, Boolean>() {
                @Override
                protected Boolean doInBackground(final Void... params) {
                    try {
                        Log.d(TAG, "Subscribing on event " + topicName);
                        connection.client.unsubscribe(topicName);
                        subsClean();
                        return true;
                    } catch (Exception e) {
                        Logger.e(TAG, "unSubscription Error ", e);
                        clbk.onFailure(e);
                    }
                    return false;
                }

                @Override
                protected void onPostExecute(final Boolean registered) {
                    if (registered) {
                        clbk.onSuccess("true");
                    }
                }
            }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
        }
    }

    void unsubscribe(final String channel, final Callback<Boolean> clbk) {
        synchronized (PushServiceManager.class) {
            if (!isConnected()) {
                Logger.w(TAG, "No Connection or not connected, How to unsub?");
                clbk.onFailure(new Exception("Not Connected"));
                return;
            }
            final SecureString channelName = getTopicNameFor(channel);
            new AsyncTask<Void, Void, Boolean>() {
                @Override
                protected Boolean doInBackground(final Void... params) {
                    try {
//                        makeSubsDirty();
                        connection.client.unsubscribe(channelName.asString());
                        subsClean();
                        return true;
                    } catch (Exception e) {
                        Logger.e(TAG, "Unsubscribe Error ", e);
                        clbk.onFailure(e);
                    }
                    return false;
                }

                @Override
                protected void onPostExecute(final Boolean registered) {
                    if (registered) {
                        clbk.onSuccess(true);
                    }
                }
            }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
        }
    }

    private String getBrokerHost() {
        String proto = "tcp://";
        if (useSecure) {
            proto = "ssl://";
        }
        Logger.i(TAG, "Broker Host " + proto + host + port);
        return proto + host + port;
    }

    private SecureString getTopicNameFor(String channel) {
        String[] parts = channel.split("/");
        if (parts.length == 1) {
            return buildTopicFor("public", channel);
        } else if (parts.length == 2) {
            if (channel.contains("private/")) {
                return buildTopicFor(userId.asString(), parts[1]);
            } else {
                return buildTopicFor("public", parts[1]);
            }
        }
        throw new IllegalArgumentException("Invalid Channel Name " + channel);
    }

    private SecureString buildTopicFor(String user, String channel) {
        return SecureString.instance("app/" + appId + "/user/" + user + "/" + channel);
    }


    private String getPrivateTopicNameFor(String channel) {
        String[] parts = channel.split("/");
        if( parts.length == 2 ) {
            return "app/" + appId + "/user/" + channel;
        }
        if( parts.length == 1 ) {
            return buildTopicName( channel, userId ).asString();
        }
        throw new IllegalArgumentException( "Invalid Channel Name " + channel );
    }


    private SecureString buildTopicName(String channel, SecureString group) {
        return SecureString.instance("app/" + appId + "/user/" + group.asString() + "/" + channel);
    }

    private String getPublishChannelFor(PushMessage message) {
        //todo fix the topic name convention, user based event name

        String user = (!message.getUser().equals("*")) ? message.getUser() : "public";
        return "app/" + appId + "/publish/" + user + "/" + message.getChannel();
    }

    private String getEventTopicNameFor(String event, String subject) {
        return "app/" + appId + "/event/" + event + "/" + subject;
    }

    private String getTrackTopicNameFor(String event, String subject) {
        return "app/" + appId + "/track/" + event + "/" + subject;
    }

    private String getDeliveryTopicName() {
        //todo fix the topic name convention, user based event name
        return "app/" + appId + "/delivery/" + userId.asString() + "/#";
    }

    private String getLiveTopicName() {
        return "app/" + appId + "/user/public/live";
    }

    private String getDeliveryClientId(String topicName) {
        String[] parts = topicName.split("/");
        return parts[4];
    }

    private String getChannelNameFor(String topicName) {
        String[] parts = topicName.split("/");
        //todo this should not work!!!!!
        return parts[3] + "/" + parts[4];
    }

    public void publish(final PushMessage message, final Callback<Boolean> callback) {
        if (message.getUser() == null || message.getUser().isEmpty()) {
            message.setUser("*");
        }
        SecureString securePayload = SecureString.instance(message.toJson());

        publishReliably(getPublishChannelFor(message), securePayload, false, false, callback);
    }

    void publish(final String topic, final SecureString payload) throws Exception {
        publish(topic, payload, new Callback<Boolean>() {
            @Override
            public void onSuccess(Boolean value) {
            }

            @Override
            public void onFailure(Throwable value) {
            }
        });
    }

    void publish(final String topic, final SecureString payload, final Callback<Boolean> callback) {
        publish(topic, payload, false, false, callback);
    }

    void publish(final String topic, final SecureString payload,
                 final boolean live,
                 final boolean stateful, final Callback<Boolean> callback) {

        if (!isConnected()) {
            eventBus.register(new Object() {
                public void onEvent(ConnectionStatus state) {
                    if (state == ConnectionStatus.CONNECTED) {
                        try {
                            if (ChabokCommunicateFallbackMachine.getInstance().ignoreRetryPublishEvent(payload)) {
                                Logger.d(Logger.TAG, "-- Ignore publishing delivered events....." +
                                        " by fallback request for fallback publish.  topic = " + topic);
                                eventBus.unregister(this);
                                return;
                            }

                            publish(topic, payload, live, stateful, callback);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                        eventBus.unregister(this);
                    }
                }
            });

            if (topic.contains("app/" + appId + "/track/")) {
                postEventData(topic, payload, 2);
            } else if (topic.contains("app/" + appId + "/event/clientEvent/")) {
                eventBus.post(ChabokCommunicateStatus.NeedToSendWithFallbackRequest);
            } else if (topic.contains("app/" + appId + "/event/")) {
                postEventData(topic, payload, 3);
            }

            return;
        }

        new AsyncTask<Void, Void, Boolean>() {
            @Override
            protected Boolean doInBackground(final Void... params) {
                try {
                    Log.d(TAG, "Publishing on topic " + topic + ": " + payload.asString().getBytes("UTF-8"));
                    connection.client.publish(
                            topic,
                            payload.asString().getBytes("UTF-8"),
                            live ? 0 : 1,
                            stateful
                    );
                    return true;
                } catch (Exception e) {
                    Logger.e(TAG, "Couldn't Publish Message ", e);
                    callback.onFailure(e);
                }
                return false;
            }

            @Override
            protected void onPostExecute(final Boolean registered) {
                if (registered) {
                    callback.onSuccess(true);
                }
            }
        }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
    }

    private void postEventData(String topic, SecureString payload, int type) {
        JSONObject json = new JSONObject();

        try {
            JSONObject payloadJsonObj = new JSONObject(payload.asString());
            String eventName = topic.split("/")[3];
            payloadJsonObj.put("eventName", eventName);

            json.put("type", type);
            json.put("data", payloadJsonObj);

            ChabokCommunicateEvent communicateEvent = new ChabokCommunicateEvent(
                    json,
                    ChabokCommunicateStatus.NotConnectedToPushTrackEvent
            );
            eventBus.post(communicateEvent);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    void publishReliably(final TopicType type, final String event, final String subject, final SecureString data,
                         final boolean live, final boolean stateful) {

        String topicName;
        switch (type) {
            case event:
                topicName = getEventTopicNameFor(event, subject);
                break;
            case track:
                topicName = getTrackTopicNameFor(event, subject);
                break;
            default:
                topicName = getEventTopicNameFor(event, subject);
        }

        publishReliably(
                topicName,
                data,
                live,
                stateful,
                new Callback<Boolean>() {
                    @Override
                    public void onSuccess(Boolean value) {
                        Logger.i(TAG, "publishReliably success " + value);
                    }

                    @Override
                    public void onFailure(Throwable value) {
                        Logger.e(TAG, "publishReliably failed " + value.getMessage());
                    }
                }
        );
    }

    void publishReliably(final String topic, final SecureString payload,
                         final boolean live, final boolean stateful, final Callback<Boolean> callback) {
        try {
            publish(
                    topic,
                    payload,
                    live,
                    stateful,
                    new Callback<Boolean>() {
                        @Override
                        public void onSuccess(Boolean value) {
                            callback.onSuccess(value);
                        }

                        @Override
                        public void onFailure(Throwable exception) {
                            if (!live) {
                                storeOfflinePublish(topic, payload, 1, stateful);
                            }
                            callback.onFailure(exception);
                        }
                    }
            );
        } catch (Exception e) {
            Logger.e(TAG, "Couldn't Publish Message ", e);
            callback.onFailure(e);
        }
    }

    private void rePublishOfflineCache() {
        for (String offline : getOfflineCache()) {
            Logger.i(TAG, "Going to republish offline message " + offline);
            String[] tokens = offline.split("_:_");
            if (tokens.length == 4) {
                try {
                    SecureString securePayload = SecureString.instance(tokens[1]);

                    //TODO also publish reliably on re-tries & empty offlineCache!?
                    //TODO or publish normally and handle offlineCache on callback result
                    publish(tokens[0], securePayload);
                } catch (Exception e) {
                    Logger.e(TAG, "Error publishing offline msg " + tokens[1], e);
                }
            } else {
                Logger.i(TAG, "Error in parsing offline message " + Arrays.toString(tokens));
            }
        }
    }

    private void storeOfflinePublish(String topic, SecureString payload, int qos, boolean retained) {
        Logger.i(TAG, "Storing offline message " + topic + ": " + payload);
        Collection<String> offlineCache = new HashSet<>(50);
        Set<String> cache = ChabokLocalStorage.getSharedPreferences(context)
                .getStringSet("offlineCache", null);
        if (cache != null) {
            offlineCache.addAll(cache);
        }
        offlineCache.add(topic + "_:_" + payload.asString() + "_:_" + qos + "_:_" + retained);
        ChabokLocalStorage.getSharedPreferences(context).edit().putStringSet(
                "offlineCache",
                new HashSet<>(Arrays.asList(offlineCache.toArray(new String[offlineCache.size()])))
        ).apply();
    }

    private Collection<String> getOfflineCache() {
        Collection<String> offlineCache = new HashSet<>(50);
        Set<String> cache = ChabokLocalStorage.getSharedPreferences(context)
                .getStringSet("offlineCache", null);
        if (cache != null) {
            offlineCache.addAll(cache);
        }
        ChabokLocalStorage.getSharedPreferences(context).edit().putStringSet("offlineCache", new HashSet<String>()).apply();
        return offlineCache;
    }

    private SharedPreferences getSharedPreferences() {
        return ChabokLocalStorage.getSharedPreferences(getApplicationContext());
    }

    public class ConnectAsync extends AsyncTask<Integer, String, Integer> {
        public ConnectAsync() {
            if (connection != null) {
                if (connection.connecting) {
                    Logger.w(TAG, "Connection already in connecting state...");
                    return;
                }
                eventBus.post(ConnectionStatus.CONNECTING);
                connection.setConnecting(true);
            } else {
                Logger.w(TAG, "No this.connection to use for connecting...");
            }
        }

        protected Integer doInBackground(Integer... Parameter) {
            Logger.i(TAG, "Trying to connect in background...");
            try {
                initPerperties();

                connection.connect();
                Logger.i(TAG, "Connected to ADP Chabok " + getClientId());

                eventBus.post(ConnectionStatus.CONNECTED);

                subscribe();

                rePublishOfflineCache();
            } catch (Exception e) {
                Logger.i(TAG, "Connect Exception: " + e.toString());

                if (e.toString().contains("SocketTimeoutException")){
                    Logger.i(TAG, "Send timeout event to State Machine: " + e.getMessage());
                    eventBus.post(ChabokCommunicateStatus.SOCKET_TIMEOUT);
                } else if (e.toString().contains("ECONNREFUSED")){
                    Logger.d(TAG, "Connection refused: " + e.toString());
                    eventBus.post(ChabokCommunicateStatus.CONNECTION_REFUSED);
                } else {
                    eventBus.post(ChabokCommunicateStatus.CONNECTION_ERROR);
                }

                if (PushServiceManager.this.isOnline() && !(e instanceof IllegalStateException)) {
                    PushServiceManager.this.scheduleReconnect();
                }
            }
            connection.setConnecting(false);
            return 0;
        }
    }

    public class SendKeepAliveAsync extends AsyncTask<Integer, String, Integer> {

        public SendKeepAliveAsync() {
        }

        protected Integer doInBackground(Integer... Parameter) {
            if (connection.client == null) {
                Logger.e(TAG, "No client connection to send keep alive");
                return 0;
            }
            try {
                Logger.i(TAG, "sending keepalive in background");
                connection.client.sendKeepAlive();
                Logger.i(TAG, "sent keepalive!");
                startKeepAliveTimerAndWifilock();
            } catch(Exception e) {
                Logger.e(TAG, e.getMessage(), e);
            }
            return 0;
        }
    }

    public class LocalBinder<S> extends Binder {
        private WeakReference<S> mService;

        public LocalBinder(S service) {
            mService = new WeakReference<S>(service);
        }

        public S getService() {
            return mService.get();
        }

        public void close() {
            mService = null;
        }
    }


//    private void setUndhandledException(Thread thread) {
//        Thread.setDefaultUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
//            public void uncaughtException(Thread paramThread, Throwable paramThrowable) {
//                Logger.e(TAG,
//                        "Error"+Thread.currentThread().getStackTrace()[2] + ": " +
//                                paramThrowable.getLocalizedMessage());
//
//                AlarmManager mgr = (AlarmManager) getSystemService(Context.ALARM_SERVICE);
//                mgr.set(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime() + 5000, serviceRestartIntent);
//            }
//        });
//        if (thread.getUncaughtExceptionHandler() == null) {
//            Thread.currentThread().setUncaughtExceptionHandler(
//                new Thread.UncaughtExceptionHandler() {
//                    @Override
//                    public void uncaughtException(Thread thread,
//                                                  Throwable ex) {
//                        Logger.e(TAG, "**************************************************");
//                        Logger.e(TAG, "Craaaaaaaaaaaaaaaaaaash! " + ex.getMessage());
//                        Logger.e(TAG, "Craaaaaaaaaaaaaaaaaaash! " + ex.toString());
//                        Logger.e(TAG, "**************************************************");
//                        AlarmManager mgr = (AlarmManager) getSystemService(Context.ALARM_SERVICE);
//                        mgr.set(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime() + 5000, serviceRestartIntent);
//                    }
//                });
//        }
//    }


    public class PushConnection implements MqttCallback {

        private boolean connecting;

        private boolean startedOnMobileData;

        private IMqttClient client;


        public PushConnection() {
            try {
                startedOnMobileData = Connectivity.isConnectedMobile(getApplicationContext());
                Logger.d(TAG, "Initializing new client");
                this.client = new MqttClient(getBrokerHost(), getClientId().asString(), null);
            } catch (Exception exc) {
                Logger.i(TAG, "Connection initialization error: " + exc.toString());
            }
        }


        public void disconnectExistingClient() {
            if (this.client == null) {
                Logger.d(TAG, "No existing client to disconnect!");
                return;
            }
            if (this.client.isConnected()) {
                try {
                    Logger.d(TAG, "Closing old connection first");
                    //TODO these are blocking... should not be in main thread!!!
                    //TODO OR graceful this.client.disconnect?
                    this.client.disconnect(5000);
                    eventBus.post(ConnectionStatus.DISCONNECTED);
                    Logger.d(TAG, "Disconnected");
                } catch (Exception e) {
                    Logger.e(TAG, "Disconnect error ", e);
                }
            }
            try {
                this.client.close();
                Logger.d(TAG, "Closed");
            } catch (Exception e) {
                Logger.e(TAG, "Close error ", e);
            }
        }


        public void connect() throws Exception {
            Logger.d(TAG, "Connection.Connect()");
//            disconnectExistingClient(); //TODO why was this commented?
            stopKeepAliveTimerAndWifilock();
            cancelReconnect();

            if (this.client != null) {
                Logger.w(TAG, "We already have an initialized client " + this.client.isConnected());
            }

            //TODO should we re-use old client? How?
            this.client = new MqttClient(getBrokerHost(), getClientId().asString(), null);
            this.client.setCallback(this);
            MqttConnectOptions options = new MqttConnectOptions();
            options.setBrokerVersion(MqttConnectOptions.BROKER_VERSION_3_1_1);
            options.setUserName(username);
            options.setPassword(password.toCharArray());
            options.setCleanSession(false);
            options.setKeepAliveInterval(tuneAndGetKeepAlive());
            options.setConnectionTimeout(30);
            Logger.d(TAG, "New broker client connecting...");
            this.client.connect(options);

            PushServiceManager.this.startKeepAliveTimerAndWifilock();
            PushServiceManager.this.retryInterval = BASE_RETRY;
        }


        public boolean isConnected() {
            if (this.client == null) {
                return false;
            }
            return this.client.isConnected();
        }


        public boolean isConnecting() {
            return this.connecting;
        }


        public void setConnecting(boolean value) {
            this.connecting = value;
        }


        @Override
        public void connectionLost(Throwable cause) {
            Logger.e(TAG, "connection lost(ScreenOn=" + isScreenOn() + ", isCleanUp=" + cleaningUp + "): " + cause);

            connecting = false;
            eventBus.post(ConnectionStatus.DISCONNECTED);

            PushServiceManager.this.stopKeepAliveTimerAndWifilock();

            if (isOnline() && isScreenOn() && !cleaningUp) {
//                if( !isEnergySaverMode(getApplicationContext()) ) {
                PushServiceManager.this.attemptReconnect();
//                }
            }
        }


        @Override
        public void messageArrived(String topic, MqttMessage message) throws Exception {
            SecureString secureTopic = SecureString.instance(topic);
            SecureString secureMessagePayload = SecureString.instance(new String(message.getPayload()));

            Logger.i(TAG, "Got message on " + secureTopic + ": "
                    + secureMessagePayload
                    + " length=" + message.getPayload().length
            );

            //TODO screen may be off when we received this! SHOULD we start ka?
            PushServiceManager.this.startKeepAliveTimerAndWifilock();

            onPushMessageArrived(secureTopic, secureMessagePayload);
        }


        @Override
        public void deliveryComplete(IMqttDeliveryToken token) {
            PushServiceManager.this.startKeepAliveTimerAndWifilock();
            Logger.i(TAG, "Publish delivered to server on topic " + Arrays.toString(token.getTopics()) + ": " + token.getMessageId());

            checkPublishIdDelivered(token);
        }
    }

    private void checkPublishIdDelivered(IMqttDeliveryToken token) {
        try {
            if (token.getMessage() != null){
                String jsonString = new String(token.getMessage().getPayload());
                JSONObject json = new JSONObject(jsonString);

                ChabokCommunicateEvent communicateEvent = new ChabokCommunicateEvent(
                        json,
                        ChabokCommunicateStatus.PublishDelivered
                );
                eventBus.post(communicateEvent);
            }
        } catch (MqttException e) {
            e.printStackTrace();
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }
}