package com.adpdigital.push;

import android.content.Context;
import android.os.RemoteException;
import android.util.Log;

import com.android.installreferrer.api.InstallReferrerClient;
import com.android.installreferrer.api.InstallReferrerStateListener;
import com.android.installreferrer.api.ReferrerDetails;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;

class InstallReferrer implements InstallReferrerStateListener {
    private static final int ONE_SECOND = 1000;
    private static final int MAX_INSTALL_REFERRER_RETRIES = 2;
    private static final int RETRY_WAIT_TIME = ONE_SECOND * 3;

    private InstallReferrerClient client;
    private int retryNumber;
    private Timer timer = new Timer();

    // injected at constructor time and final
    private final Context context;
    private final InstallReferrerReadListener referrerCallback;
    private final AtomicBoolean hasInstallReferrerBeenRead;

    public InstallReferrer(Context context, InstallReferrerReadListener referrerCallback) {
        this.context = context;
        this.referrerCallback = referrerCallback;
        this.hasInstallReferrerBeenRead = new AtomicBoolean(false);

        //TODO: Run startConnection Just once like Adjust.
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                startConnection();
            }
        }, 0);

        this.retryNumber = 0;
    }

    public void startConnection() {
        if (this.client != null) {
            try {
                Reflection.invokeInstanceMethod(this.client,
                        "startConnection", null);
                return;
            } catch (Exception e) {
                Logger.e(Logger.TAG,
                        "Call to Play startConnection error: " + e.getMessage(),
                        e);
            }
        }

        closeReferrerClient();

        if (hasInstallReferrerBeenRead.get()) {
            Logger.d(Logger.TAG,"Install referrer has already been read");
            return;
        }

        if (context == null) {
            return;
        }

        try {
            client = InstallReferrerClient.newBuilder(context).build();

            client.startConnection(this);
        } catch (Exception ex) {
            Logger.e(Logger.TAG, "startConnection error (" + ex.getMessage() + ")" +
                            " thrown by (" + ex.getClass().getCanonicalName() + ")",
                    ex);
        }
    }

    /**
     * Terminate connection to install referrer service.
     */
    private void closeReferrerClient() {
        if (client == null) {
            return;
        }

        try {
            Reflection.invokeInstanceMethod(client, "endConnection", null);
            Logger.d(Logger.TAG,"Install Referrer API connection closed");
        } catch (Exception e) {
            Logger.d(Logger.TAG, "closeReferrerClient error " +
                    e.getMessage() + "thrown by " +
                    e.getClass().getCanonicalName());
        }
        client = null;
    }

    /**
     * Called to notify that setup is complete.
     *
     * @param responseCode The response code from {@link InstallReferrerClient.InstallReferrerResponse} which returns the
     *     status of the setup process.
     */
    @Override
    public void onInstallReferrerSetupFinished(int responseCode) {
        boolean retryAtEnd = false;
        switch (responseCode) {
            /** Success. */
            case InstallReferrerClient.InstallReferrerResponse.OK:
                // Connection established
                try {
                    if (client == null) {
                        client = InstallReferrerClient.newBuilder(context).build();
                    }

                    ReferrerDetails installReferrerDetails = client.getInstallReferrer();
                    String installReferrer = installReferrerDetails.getInstallReferrer();
                    long referrerClickTimestampSeconds = installReferrerDetails.getReferrerClickTimestampSeconds();
                    long installBeginTimestampSeconds = installReferrerDetails.getInstallBeginTimestampSeconds();
                    referrerCallback.onInstallReferrerRead(installReferrer, referrerClickTimestampSeconds, installBeginTimestampSeconds);

                    hasInstallReferrerBeenRead.set(true);
                    Logger.d(Logger.TAG,"Install Referrer read successfully. Closing connection");
                } catch (RemoteException e) {
                    Logger.e(Logger.TAG,"Couldn't get install referrer from client ("+e.getMessage()+"). Retrying...",e);
                    retryAtEnd = true;
                } catch (Exception e){
                    Logger.e(Logger.TAG,"Install referrer client is null for unknown reason.",e);
                }
                break;
            /** Install Referrer API not supported by the installed Play Store app. */
            case InstallReferrerClient.InstallReferrerResponse.FEATURE_NOT_SUPPORTED:
                // API not available on the current Play Store app
                Logger.d(Logger.TAG,"Install Referrer API not supported by the installed Play Store app. Closing connection");
                break;
            /** Could not initiate connection to the Install Referrer service. */
            case InstallReferrerClient.InstallReferrerResponse.SERVICE_UNAVAILABLE:
                // Connection could not be established
                Logger.d(Logger.TAG,"Could not initiate connection to the Install Referrer service. Retrying...");
                retryAtEnd = true;
                break;
            /**
             * Play Store service is not connected now - potentially transient state.
             *
             * <p>E.g. Play Store could have been updated in the background while your app was still
             * running. So feel free to introduce your retry policy for such use case. It should lead to a
             * call to {@link #startConnection(InstallReferrerStateListener)} right after or in some time
             * after you received this code.
             */
            case InstallReferrerClient.InstallReferrerResponse.SERVICE_DISCONNECTED:
                // Connection could not be established
                Logger.d(Logger.TAG,"Play Store service is not connected now. Retrying ...");
                retryAtEnd = true;
                break;
            /** General errors caused by incorrect usage */
            case InstallReferrerClient.InstallReferrerResponse.DEVELOPER_ERROR:
                Logger.d(Logger.TAG,"Install Referrer API general errors caused by incorrect usage. Retrying...");
                retryAtEnd = true;
                break;
            default:
                Logger.d(Logger.TAG,"Unexpected response code of install referrer response: " + responseCode + ". Closing connection");
                break;
        }

        if (retryAtEnd) {
            retry();
        } else {
            closeReferrerClient();
        }
    }

    /**
     * Called to notify that connection to install referrer service was lost.
     *
     * <p>Note: This does not remove install referrer service connection itself - this binding to the
     * service will remain active, and you will receive a call to {@link
     * #onInstallReferrerSetupFinished(int)} when install referrer service is next running and setup
     * is complete.
     */
    @Override
    public void onInstallReferrerServiceDisconnected() {
        // Try to restart the connection on the next request to
        // Google Play by calling the startConnection() method.
        Logger.d(Logger.TAG,"Connection to install referrer service was lost. Retrying ...");
        retry();
    }

    private void retry() {
        if (hasInstallReferrerBeenRead.get()) {
            Logger.d(Logger.TAG,"Install referrer has already been read");
            closeReferrerClient();
            return;
        }

        // Check increase retry counter
        if (retryNumber + 1 > MAX_INSTALL_REFERRER_RETRIES) {
            Logger.d(Logger.TAG,"Limit number of retry of "+MAX_INSTALL_REFERRER_RETRIES+" for install referrer surpassed");
            return;
        }

//        long firingIn = retryTimer.getFireIn();
//        if (firingIn > 0) {
//            Logger.d(Logger.TAG,"Already waiting to retry to read install referrer in "+firingIn+" milliseconds");
//            return;
//        }

        retryNumber++;
        Logger.d(Logger.TAG,"Retry number "+retryNumber+" to connect to install referrer API");

        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                startConnection();
            }
        }, RETRY_WAIT_TIME);
    }
}