package com.adpdigital.push;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import androidx.annotation.NonNull;
import androidx.core.app.NotificationCompat;
import android.util.Log;
import android.view.View;
import android.webkit.MimeTypeMap;
import android.widget.RemoteViews;

import com.adpdigital.push.notification.SimpleCollapseNotification;
import com.adpdigital.push.notification.SimpleExpandedNotification;
import com.adpdigital.push.service.ChabokNotificationOpenedReceiver;
import com.google.firebase.messaging.FirebaseMessaging;
import com.google.firebase.messaging.FirebaseMessagingService;
import com.google.firebase.messaging.RemoteMessage;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.Socket;
import java.net.URL;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;

import de.greenrobot.event.EventBus;
import me.leolin.shortcutbadger.ShortcutBadger;

import static com.network.InstallationModel.PROPERTY_DEVICE_TOKEN;

/**
 * Created by behrad on 4/28/16.
 */
public class ChabokFirebaseMessaging extends FirebaseMessagingService {

    public static final String TAG = ChabokFirebaseMessaging.class.getName();

    private static final boolean CONNECT_FOR_GCM = false;

    private static int NOTIFICATION_ID = 0;

    private EventBus eventBus = EventBus.getDefault();

    private static final ScheduledExecutorService worker =
            Executors.newSingleThreadScheduledExecutor();


    public ChabokFirebaseMessaging() {
        super();
        eventBus.register(this);
    }

    @Override
    public void onDestroy() {
        eventBus.unregister( this );
        super.onDestroy();
    }

    @Override
    public void onNewToken(String token) {
        super.onNewToken(token);

        refreshToken(token, getApplicationContext());
    }

    /**
     * To prevent show notification by your self
     *
     * @param data is GCM push notification in onMessageReceived callback.
     * @return is chabok push notification or not
     */
    public static boolean isChabokPushNotification(Bundle data) {
        return data != null && data.containsKey("fromChabok");
    }

    /**
     * To prevent show notification by your self
     *
     * @param data is FCM push notification in onMessageReceived callback.
     * @return is chabok push notification or not
     */
    public static boolean isChabokPushNotification(Map<String, String> data) {
        return data != null && data.containsKey("fromChabok");
    }

    public static void refreshToken(String token, Context context) {
        if (AdpPushClient.isDisabledSdk()) {
            return;
        }

        if (!AdpPushClient.get().isEnabledPushNotification()) {
            return;
        }

        if (token == null) {
            Logger.w(TAG, "Push notification token is null. Please provide valid token");
            return;
        }

        if (context == null) {
            Log.e(TAG, "The context parameter is null in refreshToken method");
            return;
        }

        String currentToken = getSharedPreferences(context).getString(PROPERTY_DEVICE_TOKEN, null);
        if (token.equalsIgnoreCase(currentToken)) {
            Logger.i(TAG, "GOT same token, don't update installation");
            return;
        }

        Logger.d(TAG, "FCM Registration Token is : " + token +"\n\n");

        final DeviceToken deviceToken = new DeviceToken(SecureString.instance(token));
        deviceToken.setCallerId(ChabokFirebaseMessaging.class.getCanonicalName());
        EventBus.getDefault().post(deviceToken);

        String[] subs = AdpPushClient.get().getSubscriptions();
        for(String topic : subs) {
            if (!topic.startsWith("private/")) {
                ChabokFirebaseMessaging.subscribe(topic);
            }
        }
    }

    @Override
    public void onMessageReceived(RemoteMessage remoteMessage) {
        super.onMessageReceived(remoteMessage);

        onMessageReceived(remoteMessage, getApplicationContext());
    }

    public static void onMessageReceived(RemoteMessage remoteMessage, final Context context){
        if (AdpPushClient.isDisabledSdk()) {
            return;
        }

        if (!AdpPushClient.get().isEnabledPushNotification()) {
            return;
        }

        String from = remoteMessage.getFrom();
        Map<String, String> data = remoteMessage.getData();

        Logger.d(TAG, "Received FCM Notification for " + from + " \n\n Data : " + data);

        if (context == null){
            Log.e(TAG, "The context parameter is null in onMessageReceived method");
            return;
        }

        String deviceId = data.get("deviceId");
        if( deviceId != null ) {
            String currentDeviceId = AdpPushClient.get().getInstallationId();
            if( !deviceId.equalsIgnoreCase(currentDeviceId) ) {
                Logger.w(TAG,
                        "Ignoring FCM, unmatched deviceId " + deviceId + " != " + currentDeviceId);
                return;
            }
        }

        if (data.containsKey("cancelNotif")){
            try {
                Integer notifId = Integer.parseInt(data.get("cancelNotif"));
                NotificationManager mNotificationManager = (NotificationManager)
                        context.getSystemService(Context.NOTIFICATION_SERVICE);
                if (mNotificationManager != null) {
                    if (notifId == -1) {
                        mNotificationManager.cancelAll();
                    } else {
                        mNotificationManager.cancel(notifId);
                    }
                }
            } catch (Exception ex){
                ex.printStackTrace();
            }
            return;
        }

        // Silent FCM
        String alertText = data.get("message");
        String title = data.get("title");
        if(
                (alertText == null || alertText.equalsIgnoreCase("")) &&
                        (title == null || title.equalsIgnoreCase(""))
        ) {
            Logger.i(TAG, "Silent FCM, no title or text");
            return;
        }

        // Don't notify GCM for received messages
        final String messageId = data.get("messageId");
        if( messageId != null && PushServiceManager.hasReceivedMessage(messageId) ) {
            Logger.i(TAG, "MessageId already received, Swallow FCM!");
            return;
        }

        final String pushPayload = data.get("push");

        final String topicName = "app/" + AdpPushClient.get().getAppId()
                + "/user/" + AdpPushClient.get().getUserId()
                + "/" + data.get("collapse_key"); //TODO collapse_key == public/*

        if (title == null) {
            title = data.get("messageFrom");
        }

        int badgeNumber = 0;
        if (data.get("androidBadge") != null){
            badgeNumber = Integer.valueOf(data.get("androidBadge"));
        }

        final ChabokNotification notification = new ChabokNotification(
                messageId,
                title,
                alertText,
                badgeNumber,
                data
        );

        final String trackId = data.get("trackId");
        if (trackId != null){
            notification.setTrackId(trackId);
        }

        notification.setTopicName(data.get("collapse_key"));

        Bundle extraData = notification.getExtras();
        try {
//            if (AdpPushClient.get().isForeground()) {
//                Log.w(TAG, "Don't notify GCM when app is foreground!");
            // Ignore GCM
//            } else {
            // app is background
            try {
                // Don't notify GCM for received messages
                if( messageId != null ) {
                    if( PushServiceManager.hasReceivedMessage(messageId) ||
                            AdpPushClient.hasNotified(messageId) ) {
                        // Ignore GCM
                        Log.w(TAG, "Don't notify, messageId already delivered!");
                        return;
                    }
                }

                final Class activityToOpen = AdpPushClient.get().getNotifActivityClass(extraData);
                final Context ctx = context;

                int delay = 0;
                if( CONNECT_FOR_GCM ) {
                    PushService.performAction(context, "START");
                    delay = 15;
                }
                Runnable notifTask = new Runnable() {
                    public void run() {
                        if(pushPayload != null) {
                            Logger.w(TAG, "We have push payload... handle as a new chabok in-app message!!");
                            Set<String> inAppMsgs = getSharedPreferences(context).getStringSet("pendingInAppMsgs", null);
                            if(inAppMsgs == null) {
                                inAppMsgs = new HashSet<>();
                            }
                            inAppMsgs.add( topicName + "_BAHDRPA_" + pushPayload );
                            getSharedPreferences(context).edit().putStringSet(
                                    "pendingInAppMsgs",
                                    new HashSet<>(
                                            Arrays.asList(inAppMsgs.toArray(new String[inAppMsgs.size()]))
                                    )
                            ).apply();
                        }
                        ChabokFirebaseMessaging.sendNotification(ctx, activityToOpen, notification);
                    }
                };
                worker.schedule(notifTask, delay, TimeUnit.SECONDS);

            } catch (ClassNotFoundException e) {
                Log.e(TAG, "cannot send notification when no subscriber for Intents", e);
            }
//            }
        } catch(Exception e) {
            Log.e(TAG, "Error handling FCM ", e);
            ChabokFirebaseMessaging.sendNotification(context, null, notification);
        }
    }

    public void onEvent(createNotificationEvent event) {
        Log.w(TAG, "Should? Notify FCM message (canNotify,isCanceled): " + event.isCanNofity() + "," + event.isCanceled());
        if( !event.isCanceled() ) {
            sendNotification(event);
        }
    }

    private void sendNotification(createNotificationEvent event) {
        String title = event.getIntent().getExtras().getString("title");
        if (title == null) {
            title = event.getIntent().getExtras().getString("messageFrom");
        }
        ChabokNotification gcmNotif = new ChabokNotification(
                event.getIntent().getExtras().getString("messageId"),
                title,
                event.getIntent().getExtras().getString("message"),
                Integer.valueOf(event.getIntent().getExtras().getString("androidBadge", "0"))
                        + getUnseenBadge(),
                event.getIntent().getExtras()
        );
        sendNotification(this, event.getClient().getActivityClass(), gcmNotif);
    }

    // Put the message into a notification and post it.
    @SuppressLint("RestrictedApi")
    @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
    public static void sendNotification(Context ctx, Class viewClass, ChabokNotification msg ) {
        try {
            boolean notifDelivery = false;
            PackageInfo packageInfo = ctx.getPackageManager().getPackageInfo(ctx.getPackageName(), 0);
            ApplicationInfo appInfo = ctx.getPackageManager().getApplicationInfo(packageInfo.packageName, 0);
            CharSequence title = appInfo.nonLocalizedLabel;
            CharSequence body = null;
            CharSequence channelId = "default";

            Intent intent = new Intent(ctx, ChabokNotificationOpenedReceiver.class);
//            if( viewClass != null ) {
//                intent = new Intent(ctx, viewClass);
//            } else {
//                intent = new Intent(ctx, ChabokNotificationOpenedReceiver.class);
//            }
            if (android.os.Build.VERSION.SDK_INT <= Build.VERSION_CODES.KITKAT) {
                //This a bug in android lower than 4.4
                //https://stackoverflow.com/a/22883183/5229540
                intent.setFlags(0);
            } else {
                intent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            }
            intent.putExtra("CHABOK_NOTIFICATION", true);
            if( viewClass != null ) {
                intent.putExtra("CHABOK_NOTIFICATION_HANDLER", viewClass.getName());
            }

            int notificationId = ++NOTIFICATION_ID;
            if (!intent.hasExtra("notificationId")){
                intent.putExtra("notificationId", notificationId);
            } else {
                notificationId = intent.getIntExtra("notificationId", notificationId);
            }

            String msgId = msg.getId();
            String trackId = msg.getTrackId();
            if(msg.getExtras() != null) {
                Log.i(TAG, "Notification Extras " + SecureString.instance(msg.getExtras().toString()));
                intent.putExtras(msg.getExtras());
                if(msg.getExtras().getString("messageId") != null) {
                    msgId = msg.getExtras().getString("messageId");
                }

                if(msg.getExtras().getString("trackId") != null) {
                    trackId = msg.getExtras().getString("trackId");
                }

                if (msg.getExtras().getString("title") != null) {
                    title = msg.getExtras().getString("title");
                }

                if (msg.getExtras().getString("body") != null) {
                    body = msg.getExtras().getString("body");
                } else if (msg.getExtras().getString("message", null) != null){
                    body = msg.getExtras().getString("message");
                }

            }

            if (msg != null && msg.getTopicName() != null){
                channelId = getChannelIdFromTopic(msg.getTopicName());
            }

            if (msg.getMessage() != null) {
                channelId = getChannelIdFromTopic(msg.getMessage().getChannel());
                msg.setTopicName(msg.getMessage().getChannel());
                intent.putExtra("collapse_key", msg.getMessage().getChannel());
                intent.putExtra("messageId", msg.getMessage().getId());
                intent.putExtra("trackId", msg.getMessage().getTrackId());
                if (msg.getMessage().getData() != null) {
                    intent.putExtra("data", msg.getMessage().getData().toString());
                }
                intent.putExtra("live", msg.getMessage().isLive());
                intent.putExtra("stateful", msg.getMessage().isStateful());
                intent.putExtra("inapp", msg.getMessage().isInApp());
                intent.putExtra("topicName", msg.getMessage().getChannel());
                intent.putExtra("expireAt", msg.getMessage().getExpireAt());
                intent.putExtra("receivedAt", msg.getMessage().getReceivedAt());
                if (msg.getMessage() instanceof PushMessage) {
                    intent.putExtra("title", msg.getMessage().getAlertTitle());
                    intent.putExtra("body", msg.getMessage().getAlertText());
                    if (msg.getMessage().getAlertText() == null ||
                            msg.getMessage().getAlertText().trim() == ""){
                        intent.putExtra("body", msg.getMessage().getBody());
                        body = msg.getMessage().getBody();
                    } else {
                        body = msg.getMessage().getAlertText();
                    }
                }

                title = msg.getMessage().getAlertTitle();
            }

            if(msgId != null) {
                AdpPushClient.get().addNotifiedMessage(msgId);
            }


            NotificationManager notificationManager = (NotificationManager) ctx.getSystemService(Context.NOTIFICATION_SERVICE);

            JSONObject notificationObject = null;
            if (msg.getMessage() != null){
                notificationObject = msg.getMessage().getNotification();
            } else if (msg.getExtras() != null) {
                String json = bundleToJson(msg.getExtras());

                notificationObject = new JSONObject(json);
                if (notificationObject.has("notifDelivery")) {
                    notifDelivery = notificationObject.getBoolean("notifDelivery");
                }
            }

            if (notificationObject != null){
                if (notificationObject.has("clickUrl")){
                    intent.putExtra("clickUrl", notificationObject.getString("clickUrl"));
                }
            }

            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
                String channelName = "Personal default";

                if (notificationObject != null && notificationObject.has("androidChannelId")) {
                    channelId = notificationObject.optString("androidChannelId");
                    if (channelId != null) {
                        channelName = channelId.toString();
                    }
                }

                if (notificationObject != null && notificationObject.has("androidChannelName")) {
                    channelName = notificationObject.optString("androidChannelName");
                }

                if (channelId.equals(AdpPushClient.get().getInstallationId()) ||
                        channelId.equals("This Device")){
                    channelId = Constants.DEFAULT_CHANNEL;
                }

                if ("Personal default".contains(channelId)) {
                    channelId = Constants.DEFAULT_CHANNEL;
                }

                int ledColor = getLedColorFromNotificationObject(notificationObject);

                NotificationChannel notificationChannel = new ChabokNotificationChannel(channelId.toString(), ctx)
                        .setLightColor(ledColor)
                        .setChannelName(channelName)
                        .setSound(getSoundNameFromNotification(msg))
                        .setImportance(NotificationManager.IMPORTANCE_HIGH)
                        .build();

                if (notificationChannel != null) {
                    channelId = notificationChannel.getId();
                }
            }

            int uniqueInt = (int) (System.currentTimeMillis() & 0xfffffff);
//            PendingIntent contentIntent = PendingIntent.getActivity(
//                    ctx,
//                    uniqueInt,
//                    intent,
//                    PendingIntent.FLAG_UPDATE_CURRENT
//            );

            PendingIntent contentIntent;
            PendingIntent dismissPendingIntent = getDimissPendingIntent(ctx, msgId, trackId, intent.getExtras());

            if (android.os.Build.VERSION.SDK_INT >= 31) {
                contentIntent = PendingIntent.getBroadcast(ctx, uniqueInt, intent, PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE);
            } else {
                contentIntent = PendingIntent.getBroadcast(ctx, uniqueInt, intent, PendingIntent.FLAG_UPDATE_CURRENT);
            }

            boolean hideLargeIcon = false;
            int notificationTemplateType = 0;

            try {
                notificationTemplateType = 0;
                hideLargeIcon = false;

                if (notificationObject != null && notificationObject.has("template")){
                    JSONObject template;
                    if (notificationObject.get("template") != null &&
                            notificationObject.get("template").getClass() == String.class) {
                        String templateString = notificationObject.getString("template");
                        template = new JSONObject(templateString);
                    } else {
                        template = notificationObject.getJSONObject("template");
                    }

                    if (template != null && template.has("type")){
                        notificationTemplateType = template.getInt("type");
                    }

                    if (template != null && template.has("hideLargeIcon")){
                        hideLargeIcon = template.getBoolean("hideLargeIcon");
                    }
                }
            } catch (JSONException e) {
                e.printStackTrace();
            } catch (Exception e) {
                e.printStackTrace();
            }

            CharSequence mContentTitle;
            CharSequence mContentText;
            Bitmap mLargeIcon;

            NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(ctx,channelId.toString())
                    .setAutoCancel(true)
                    .setPriority(Notification.PRIORITY_HIGH)
                    .setTicker(msg.getText())
                    .setContentTitle(title)
                    .setContentText(msg.getText())
                    .setContentIntent(contentIntent)
                    .setDeleteIntent(dismissPendingIntent);

            mContentTitle = title;
            mContentText = msg.getText();

            if (body != null){
                if (!body.toString().trim().equals("")){
                    mBuilder.setContentText(body);
                }
            }

            mBuilder.setSmallIcon(getNotificationIcon());

            Drawable appIcon = appInfo.loadIcon(ctx.getPackageManager());
            Bitmap bm = DrawableUtil.drawableToBitmap(appIcon);

            mBuilder.setLargeIcon(bm);
            mLargeIcon = bm;

            int badge = -1;
            Bitmap bigPictureBitmap = null;
            if (notificationObject != null) {
                if (notificationObject.has("androidBadge")){
                    badge = notificationObject.getInt("androidBadge");
                }

                if (notificationObject.has("smallIcon")) {
                    String smallIcon = notificationObject.getString("smallIcon");
                    if (smallIcon != null && smallIcon.trim() != "") {
                        int icon = getResourceIcon(smallIcon, ctx);
                        if (icon != 0) {
                            mBuilder.setSmallIcon(icon);
                        }
                    }
                }
                if (notificationObject.has("ledColor")){
                    int ledColor = getLedColorFromNotificationObject(notificationObject);
                    if (ledColor != -1) {
                        mBuilder.setLights(ledColor, 300,3000);
                    }
                }

                if (notificationObject.has("actions")){
                    JSONArray actions = null;
                    if (notificationObject.get("actions").getClass() == String.class) {
                        String jsonActions = notificationObject.getString("actions");
                        actions = new JSONArray(jsonActions);
                    } else {
                        actions = notificationObject.getJSONArray("actions");
                    }
                    if (actions != null){
                        addNotficationActionToBuilder(
                                mBuilder,
                                actions,
                                ctx,
                                msgId,
                                trackId,
                                intent.getExtras()
                        );
                    }
                }

                if (notificationObject.has("mediaUrl")) {
                    String mediaUrl = notificationObject.getString("mediaUrl");
                    if (mediaUrl != null && mediaUrl.trim() != "") {
                        bigPictureBitmap = getBitmapFromURL(mediaUrl);
                        if (bigPictureBitmap != null) {
                            NotificationCompat.BigPictureStyle bigPictureStyle =  new NotificationCompat
                                    .BigPictureStyle()
                                    .bigPicture(bigPictureBitmap);


                            if (title != null){
                                bigPictureStyle.setBigContentTitle(title);
                            }

                            if (body != null){
                                bigPictureStyle.setSummaryText(body);
                            }

                            mBuilder.setStyle(bigPictureStyle);

                            if (notificationObject.has("useAsLargeIcon")){
                                boolean useAsLargeIcon = notificationObject.getBoolean("useAsLargeIcon");
                                if (useAsLargeIcon) {
                                    mBuilder.setLargeIcon(bigPictureBitmap);
                                }
                            }
                        }
                    }
                }

                if (notificationObject.has("color")){
                    String colorHex = notificationObject.getString("color");
                    if (colorHex != null && colorHex.trim() != ""){
                        String sharpSign = "#";
                        if (colorHex.startsWith("#")){
                            sharpSign = "";
                        }
                        colorHex = sharpSign + colorHex;
                        if (isValidHexColor(colorHex)) {
                            mBuilder.setColor(Color.parseColor(colorHex));
                        }
                    }
                } else {
                    try {
                        int color = AdpPushClient.get().getDefaultNotificationColorFromMetadata();
                        mBuilder.setColor(color);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                if (notificationObject.has("largeIcon")){
                    String largeIcon = notificationObject.getString("largeIcon");
                    if (largeIcon != null && largeIcon.trim() != "") {
                        Bitmap largeIconBitmap = getBitmapFromURL(largeIcon);
                        if (largeIconBitmap != null) {
                            mBuilder.setLargeIcon(largeIconBitmap);
                        }
                    }
                }

                if (notificationObject.has("groupKey")) {
                    mBuilder.setGroup(notificationObject.getString("groupKey"));
                }

                if (notificationObject.has("groupSummery")) {
                    boolean groupSummery = notificationObject.getBoolean("notificationObject");
                    //set this notification as the summary for the group
                    mBuilder.setGroupSummary(groupSummery);
                }
            }

            HashMap<String, HashMap> settings = AdpPushClient.get().getNotificationSettings();

            if(settings != null && settings.containsKey(msg.getTopicName())) {
                msg.setAlert(Boolean.getBoolean(settings.get(msg.getTopicName()).get("alert").toString()));
                if (settings.get(msg.getTopicName()).get("sound") != null) {
                    msg.setSound((String)settings.get(msg.getTopicName()).get("sound"));
                }
            } else {
                if (settings != null) {
                    Logger.d(TAG, "No Notification settings? " + Arrays.toString(settings.keySet().toArray()) + ", " + msg.getTopicName());
                } else {
                    Logger.d(TAG, "No Notification settings = null " + SecureString.instance(msg.getTopicName()));
                }
            }

            if (msg.isAlert()) {
                Uri soundUri = getSoundUriFromNotification(msg, ctx);
                if (soundUri != null) {
                    mBuilder.setSound(soundUri);
                }
            }


            if( Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP ) { // Android 5.0 or higher
                mBuilder.setVisibility(NotificationCompat.VISIBILITY_PUBLIC);
            }

            if (!isRichNotification(msg)){
                changeNotificationStyleToBigText(msg, mBuilder);
            }

            try {
                if (notificationTemplateType == 1) {
                    SimpleCollapseNotification.Builder collapseBuilder = new SimpleCollapseNotification
                            .Builder(ctx)
                            .setColor(mBuilder.getColor())
                            .setTitle(mContentTitle)
                            .setBody(mContentText);

                    SimpleExpandedNotification.Builder expandedNotification = new SimpleExpandedNotification
                            .Builder(ctx)
                            .setColor(mBuilder.getColor())
                            .setTitle(mContentTitle)
                            .setBody(mContentText)
                            .setImage(bigPictureBitmap);

                    if (hideLargeIcon) {
                        collapseBuilder.setLargeIcon(null);
                        expandedNotification.setLargeIcon(null);
                        mBuilder.setLargeIcon(null);
                    } else {
                        collapseBuilder.setLargeIcon(mLargeIcon);
                        expandedNotification.setLargeIcon(mLargeIcon);
                        mBuilder.setLargeIcon(null);
                    }

                    if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.M) {
                        mBuilder.setStyle(null);
                        collapseBuilder.setActions(mBuilder.mActions);
                        expandedNotification.setActions(mBuilder.mActions);
                    } else {
                        mBuilder.setStyle(new NotificationCompat.DecoratedCustomViewStyle());
                    }

                    RemoteViews collapseView = collapseBuilder.remoteViewsBuilder();

                    mBuilder.setCustomContentView(collapseView);
                    mBuilder.setCustomHeadsUpContentView(collapseView);
                    mBuilder.setCustomBigContentView(expandedNotification.remoteViewsBuilder());
                }

            } catch (Exception e) {
                e.printStackTrace();
            }

            if(AdpPushClient.get().prepareNotification(msg, mBuilder)) {
                Notification notification = mBuilder.build();

                try {
                    if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.M) {
                        if (notification.contentView != null) {
                            notification.contentView.setViewVisibility(R.id.icon, View.GONE);
                            notification.contentView.setViewVisibility(R.id.right_icon, View.GONE);
                            notification.contentView.setViewVisibility(R.id.right_side, View.GONE);
                        }

                        if (notification.bigContentView != null) {
                            notification.bigContentView.setViewVisibility(R.id.icon, View.GONE);
                            notification.bigContentView.setViewVisibility(R.id.right_icon, View.GONE);
                            notification.bigContentView.setViewVisibility(R.id.right_side, View.GONE);
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }

                if (badge > -1){
                    AdpPushClient.get().setBadge(badge);

                    /* NOTE :
                        Since from Xiaomi changed the way to show badge by notification from Miui 6.
                        `ShortcutBadger.applyCount` will not support Xiaomi devices.
                        If you want to support badges for Xiaomi, you have to send notifications. And usem `applyNotification`
                     */
                    if (AdpPushClient.get().isBadgeEnabled()){
                        ShortcutBadger.applyNotification(ctx, notification, AdpPushClient.get().getBadge());
                    }
                } else {
                    if (AdpPushClient.get().isBackground()){
                        AdpPushClient.get().incBadge();
                    }
                }
                notification.flags |= Notification.FLAG_AUTO_CANCEL;

                //TODO throws bad array lengths on Huawei 4.2.2
                notificationManager.notify(notificationId, notification);
                long clickTime = System.currentTimeMillis();
                JSONObject notifActionJson = getShownNotifActionDataJson(intent, clickTime);

                //Ignore send show notification event.
                if (notifDelivery) {
                    ChabokCommunicateEvent shownNotificationEvent = new ChabokCommunicateEvent(
                            notifActionJson,
                            ChabokCommunicateStatus.ShownNotification
                    );
                    EventBus.getDefault().post(shownNotificationEvent);
                }

                storeLastNotificationInfluenced(notifActionJson);
                ChabokEventDataStorage.setIsDirectInfluenced(true);
            }
        } catch( Exception e ) {
            Logger.e(TAG, "Error notifying user ", e);
        }

    }

    private static int getLedColorFromNotificationObject(JSONObject notificationObject) {
        if (notificationObject == null){
            return -1;
        }

        try {
            if (!notificationObject.has("ledColor")) {
                return -1;
            }

            String ledColorHex = notificationObject.getString("ledColor");
            if (ledColorHex == null || ledColorHex.trim().equals("")) {
                return -1;
            }

            String sharpSign = "#";
            if (ledColorHex.startsWith("#")){
                sharpSign = "";
            }
            ledColorHex = sharpSign + ledColorHex;
            if (isValidHexColor(ledColorHex)) {
            return Color.parseColor(ledColorHex);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return -1;
    }

    private static Uri getSoundUriFromNotification(ChabokNotification msg, Context ctx) {
        String soundName = getSoundNameFromNotification(msg);

        if (soundName != null && (soundName.contentEquals("null") || soundName.contentEquals("nil"))){
            return null;
        }

        if (soundName == null || !DrawableUtil.isResourceFileExists(soundName, "raw", ctx)) {
            return RingtoneManager.getActualDefaultRingtoneUri(
                    ctx, RingtoneManager.TYPE_NOTIFICATION
            );
        }

        return Uri.parse("android.resource://" + ctx.getPackageName() + "/raw/" + soundName);
    }

    private static String getSoundNameFromNotification(ChabokNotification msg) {
        String soundName = msg.getSound();
        if(soundName == null || soundName.isEmpty()) {
            return null;
        }

        String extension = MimeTypeMap.getFileExtensionFromUrl(msg.getSound());

        if (extension != null && !extension.equals("")){
            soundName =  soundName.replace("."+extension,"");
        }

        return soundName;
    }

        private static boolean isRichNotification(ChabokNotification chabokNotification){
        boolean isRichNotification = false;

        if (chabokNotification.getExtras() != null) {
            Bundle payload = chabokNotification.getExtras();

            //FCM message
            isRichNotification = payload.containsKey("mediaUrl");
        } else if (chabokNotification.getMessage() != null) {
            PushMessage payload = chabokNotification.getMessage();

            //Chabok message
            if (payload.getNotification() != null) {
                isRichNotification = payload.getNotification().has("mediaUrl");
            }
        }

        return isRichNotification;
    }

    private static void changeNotificationStyleToBigText(ChabokNotification chabokNotification, NotificationCompat.Builder builder) {
        try {

            String notifText = chabokNotification.getText();

            if (chabokNotification.getExtras() != null) {
                Bundle payload = chabokNotification.getExtras();

                //FCM message
                if (payload.containsKey("body")) {
                    notifText = payload.getString("body");
                }
            } else if (chabokNotification.getMessage() != null) {
                PushMessage payload = chabokNotification.getMessage();

                //Chabok message
                if (payload.getNotification() != null && payload.getNotification().has("body")) {
                    notifText = payload.getNotification().getString("body");
                }
            }

            if (notifText != null) {
                builder.setStyle(new NotificationCompat.BigTextStyle().bigText(notifText));
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                    builder.setPriority(Notification.PRIORITY_MAX);
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private static PendingIntent getDimissPendingIntent(Context ctx,
                                                        String msgId,
                                                        String trackId,
                                                        Bundle bundle) {
        int uniqueInt;Intent dismissIntent = new Intent(ctx, ChabokNotificationOpenedReceiver.class);
        dismissIntent.putExtra(Constants.CHABOK_DISMISS_NOTIFICATION, true);
        dismissIntent.putExtra("messageId", msgId);
        dismissIntent.putExtra("trackId", trackId);
        if (bundle != null) {
            dismissIntent.putExtras(bundle);
        }

        uniqueInt = (int) (System.currentTimeMillis() & 0xfffffff);

        if (android.os.Build.VERSION.SDK_INT >= 31) {
            return PendingIntent.getBroadcast(ctx, uniqueInt, dismissIntent, PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE);
        } else {
            return PendingIntent.getBroadcast(ctx, uniqueInt, dismissIntent, PendingIntent.FLAG_UPDATE_CURRENT);
        }
    }

    static String getChannelIdFromTopic(String topic){
        if (topic == null) {
            return "UNKNOWN";
        }

        String[] channels = topic.split("/");

        String deviceId = AdpPushClient.get().getInstallationId();
        String userId = AdpPushClient.get().getUserId();

        if (channels.length > 1){
            if (channels[0].contentEquals("public")){
                return channels[1];
            }

            if (deviceId != null && channels[1].contentEquals(deviceId)){
                return "This Device";
            }

            if (channels[0].contentEquals("private") ||
                    (userId != null && channels[0].contentEquals(userId))){
                return "Personal " + channels[1];
            }
        }

        return channels[0];
    }

    private static boolean isValidHexColor(String color){
        if (color == null){
            return false;
        }
        Pattern colorPattern = Pattern.compile("^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$");
        Matcher m = colorPattern.matcher(color);
        return m.matches();
    }

    private static String bundleToJson(Bundle bundle) {
        JSONObject json = new JSONObject();
        Set<String> keys = bundle.keySet();
        for (String key : keys) {
            try {
                json.put(key, bundle.get(key));
            } catch (JSONException e) {

            }
        }

        return json.toString();
    }

    private static Bitmap getBitmapFromURL(String strURL) {
        try {
            if (!isValidUrl(strURL)) {
                return null;
            }
            URL url = new URL(strURL);
            HttpsURLConnection connection = (HttpsURLConnection) url.openConnection();
            connection.setDoInput(true);
            TLSSocketFactory sc = new TLSSocketFactory();
            connection.setSSLSocketFactory(sc);

            connection.setDoInput(true);
            connection.connect();
            InputStream input = connection.getInputStream();
            return BitmapFactory.decodeStream(input);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        } catch (KeyManagementException e) {
            e.printStackTrace();
        }
        return null;
    }

    private static void addNotficationActionToBuilder(NotificationCompat.Builder mBuilder,
                                                      JSONArray actions,
                                                      Context ctx,
                                                      String msgId,
                                                      String trackId,
                                                      Bundle bundle) throws JSONException {
        if (actions != null) {
            for (int i=0; i < actions.length(); i++) {
                JSONObject action = actions.getJSONObject(i);

                int icon = 0;
                String id = "";
                String title = "";
                String iconName = "";

                if (action.has("id")){
                    id = action.getString("id");
                }
                if (action.has("title")){
                    title = action.getString("title");
                }
                if (action.has("icon")) {
                    iconName = action.getString("icon");
                    icon = getResourceIcon(iconName,ctx);
                }

                if (title.trim() == "" || id.trim() == ""){
                    continue;
                }

                Intent actionReceive = new Intent(ctx, ChabokNotificationOpenedReceiver.class);
                actionReceive.putExtra(Constants.CHABOK_ACTION_BUTTON, true);
                actionReceive.putExtra(Constants.ACTION_ID, id);
                actionReceive.putExtra(Constants.ACTION_TITLE, title);

                if (action.has("url")){
                    String url = action.getString("url");
                    actionReceive.putExtra(Constants.ACTION_URL, url);
                }

                if (msgId != null){
                    actionReceive.putExtra("messageId", msgId);
                }

                if (trackId != null){
                    actionReceive.putExtra("trackId", trackId);
                }

                if (bundle != null){
                    actionReceive.putExtras(bundle);
                }

                actionReceive.putExtra("action_button", true);

                actionReceive.setAction(id);
                PendingIntent pendingIntentAction;

                if (android.os.Build.VERSION.SDK_INT >= 31) {
                    pendingIntentAction = PendingIntent.getBroadcast(ctx, i, actionReceive,PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE);
                } else {
                    pendingIntentAction = PendingIntent.getBroadcast(ctx, i, actionReceive, PendingIntent.FLAG_UPDATE_CURRENT);
                }

                mBuilder.addAction(icon, title, pendingIntentAction);
            }
        }
    }

    private static int getResourceIcon(String iconName, Context ctx) {
        if (iconName == null)
            return 0;

        String trimmedIconName = iconName.trim();
        if (!isValidResourceName(trimmedIconName))
            return 0;

        int notificationIcon = getDrawableId(trimmedIconName, ctx);
        if (notificationIcon != 0)
            return notificationIcon;

        // Get system icon resource
//        try {
//            return drawable.class.getField(iconName).getInt(null);
//        } catch (Throwable t) {}

        return 0;
    }

    private static boolean isValidResourceName(String name) {
        return (name != null && !name.matches("^[0-9]"));
    }

    private static int getDrawableId(String name, Context ctx) {
        Resources resources = ctx.getResources();
        if (resources != null) {
            String packageName = ctx.getPackageName();
            if (packageName != null)
                return resources.getIdentifier(name, "drawable", packageName);
        }
        return 0;
    }

    private static int getNotificationIcon() {
        boolean useSilhouette = Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP;
        return useSilhouette ?
                AdpPushClient.get().getNotificationIconSilhouette()
                :
                AdpPushClient.get().getNotificationIcon();
    }

    private static boolean isValidUrl(String url){
        String URL_REGEX = "^((https?|ftp)://|(www|ftp)\\.)?[a-z0-9-]+(\\.[a-z0-9-]+)+([/?].*)?$";

        Pattern pattern = Pattern.compile(URL_REGEX);
        Matcher matcher = pattern.matcher(url);
        return matcher.find();
    }

    private int getUnseenBadge() {
        return ChabokLocalStorage.getSharedPreferences(getApplicationContext())
                .getInt( "androidUnseenBadge", 0 );
    }

    private static SharedPreferences getSharedPreferences(Context context) {
        return ChabokLocalStorage.getSharedPreferences(context);
    }

    public static void subscribe(final String channel) {
        new AsyncTask<Void, Void, Boolean>() {
            @Override
            protected Boolean doInBackground(final Void... params) {
                String[] tokens = channel.split("public/");
                String channel = tokens[0];
                if(tokens.length == 2) {
                    channel = tokens[1];
                }
                if (channel.equalsIgnoreCase("default")) {
                    channel = "chabok";
                }
                FirebaseMessaging.getInstance().subscribeToTopic(channel);
                return true;
            }

            @Override
            protected void onPostExecute(final Boolean registered) {
            }
        }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
    }

    public static void unsubscribe(final String topic) {
        new AsyncTask<Void, Void, Boolean>() {
            @Override
            protected Boolean doInBackground(final Void... params) {
                try{
                    String[] tokens = topic.split("public/");
                    String channel = tokens[0];
                    if(tokens.length == 2) {
                        channel = tokens[1];
                    }
                    if (topic.equalsIgnoreCase("default")) {
                        channel = "chabok";
                    }
                    FirebaseMessaging.getInstance().subscribeToTopic(channel);
                } catch (Exception e) {
                    Logger.e(TAG, "Error subscribing to firebase" , e);
                }
                return true;
            }

            @Override
            protected void onPostExecute(final Boolean registered) {
            }
        }.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, null, null, null);
    }


    /**
     * HANDLE REMOVE TOKEN?
     *
     * if(intent != null && intent.getExtras() != null) {
     if(intent.getExtras().getBoolean("unregister")) {
     instanceID.deleteToken(AdpPushClient.get().getSenderId(),
     GoogleCloudMessaging.INSTANCE_ID_SCOPE);
     //TODO send unregister token to the Chabok server
     return;
     }
     }
     */

    static boolean isChabokIntent(Intent intent){
        return intent != null && intent.hasExtra("CHABOK_NOTIFICATION");
    }

    static boolean hasActionButton(Intent intent){
        return intent != null && intent.getBooleanExtra(Constants.CHABOK_ACTION_BUTTON, false);
    }

    static boolean isDismissed(Intent intent){
        return intent != null && intent.getBooleanExtra(Constants.CHABOK_DISMISS_NOTIFICATION, false);
    }

    public static void handleClickNotification(Context context, Intent intent) {
        long clickTime = System.currentTimeMillis();

        if (!isDismissed(intent)
                && !hasActionButton(intent)
                && !isChabokIntent(intent)) {
            Logger.d(Logger.TAG, "----> It's not chabok intent :/ = " + intent);
            return;
        }
        ChabokNotificationAction notificationAction = getChabokNotificationAction(intent);


        JSONObject notifActionJson = getNotifActionDataJson(intent, clickTime, notificationAction);
        ChabokCommunicateEvent communicateEvent = new ChabokCommunicateEvent(
                notifActionJson,
                ChabokCommunicateStatus.ClickedNotification
        );

        if (notificationAction.type == ChabokNotificationAction.ActionType.Dismissed) {
            communicateEvent.status = ChabokCommunicateStatus.DismissedNotification;
        }
        EventBus.getDefault().post(communicateEvent);


        ChabokEventDataStorage.setIsDirectInfluenced(true);

        storeLastNotificationInfluenced(notifActionJson);

        boolean preventStartActivity = !AdpPushClient.get().notificationOpened(intent.getExtras(), notificationAction);

        //Prevent the launching of default Activity
        if (preventStartActivity || isDismissed(intent)) {
            Logger.d(Logger.TAG, "-- prevent start activity = " + preventStartActivity + " , isDismissed = " + isDismissed(intent));
            return;
        }

        if (notificationAction.type == ChabokNotificationAction.ActionType.ActionTaken) {

            //Backward compatibility for old version get rich notification action with broadcast by sending broadcast from OS
            //It should be deprecated soon.
            sendBroadcastForNotifAction(context, intent, notificationAction);

            if (intent.hasExtra(Constants.ACTION_URL)) {
                launchUrl(context, intent, Constants.ACTION_URL, "----> Start launch action url = ");
            }

            return;
        }

        if (notificationAction.type == ChabokNotificationAction.ActionType.Opened) {
            if (intent.hasExtra(Constants.CLICK_URL)) {
                launchUrl(context, intent, Constants.CLICK_URL, "----> Start launch click url = ");

                return;
            }
        }

        launchDefaultActivity(context, intent);
    }

    private static void storeLastNotificationInfluenced(JSONObject notifActionJson) {
        try {
            JSONObject influenceData = new JSONObject(notifActionJson.toString());

            influenceData.remove("type");
            influenceData.remove("dirty");

            ChabokEventDataStorage.storeLastNotificationOpenedInfluence(influenceData);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private static JSONObject getShownNotifActionDataJson(Intent intent, long clickTime) {
        return getNotifActionDataJson(intent, clickTime, null);
    }

    private static JSONObject getNotifActionDataJson(Intent intent, long clickTime, ChabokNotificationAction notificationAction) {
        JSONObject notifActionJson = new JSONObject();
        try {
            notifActionJson.put("type", 1);
            notifActionJson.put("actionTs", clickTime);

            String messageId = intent.getExtras().getString("messageId", null);
            Logger.d(TAG, "-- MessageId is " + messageId);
            if (messageId != null) {
                notifActionJson.put("msgId", messageId);
            }

            String trackId = intent.getExtras().getString("trackId");
            Logger.d(TAG, "-- trackId is " + trackId);
            if (trackId != null) {
                notifActionJson.put("trackId", trackId);
            }

            if (notificationAction == null) {
                notifActionJson.put("actionType", "SHOWN");
                return notifActionJson;
            }

            String actionId = notificationAction.actionID;
            if (actionId != null) {
                notifActionJson.put("actionId", actionId);
            }


            if (notificationAction.type == ChabokNotificationAction.ActionType.Opened) {
                notifActionJson.put("actionType", "OPENED");
            } else if (notificationAction.type == ChabokNotificationAction.ActionType.Dismissed) {
                notifActionJson.put("actionType", "DISMISSED");
            } else if (notificationAction.type == ChabokNotificationAction.ActionType.ActionTaken) {
                notifActionJson.put("actionType", "ACTION_TAKEN");
            }

        } catch (JSONException e) {
            e.printStackTrace();
        }
        return notifActionJson;
    }

    private static void launchDefaultActivity(Context context, Intent intent) {
        try {
            Class activity = AdpPushClient.get().getNotifActivityClass(intent.getExtras());

            Intent intentActivity = new Intent(context, activity);
            intentActivity.setFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT | Intent.FLAG_ACTIVITY_NEW_TASK);

            Logger.d(Logger.TAG, "----> Start launching activity = " + activity);

            context.startActivity(intentActivity);

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /** Backward compatibility for old version get rich notification action with broadcast by sending broadcast from OS
     * @param context is application context
     * @param intent main intent for notification, need to replace intent extras.
     * @param notificationAction contain notification action
     *
     * @deprecated will be removed after new major version.
     */
    @Deprecated
    private static void sendBroadcastForNotifAction(Context context, Intent intent, ChabokNotificationAction notificationAction) {
        Intent appIntent = new Intent();
        appIntent.setAction(notificationAction.actionID);
        appIntent.replaceExtras(intent);
        context.sendBroadcast(appIntent);
    }

    private static void launchUrl(Context context, Intent intent, String actionUrl, String logMessage) {
        try {

            String url = intent.getStringExtra(actionUrl);
            if (!url.contains("://"))
                url = "http://" + url;

            Intent i = new Intent(Intent.ACTION_VIEW);
            i.addFlags(
                    Intent.FLAG_ACTIVITY_NO_HISTORY |
                            Intent.FLAG_ACTIVITY_CLEAR_WHEN_TASK_RESET |
                            Intent.FLAG_ACTIVITY_MULTIPLE_TASK |
                            Intent.FLAG_ACTIVITY_NEW_TASK
            );

            i.setData(Uri.parse(url));

            Logger.d(Logger.TAG, logMessage + url);
            context.startActivity(i);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @NonNull
    private static ChabokNotificationAction getChabokNotificationAction(Intent intent) {
        ChabokNotificationAction notificationAction = new ChabokNotificationAction();

        if (isDismissed(intent)) {
            notificationAction.type = ChabokNotificationAction.ActionType.Dismissed;
            Logger.d(Logger.TAG, "----> Notification dismissed " + intent.getExtras());
        } else if (hasActionButton(intent)) {
            notificationAction.type = ChabokNotificationAction.ActionType.ActionTaken;
            notificationAction.actionID = intent.getStringExtra(Constants.ACTION_ID);
            if (intent.hasExtra(Constants.ACTION_URL)) {
                notificationAction.actionUrl = intent.getStringExtra(Constants.ACTION_URL);
            }
            Logger.d(Logger.TAG, "----> User tapped on ('" +
                    intent.getStringExtra(Constants.ACTION_TITLE) + "') action with id = " +
                    notificationAction.actionID + " from the notification");
        } else {
            notificationAction.type = ChabokNotificationAction.ActionType.Opened;
            if (intent.hasExtra(Constants.CLICK_URL)) {
                notificationAction.actionUrl = intent.getStringExtra(Constants.CLICK_URL);
            }
            Logger.d(Logger.TAG, "----> Notification was tapped on");
        }
        return notificationAction;
    }
}

class TLSSocketFactory extends SSLSocketFactory {

    private SSLSocketFactory delegate;

    public TLSSocketFactory() throws KeyManagementException, NoSuchAlgorithmException {
        SSLContext context = SSLContext.getInstance("TLS");
        context.init(null, null, null);
        delegate = context.getSocketFactory();
    }

    @Override
    public String[] getDefaultCipherSuites() {
        return delegate.getDefaultCipherSuites();
    }

    @Override
    public String[] getSupportedCipherSuites() {
        return delegate.getSupportedCipherSuites();
    }

    @Override
    public Socket createSocket() throws IOException {
        return enableTLSOnSocket(delegate.createSocket());
    }

    @Override
    public Socket createSocket(Socket s, String host, int port, boolean autoClose) throws IOException {
        return enableTLSOnSocket(delegate.createSocket(s, host, port, autoClose));
    }

    @Override
    public Socket createSocket(String host, int port) throws IOException, UnknownHostException {
        return enableTLSOnSocket(delegate.createSocket(host, port));
    }

    @Override
    public Socket createSocket(String host, int port, InetAddress localHost, int localPort) throws IOException, UnknownHostException {
        return enableTLSOnSocket(delegate.createSocket(host, port, localHost, localPort));
    }

    @Override
    public Socket createSocket(InetAddress host, int port) throws IOException {
        return enableTLSOnSocket(delegate.createSocket(host, port));
    }

    @Override
    public Socket createSocket(InetAddress address, int port, InetAddress localAddress, int localPort) throws IOException {
        return enableTLSOnSocket(delegate.createSocket(address, port, localAddress, localPort));
    }

    private Socket enableTLSOnSocket(Socket socket) {
        if(socket != null && (socket instanceof SSLSocket)) {
            ((SSLSocket)socket).setEnabledProtocols(new String[] {"TLSv1.1", "TLSv1.2"});
        }
        return socket;
    }
}
