package com.adpdigital.push;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import java.util.UUID;

import de.greenrobot.event.EventBus;

class ChabokCommunicateFallbackMachine {

    private boolean launched = false;
    private boolean connected = false;
    private boolean connecting = false;
    private boolean foreground = false;
    private boolean background = false;
    private boolean isSocketTimeout = false;
    private boolean sendingWithInstallation = false;

    private String TAG = ChabokCommunicateStatus.class.getName();

    private EventBus eventBus = EventBus.getDefault();
    private static ChabokCommunicateFallbackMachine instance;

    public static ChabokCommunicateFallbackMachine getInstance() {
        if (instance == null){
            instance = new ChabokCommunicateFallbackMachine();
            instance.eventBus.register(instance);
        }
        return instance;
    }

    public void onEvent(ConnectionStatus state) {
        connected = false;
        connecting = false;

        if (state == ConnectionStatus.CONNECTED) {
            connected = true;
            isSocketTimeout = false;

            Logger.d(TAG, "-- Connected to server. Check storage to publish dirty event data.");
            if (!sendingWithInstallation && ChabokEventDataStorage.hasData()) {
                sendEventDataWithPublishFallback();
            }
            sendingWithInstallation = false;

        } else if (state == ConnectionStatus.CONNECTING){
            connecting = true;
            
        }
    }

    public void onEvent(AppState state){
        if (state == AppState.LAUNCH){
            launched = true;
            background = false;
        } else if (state == AppState.FOREGROUND){
            if (launched && foreground){
                launched = false;

                if (!isSocketTimeout){
                    sendEventDataWithPublishFallback();
                }

            }
            background = false;
            foreground = true;
        } else if (state == AppState.BACKGROUND){
            launched = false;
            foreground = true;
            background = true;
        } else if (state == AppState.INITIALIZED) {
            Logger.d(TAG, "-- Initialize sdk, Clear dirty event data status from storage");

            if (ChabokEventDataStorage.hasData()){
                ChabokEventDataStorage.resetAllDirtyDataStatus();
            }
        } else if (state == AppState.REGISTERING) {
            if (ChabokEventDataStorage.hasData()){
                Logger.d(TAG, "-- We have events data. Start registering to server. Change sendingWithInstallation to TRUE.");
                sendingWithInstallation = true;
            }
        } else if (state == AppState.REGISTERED){
            if (ChabokEventDataStorage.hasData()) {
                Logger.d(TAG, "Device registered and clean dirty data from storage");
                sendingWithInstallation = false;
                ChabokEventDataStorage.removeDirtyData();
            }
        }
    }

    public void onEvent(ChabokCommunicateEvent communicateEvent){
        Logger.d(TAG,"-- Has communicateEvent status = " + communicateEvent.status);

        if (communicateEvent.data != null){
            String pId = communicateEvent.data.optString("id");
            if (communicateEvent.status == ChabokCommunicateStatus.PublishDelivered){
                if (pId != null) {
                    if (ChabokEventDataStorage.removePublishId(pId)) {
                        ChabokEventDataStorage.removeDirtyData();
                    }
                }
                return;
            } else if (communicateEvent.status == ChabokCommunicateStatus.NotConnectedToPushTrackEvent) {
                try {
                    pId = communicateEvent.data.getJSONObject("data").optString("id");
                } catch (JSONException e) {
                    e.printStackTrace();
                }

                if (pId != null) {
                    ChabokEventDataStorage.addPublishId(pId);
                }
            }

            ChabokEventDataStorage.insertData(communicateEvent.data);
        }

        if (communicateEvent.status == ChabokCommunicateStatus.DismissedNotification ||
                communicateEvent.status == ChabokCommunicateStatus.ShownNotification){
            sendEventDataWithFallbackReq();

            return;
        } else if (communicateEvent.status == ChabokCommunicateStatus.ClickedNotification){
//            if (background && !launched) {
//                Logger.d(TAG, "-- Ignore sending notification click event if we are not in foreground.");
//                return;
//            }

            Logger.d(TAG, "-- User tapped on notification when, background = "
                    + background + " , launched = " + launched + " , foreground = " + foreground);
        } else if (communicateEvent.status == ChabokCommunicateStatus.GotDeepLink){
            Logger.d(TAG, "-- Get deep-link data...");

            if (!background && !foreground){
                Logger.d(TAG, "-- Ignore send deep-link data now, wait for launching the app and send by installation or publish request");
                return;
            }
        } else if (communicateEvent.status == ChabokCommunicateStatus.PublishInBackground){
            Logger.d(TAG, "~~~~~~~~~> Sending publish in background data with event data if has eventData in storage");

            sendEventDataWithFallbackReq();

            return;
        }

        if (connected) {
            sendEventDataWithPublishFallback();
            return;
        } else if (connecting && !isSocketTimeout) {
            Logger.d(TAG, "-- Ignore for sending data to server because of is in connecting mode.");
            return;
        }

        sendEventDataWithFallbackReq();
    }

    public void onEvent(ChabokCommunicateStatus status) {
        if (status == ChabokCommunicateStatus.FailInstallationReq) {
            Logger.d(TAG, "-- Couldn't append event data to installation:(");

            if (ChabokEventDataStorage.hasData()){
                ChabokEventDataStorage.resetAllDirtyDataStatus();
            }

            return;
        } else if (status == ChabokCommunicateStatus.FailEventFallbackReq) {
            Logger.d(TAG, "-- Couldn't sent event data with fallback request. :(");

            if (ChabokEventDataStorage.hasData()){
                ChabokEventDataStorage.resetAllDirtyDataStatus();
            }

            return;
        } else if (status == ChabokCommunicateStatus.InstallationSuccessfullySent) {
            if (ChabokEventDataStorage.hasData()) {
                ChabokEventDataStorage.removeDirtyData();
            }

            return;
        } else if (status == ChabokCommunicateStatus.NeedToSendWithFallbackRequest) {
            Logger.d(TAG, "-- Need to send event data with fallback request.");

        } else if (status == ChabokCommunicateStatus.UpdatingInstallation) {
            if (ChabokEventDataStorage.hasData()){
                Logger.d(TAG, "-- We have events data and updating installation. Change sendingWithInstallation to TRUE.");
                sendingWithInstallation = true;
            }

            return;
        } else if (status == ChabokCommunicateStatus.SOCKET_TIMEOUT) {
            Logger.d(TAG, "-- Connection timeout = " + isSocketTimeout + ", Check storage to send data if needed.");
            isSocketTimeout = true;
            connected = false;

            if (ChabokEventDataStorage.hasData()) {
                sendEventDataWithFallbackReq();
            }
            return;
        } else if (status == ChabokCommunicateStatus.CONNECTION_REFUSED ||
                status == ChabokCommunicateStatus.CONNECTION_ERROR){
            Logger.d(TAG, "------------ Connection has error. try to use events request.");
            isSocketTimeout = true;
            connected = false;

            if (ChabokEventDataStorage.hasData()) {
                sendEventDataWithFallbackReq();
            }
            return;
        }

        if (ChabokEventDataStorage.hasData()) {
            sendEventDataWithFallbackReq();
        }
    }

    private void sendEventDataWithPublishFallback(){
        Logger.d(TAG, "-- sendEventDataWithPublishFallback: hasData = " +
                ChabokEventDataStorage.hasData());

        if (ChabokEventDataStorage.hasData()) {
            Logger.d(TAG, "-- Sending Event Data With Publish Fallback:");

            JSONArray cachedData = getReadyToSendEventData();

            String id =  UUID.randomUUID().toString();
            ChabokEventDataStorage.addPublishId(id);

            AdpPushClient.get().publishClientEvents(id, cachedData);
        }
    }

    private void sendEventDataWithFallbackReq() {
        boolean hasData = ChabokEventDataStorage.hasData();

        Logger.d(TAG, "-- sendEventDataWithFallbackReq: isHasData = " + hasData);

        if (AdpPushClient.get().getInstallationId() == null){
            Logger.d(TAG, "Not initialized yet to send fallback event...");
            return;
        }

        if (hasData){
            final JSONArray eventData = getReadyToSendEventData();
            if (eventData == null){
                Logger.d(TAG, "-- For unknown reason eventData is null but try to send eventData with fallback!!!!! :-|");
                ChabokEventDataStorage.removeDirtyData();

                return;
            }

            AdpPushClient.get().sendEventData(eventData, new Callback() {
                @Override
                public void onSuccess(Object value) {
                    onEvent(ChabokCommunicateStatus.InstallationSuccessfullySent);
                }

                @Override
                public void onFailure(Throwable value) {
                    onEvent(ChabokCommunicateStatus.FailEventFallbackReq);
                }
            });
        }
    }

    public JSONArray getReadyToSendEventData(){
        if (ChabokEventDataStorage.hasData()) {
            JSONArray eventData = ChabokEventDataStorage.changeToDirtyInStorage();
            if (eventData == null || eventData.length() == 0){
                return null;
            }
            return eventData;
        }
        return null;
    }

    public boolean ignoreRetryPublishEvent(SecureString payload){
        JSONObject json = null;
        try {
            json = new JSONObject(payload.asString());

            if (ChabokEventDataStorage.removePublishId(json.optString("id"))){
                return true;
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return false;
    }
}
