/*
 * Decompiled with CFR 0.152.
 */
package com.adobe.testing.s3mock.domain;

import com.adobe.testing.s3mock.domain.Bucket;
import com.adobe.testing.s3mock.domain.MultipartUploadInfo;
import com.adobe.testing.s3mock.domain.S3Object;
import com.adobe.testing.s3mock.domain.Tag;
import com.adobe.testing.s3mock.dto.CopyObjectResult;
import com.adobe.testing.s3mock.dto.MultipartUpload;
import com.adobe.testing.s3mock.dto.Owner;
import com.adobe.testing.s3mock.dto.Part;
import com.adobe.testing.s3mock.util.AwsChunkDecodingInputStream;
import com.adobe.testing.s3mock.util.HashUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BoundedInputStream;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class FileStore {
    private static final DateTimeFormatter S3_OBJECT_DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'").withZone(ZoneId.of("UTC"));
    private static final String META_FILE = "metadata";
    private static final String DATA_FILE = "fileData";
    private static final String PART_SUFFIX = ".part";
    private static final String DEFAULT_CONTENT_TYPE = "binary/octet-stream";
    private static final Logger LOG = LoggerFactory.getLogger(FileStore.class);
    private final File rootFolder;
    private final boolean retainFilesOnExit;
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final Map<String, MultipartUploadInfo> uploadIdToInfo = new ConcurrentHashMap<String, MultipartUploadInfo>();

    public FileStore(@Value(value="${root:}") String rootDirectory, @Value(value="${retainFilesOnExit:false}") boolean retainFilesOnExit) {
        this.rootFolder = this.createRootFolder(rootDirectory);
        this.retainFilesOnExit = retainFilesOnExit;
        LOG.info("Using \"{}\" as root folder. Will retain files on exit: {}", (Object)this.rootFolder.getAbsolutePath(), (Object)retainFilesOnExit);
    }

    private File createRootFolder(String rootDirectory) {
        File root = rootDirectory == null || rootDirectory.isEmpty() ? new File(FileUtils.getTempDirectory(), "s3mockFileStore" + new Date().getTime()) : new File(rootDirectory);
        if (!this.retainFilesOnExit) {
            root.deleteOnExit();
        }
        root.mkdir();
        return root;
    }

    public Bucket createBucket(String bucketName) throws IOException {
        File newBucket = new File(this.rootFolder, bucketName);
        FileUtils.forceMkdir((File)newBucket);
        if (!this.retainFilesOnExit) {
            newBucket.deleteOnExit();
        }
        return this.bucketFromPath(newBucket.toPath());
    }

    public List<Bucket> listBuckets() {
        DirectoryStream.Filter<Path> filter = x$0 -> Files.isDirectory(x$0, new LinkOption[0]);
        return this.findBucketsByFilter(filter);
    }

    public Bucket getBucket(String bucketName) {
        DirectoryStream.Filter<Path> filter = file -> Files.isDirectory(file, new LinkOption[0]) && file.getFileName().endsWith(bucketName);
        List<Bucket> buckets = this.findBucketsByFilter(filter);
        return buckets.size() > 0 ? buckets.get(0) : null;
    }

    private List<Bucket> findBucketsByFilter(DirectoryStream.Filter<Path> filter) {
        ArrayList<Bucket> buckets = new ArrayList<Bucket>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.rootFolder.toPath(), filter);){
            for (Path path : stream) {
                buckets.add(this.bucketFromPath(path));
            }
        }
        catch (IOException e) {
            LOG.error("Could not Iterate over Bucket-Folders", (Throwable)e);
        }
        return buckets;
    }

    private Bucket bucketFromPath(Path path) {
        Bucket result = null;
        try {
            BasicFileAttributes attributes = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
            result = new Bucket(path, path.getFileName().toString(), S3_OBJECT_DATE_FORMAT.format(attributes.creationTime().toInstant()));
        }
        catch (IOException e) {
            LOG.error("File can not be read!", (Throwable)e);
        }
        return result;
    }

    @Deprecated
    public S3Object putS3Object(String bucketName, String fileName, String contentType, String contentEncoding, InputStream dataStream, boolean useV4ChunkedWithSigningFormat) throws IOException {
        return this.putS3Object(bucketName, fileName, contentType, contentEncoding, dataStream, useV4ChunkedWithSigningFormat, Collections.emptyMap(), null, null);
    }

    @Deprecated
    public S3Object putS3Object(String bucketName, String fileName, String contentType, String contentEncoding, InputStream dataStream, boolean useV4ChunkedWithSigningFormat, Map<String, String> userMetadata) throws IOException {
        return this.putS3Object(bucketName, fileName, contentType, contentEncoding, dataStream, useV4ChunkedWithSigningFormat, userMetadata, null, null);
    }

    public S3Object putS3Object(String bucketName, String fileName, String contentType, String contentEncoding, InputStream dataStream, boolean useV4ChunkedWithSigningFormat, Map<String, String> userMetadata, String encryption, String kmsKeyId) throws IOException {
        boolean encrypted = StringUtils.isNotBlank((CharSequence)encryption) && StringUtils.isNotBlank((CharSequence)kmsKeyId);
        S3Object s3Object = new S3Object();
        s3Object.setName(fileName);
        s3Object.setContentType(contentType != null ? contentType : DEFAULT_CONTENT_TYPE);
        s3Object.setContentEncoding(contentEncoding);
        s3Object.setUserMetadata(userMetadata);
        s3Object.setEncrypted(encrypted);
        s3Object.setKmsEncryption(encryption);
        s3Object.setKmsEncryptionKeyId(kmsKeyId);
        Bucket theBucket = this.getBucketOrCreateNewOne(bucketName);
        File objectRootFolder = this.createObjectRootFolder(theBucket, s3Object.getName());
        if (!this.retainFilesOnExit) {
            objectRootFolder.deleteOnExit();
        }
        File dataFile = this.inputStreamToFile(this.wrapStream(dataStream, useV4ChunkedWithSigningFormat), objectRootFolder.toPath().resolve(DATA_FILE));
        s3Object.setDataFile(dataFile);
        s3Object.setSize(Long.toString(dataFile.length()));
        BasicFileAttributes attributes = Files.readAttributes(dataFile.toPath(), BasicFileAttributes.class, new LinkOption[0]);
        s3Object.setCreationDate(S3_OBJECT_DATE_FORMAT.format(attributes.creationTime().toInstant()));
        s3Object.setModificationDate(S3_OBJECT_DATE_FORMAT.format(attributes.lastModifiedTime().toInstant()));
        s3Object.setLastModified(attributes.lastModifiedTime().toMillis());
        s3Object.setMd5(this.digest(kmsKeyId, dataFile));
        File metaFile = new File(objectRootFolder, META_FILE);
        if (!this.retainFilesOnExit) {
            metaFile.deleteOnExit();
        }
        this.objectMapper.writeValue(metaFile, (Object)s3Object);
        return s3Object;
    }

    @Deprecated
    public S3Object putS3ObjectWithKMSEncryption(String bucketName, String fileName, String contentType, InputStream dataStream, boolean useV4ChunkedWithSigningFormat, String encryption, String kmsKeyId) throws IOException {
        return this.putS3Object(bucketName, fileName, contentType, null, dataStream, useV4ChunkedWithSigningFormat, Collections.emptyMap(), encryption, kmsKeyId);
    }

    @Deprecated
    public S3Object putS3ObjectWithKMSEncryption(String bucketName, String fileName, String contentType, InputStream dataStream, boolean useV4ChunkedWithSigningFormat, Map<String, String> userMetadata, String encryption, String kmsKeyId) throws IOException {
        return this.putS3Object(bucketName, fileName, contentType, null, dataStream, useV4ChunkedWithSigningFormat, userMetadata, encryption, kmsKeyId);
    }

    private InputStream wrapStream(InputStream dataStream, boolean useV4ChunkedWithSigningFormat) {
        InputStream inStream = useV4ChunkedWithSigningFormat ? new AwsChunkDecodingInputStream(dataStream) : dataStream;
        return inStream;
    }

    public void setObjectTags(String bucketName, String fileName, List<Tag> tags) throws IOException {
        S3Object s3Object = this.getS3Object(bucketName, fileName);
        Bucket theBucket = this.getBucket(bucketName);
        File objectRootFolder = this.createObjectRootFolder(theBucket, s3Object.getName());
        s3Object.setTags(tags);
        this.objectMapper.writeValue(new File(objectRootFolder, META_FILE), (Object)s3Object);
    }

    private Bucket getBucketOrCreateNewOne(String bucketName) throws IOException {
        Bucket theBucket = this.getBucket(bucketName);
        if (theBucket == null) {
            theBucket = this.createBucket(bucketName);
        }
        return theBucket;
    }

    private File createObjectRootFolder(Bucket theBucket, String objectName) {
        Path bucketPath = theBucket.getPath();
        File objectRootFolder = new File(bucketPath.toFile(), objectName);
        objectRootFolder.mkdirs();
        if (!this.retainFilesOnExit) {
            objectRootFolder.deleteOnExit();
        }
        return objectRootFolder;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private File inputStreamToFile(InputStream inputStream, Path filePath) {
        FileOutputStream outputStream = null;
        File targetFile = filePath.toFile();
        try {
            int read;
            if (!targetFile.exists()) {
                targetFile.createNewFile();
                if (!this.retainFilesOnExit) {
                    targetFile.deleteOnExit();
                }
            }
            outputStream = new FileOutputStream(targetFile);
            byte[] bytes = new byte[1024];
            while ((read = inputStream.read(bytes)) != -1) {
                ((OutputStream)outputStream).write(bytes, 0, read);
            }
        }
        catch (IOException e) {
            LOG.error("Wasn't able to store file on disk!", (Throwable)e);
        }
        finally {
            if (inputStream != null) {
                try {
                    inputStream.close();
                }
                catch (IOException e) {
                    LOG.error("InputStream can not be closed!", (Throwable)e);
                }
            }
            if (outputStream != null) {
                try {
                    ((OutputStream)outputStream).close();
                }
                catch (IOException e) {
                    LOG.error("OutputStream can not be closed!", (Throwable)e);
                }
            }
        }
        return targetFile;
    }

    private String normalizePrefix(Bucket bucket, String prefix) {
        if (prefix == null) {
            return null;
        }
        FileSystem fileSystem = bucket.getPath().getFileSystem();
        String normalized = fileSystem.getPath(prefix, new String[0]).toString();
        return normalized.length() != prefix.length() ? normalized + fileSystem.getSeparator() : normalized;
    }

    public S3Object getS3Object(String bucketName, String objectName) {
        Bucket theBucket = this.getBucket(Objects.requireNonNull(bucketName, "bucketName == null"));
        S3Object theObject = null;
        String relativeObjectName = StringUtils.removeStart((String)objectName, (String)"/");
        Path metaPath = theBucket.getPath().resolve(relativeObjectName + "/" + META_FILE);
        if (Files.exists(metaPath, new LinkOption[0])) {
            try {
                theObject = (S3Object)this.objectMapper.readValue(metaPath.toFile(), S3Object.class);
                theObject.setDataFile(theBucket.getPath().resolve(relativeObjectName + "/" + DATA_FILE).toFile());
            }
            catch (IOException e) {
                LOG.error("File can not be read", (Throwable)e);
                e.printStackTrace();
            }
        }
        return theObject;
    }

    public List<S3Object> getS3Objects(String bucketName, String prefix) throws IOException {
        Bucket theBucket = this.getBucket(Objects.requireNonNull(bucketName, "bucketName == null"));
        ArrayList<S3Object> resultObjects = new ArrayList<S3Object>();
        String normalizedPrefix = this.normalizePrefix(theBucket, prefix);
        Stream<Path> directoryHierarchy = Files.walk(theBucket.getPath(), new FileVisitOption[0]);
        Set collect = directoryHierarchy.filter(path -> path.toFile().isDirectory()).map(path -> theBucket.getPath().relativize((Path)path)).filter(path -> StringUtils.isBlank((CharSequence)prefix) || null != normalizedPrefix && path.toString().startsWith(normalizedPrefix)).collect(Collectors.toSet());
        for (Path path2 : collect) {
            S3Object s3Object = this.getS3Object(bucketName, path2.toString());
            if (s3Object == null) continue;
            resultObjects.add(s3Object);
        }
        return resultObjects;
    }

    public CopyObjectResult copyS3Object(String sourceBucketName, String sourceObjectName, String destinationBucketName, String destinationObjectName) throws IOException {
        return this.copyS3ObjectEncrypted(sourceBucketName, sourceObjectName, destinationBucketName, destinationObjectName, null, null, Collections.emptyMap());
    }

    @Deprecated
    public CopyObjectResult copyS3Object(String sourceBucketName, String sourceObjectName, String destinationBucketName, String destinationObjectName, Map<String, String> userMetadata) throws IOException {
        return this.copyS3ObjectEncrypted(sourceBucketName, sourceObjectName, destinationBucketName, destinationObjectName, null, null, userMetadata);
    }

    public CopyObjectResult copyS3ObjectEncrypted(String sourceBucketName, String sourceObjectName, String destinationBucketName, String destinationObjectName, String encryption, String kmsKeyId) throws IOException {
        return this.copyS3ObjectEncrypted(sourceBucketName, sourceObjectName, destinationBucketName, destinationObjectName, encryption, kmsKeyId, Collections.emptyMap());
    }

    public CopyObjectResult copyS3ObjectEncrypted(String sourceBucketName, String sourceObjectName, String destinationBucketName, String destinationObjectName, String encryption, String kmsKeyId, Map<String, String> userMetadata) throws IOException {
        S3Object sourceObject = this.getS3Object(sourceBucketName, sourceObjectName);
        if (sourceObject == null) {
            return null;
        }
        Map<String, String> copyUserMetadata = sourceObject.getUserMetadata();
        if (userMetadata != null && !userMetadata.isEmpty()) {
            copyUserMetadata = userMetadata;
        }
        S3Object copiedObject = this.putS3Object(destinationBucketName, destinationObjectName, sourceObject.getContentType(), sourceObject.getContentEncoding(), new FileInputStream(sourceObject.getDataFile()), false, copyUserMetadata, encryption, kmsKeyId);
        return new CopyObjectResult(copiedObject.getModificationDate(), copiedObject.getMd5());
    }

    public Boolean doesBucketExist(String bucketName) {
        return this.getBucket(bucketName) != null;
    }

    public boolean deleteObject(String bucketName, String objectName) throws IOException {
        S3Object s3Object = this.getS3Object(bucketName, objectName);
        if (s3Object != null) {
            FileUtils.deleteDirectory((File)s3Object.getDataFile().getParentFile());
            return true;
        }
        return false;
    }

    public boolean deleteBucket(String bucketName) throws IOException {
        Bucket bucket = this.getBucket(bucketName);
        if (bucket != null) {
            FileUtils.deleteDirectory((File)bucket.getPath().toFile());
            return true;
        }
        return false;
    }

    public MultipartUpload prepareMultipartUpload(String bucketName, String fileName, String contentType, String contentEncoding, String uploadId, Owner owner, Owner initiator, Map<String, String> userMetadata) {
        if (!Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId).toFile().mkdirs()) {
            throw new IllegalStateException("Directories for storing multipart uploads couldn't be created.");
        }
        MultipartUpload upload = new MultipartUpload(fileName, uploadId, owner, initiator, new Date());
        this.uploadIdToInfo.put(uploadId, new MultipartUploadInfo(upload, contentType, contentEncoding, userMetadata));
        return upload;
    }

    public MultipartUpload prepareMultipartUpload(String bucketName, String fileName, String contentType, String contentEncoding, String uploadId, Owner owner, Owner initiator) {
        return this.prepareMultipartUpload(bucketName, fileName, contentType, contentEncoding, uploadId, owner, initiator, Collections.emptyMap());
    }

    public Collection<MultipartUpload> listMultipartUploads() {
        return this.uploadIdToInfo.values().stream().map(info -> info.upload).collect(Collectors.toList());
    }

    public void abortMultipartUpload(String bucketName, String fileName, String uploadId) {
        this.synchronizedUpload(uploadId, uploadInfo -> {
            try {
                File partFolder = this.retrieveFile(bucketName, fileName, uploadId);
                FileUtils.deleteDirectory((File)partFolder);
                File entireFile = this.retrieveFile(bucketName, fileName, DATA_FILE);
                FileUtils.deleteQuietly((File)entireFile);
                this.uploadIdToInfo.remove(uploadId);
                return null;
            }
            catch (IOException e) {
                throw new IllegalStateException("Could not delete multipart upload tmp data.", e);
            }
        });
    }

    public String putPart(String bucketName, String fileName, String uploadId, String partNumber, InputStream inputStream, boolean useV4ChunkedWithSigningFormat) throws IOException {
        String string;
        DigestInputStream digestingInputStream = new DigestInputStream(this.wrapStream(inputStream, useV4ChunkedWithSigningFormat), MessageDigest.getInstance("MD5"));
        try {
            this.inputStreamToFile(digestingInputStream, Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId, partNumber + PART_SUFFIX));
            string = new String(Hex.encodeHex((byte[])digestingInputStream.getMessageDigest().digest()));
        }
        catch (Throwable throwable) {
            try {
                try {
                    digestingInputStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (NoSuchAlgorithmException e) {
                throw new IllegalStateException(e);
            }
        }
        digestingInputStream.close();
        return string;
    }

    public String completeMultipartUpload(String bucketName, String fileName, String uploadId, List<Part> parts) {
        return this.completeMultipartUpload(bucketName, fileName, uploadId, parts, null, null);
    }

    public String completeMultipartUpload(String bucketName, String fileName, String uploadId, List<Part> parts, String encryption, String kmsKeyId) {
        return this.synchronizedUpload(uploadId, uploadInfo -> {
            S3Object s3Object = new S3Object();
            s3Object.setName(fileName);
            s3Object.setEncrypted(encryption != null || kmsKeyId != null);
            if (encryption != null) {
                s3Object.setKmsEncryption(encryption);
            }
            if (kmsKeyId != null) {
                s3Object.setKmsEncryptionKeyId(kmsKeyId);
            }
            File partFolder = this.retrieveFile(bucketName, fileName, uploadId);
            File entireFile = this.retrieveFile(bucketName, fileName, DATA_FILE);
            String[] partNames = (String[])parts.stream().map(part -> part.getPartNumber() + PART_SUFFIX).toArray(String[]::new);
            long size = this.writeEntireFile(entireFile, partFolder, partNames);
            try {
                byte[] allMd5s = this.concatenateMd5sForAllParts(partFolder, partNames);
                FileUtils.deleteDirectory((File)partFolder);
                BasicFileAttributes attributes = Files.readAttributes(entireFile.toPath(), BasicFileAttributes.class, new LinkOption[0]);
                s3Object.setCreationDate(S3_OBJECT_DATE_FORMAT.format(attributes.creationTime().toInstant()));
                s3Object.setModificationDate(S3_OBJECT_DATE_FORMAT.format(attributes.lastModifiedTime().toInstant()));
                s3Object.setLastModified(attributes.lastModifiedTime().toMillis());
                s3Object.setMd5(DigestUtils.md5Hex((byte[])allMd5s) + "-" + partNames.length);
                s3Object.setSize(Long.toString(size));
                s3Object.setContentType(uploadInfo.contentType != null ? uploadInfo.contentType : DEFAULT_CONTENT_TYPE);
                s3Object.setContentEncoding(uploadInfo.contentEncoding);
                s3Object.setUserMetadata(uploadInfo.userMetadata);
                this.uploadIdToInfo.remove(uploadId);
            }
            catch (IOException e) {
                throw new IllegalStateException("Error finishing multipart upload", e);
            }
            try {
                this.objectMapper.writeValue(Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, META_FILE).toFile(), (Object)s3Object);
            }
            catch (IOException e) {
                throw new IllegalStateException("Could not write metadata-file", e);
            }
            return s3Object.getMd5();
        });
    }

    private String[] listAndSortPartsInFromDirectory(File partFolder) {
        String[] partNames = partFolder.list((dir, name) -> name.endsWith(PART_SUFFIX));
        Arrays.sort(partNames != null ? partNames : new String[]{}, Comparator.comparingInt(s -> Integer.parseInt(s.substring(0, s.indexOf(PART_SUFFIX)))));
        return partNames;
    }

    private byte[] concatenateMd5sForAllParts(File partFolder, String[] partNames) throws IOException {
        byte[] allMd5s = new byte[]{};
        for (String partName : partNames) {
            try (InputStream inputStream = Files.newInputStream(Paths.get(partFolder.getAbsolutePath(), partName), new OpenOption[0]);){
                allMd5s = ArrayUtils.addAll((byte[])allMd5s, (byte[])DigestUtils.md5((InputStream)inputStream));
            }
        }
        return allMd5s;
    }

    public List<Part> getMultipartUploadParts(String bucketName, String fileName, String uploadId) {
        File partsDirectory = this.retrieveFile(bucketName, fileName, uploadId);
        String[] partNames = this.listAndSortPartsInFromDirectory(partsDirectory);
        if (partNames != null) {
            File[] files = (File[])Arrays.stream(partNames).map(File::new).toArray(File[]::new);
            return this.arrangeSeparateParts(files, bucketName, fileName, uploadId);
        }
        return Collections.emptyList();
    }

    private File retrieveFile(String bucketName, String fileName, String uploadId) {
        return Paths.get(this.rootFolder.getAbsolutePath(), bucketName, fileName, uploadId).toFile();
    }

    private List<Part> arrangeSeparateParts(File[] files, String bucketName, String fileName, String uploadId) {
        ArrayList<Part> parts = new ArrayList<Part>();
        for (int i = 0; i < files.length; ++i) {
            String filePartPath = this.concatUploadIdAndPartFileName(files[i], uploadId);
            File currentFilePart = this.retrieveFile(bucketName, fileName, filePartPath);
            int partNumber = i + 1;
            String partMd5 = this.calculateHashOfFilePart(currentFilePart);
            Date lastModified = new Date(currentFilePart.lastModified());
            Part part = new Part();
            part.setLastModified(lastModified);
            part.setETag(partMd5);
            part.setPartNumber(partNumber);
            part.setSize(currentFilePart.length());
            parts.add(part);
        }
        return parts;
    }

    private String calculateHashOfFilePart(File currentFilePart) {
        String string;
        block8: {
            FileInputStream is = FileUtils.openInputStream((File)currentFilePart);
            try {
                String partMd5 = DigestUtils.md5Hex((InputStream)is);
                string = String.format("%s", partMd5);
                if (is == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            ((InputStream)is).close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    LOG.error("Hash could not be calculated. File access did not succeed", (Throwable)e);
                    return "";
                }
            }
            ((InputStream)is).close();
        }
        return string;
    }

    private String concatUploadIdAndPartFileName(File file, String uploadId) {
        return String.format("%s/%s", uploadId, file.getName());
    }

    private long writeEntireFile(File entireFile, File partFolder, String ... partNames) {
        long l;
        FileOutputStream targetStream = new FileOutputStream(entireFile);
        try {
            long size = 0L;
            for (String partName : partNames) {
                size += Files.copy(Paths.get(partFolder.getAbsolutePath(), partName), targetStream);
            }
            l = size;
        }
        catch (Throwable throwable) {
            try {
                try {
                    ((OutputStream)targetStream).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new IllegalStateException("Error writing entire file " + entireFile.getAbsolutePath(), e);
            }
        }
        ((OutputStream)targetStream).close();
        return l;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private <T> T synchronizedUpload(String uploadId, Function<MultipartUploadInfo, T> callback) {
        MultipartUploadInfo uploadInfo = this.uploadIdToInfo.get(uploadId);
        if (uploadInfo == null) {
            throw new IllegalArgumentException("Unknown upload " + uploadId);
        }
        MultipartUploadInfo multipartUploadInfo = uploadInfo;
        synchronized (multipartUploadInfo) {
            if (!this.uploadIdToInfo.containsKey(uploadId)) {
                throw new IllegalStateException("Upload " + uploadId + " was aborted or completed concurrently");
            }
            return callback.apply(uploadInfo);
        }
    }

    private String digest(String salt, File dataFile) throws IOException {
        String string;
        FileInputStream inputStream = new FileInputStream(dataFile);
        try {
            string = HashUtil.getDigest(salt, inputStream);
        }
        catch (Throwable throwable) {
            try {
                try {
                    inputStream.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (NoSuchAlgorithmException e) {
                LOG.error("Hash can not be calculated!", (Throwable)e);
                return null;
            }
        }
        inputStream.close();
        return string;
    }

    public String copyPart(String bucket, String key, long from, long to, String partNumber, String destinationBucket, String destinationFilename, String uploadId) throws IOException {
        this.verifyMultipartUploadPreparation(destinationBucket, destinationFilename, uploadId);
        File targetPartFile = this.ensurePartFile(partNumber, destinationBucket, destinationFilename, uploadId);
        return this.copyPart(bucket, key, from, to, targetPartFile);
    }

    private String copyPart(String bucket, String key, long from, long to, File partFile) throws IOException {
        long len = to - from + 1L;
        S3Object s3Object = this.resolveS3Object(bucket, key);
        try (FileInputStream sourceStream = FileUtils.openInputStream((File)s3Object.getDataFile());
             FileOutputStream targetStream = new FileOutputStream(partFile);){
            ((InputStream)sourceStream).skip(from);
            IOUtils.copy((InputStream)new BoundedInputStream((InputStream)sourceStream, len), (OutputStream)targetStream);
        }
        try (FileInputStream is = FileUtils.openInputStream((File)partFile);){
            String string = DigestUtils.md5Hex((InputStream)is);
            return string;
        }
    }

    private File ensurePartFile(String partNumber, String destinationBucket, String destinationFilename, String uploadId) throws IOException {
        File partFile = Paths.get(this.rootFolder.getAbsolutePath(), destinationBucket, destinationFilename, uploadId, partNumber + PART_SUFFIX).toFile();
        if (!partFile.exists() && !partFile.createNewFile()) {
            throw new IllegalStateException("Could not create buffer file");
        }
        return partFile;
    }

    private void verifyMultipartUploadPreparation(String destinationBucket, String destinationFilename, String uploadId) {
        Path partsFolder = Paths.get(this.rootFolder.getAbsolutePath(), destinationBucket, destinationFilename, uploadId);
        if (!partsFolder.toFile().exists() || !partsFolder.toFile().isDirectory()) {
            throw new IllegalStateException("Missed preparing Multipart Request");
        }
    }

    private S3Object resolveS3Object(String bucket, String key) {
        S3Object s3Object = this.getS3Object(bucket, key);
        if (s3Object == null) {
            throw new IllegalStateException("Source Object not found");
        }
        return s3Object;
    }
}

