/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops.options.protectpdf;

import com.adobe.pdfservices.operation.pdfops.ProtectPDFOperation;

import java.util.Objects;

/**
 * Parameters for securing PDF file with passwords and document permissions using {@link ProtectPDFOperation}.
 */
public class PasswordProtectOptions extends ProtectPDFOptions{

    private String userPassword;

    private String ownerPassword;

    private EncryptionAlgorithm encryptionAlgorithm;

    private ContentEncryption contentEncryption;

    private Permissions permissions;

    private PasswordProtectOptions(Builder builder) {
        this.userPassword = builder.userPassword;
        this.ownerPassword = builder.ownerPassword;
        this.encryptionAlgorithm = builder.encryptionAlgorithm;
        this.contentEncryption = builder.contentEncryption;
        this.permissions = builder.permissions;
    }

    /**
     * Returns the intended user password of the resulting encrypted PDF file.
     *
     * @return the user password
     */
    public String getUserPassword() {
        return userPassword;
    }

    /**
     * Returns the intended owner password of the resulting encrypted PDF file.
     *
     * @return the owner password
     */
    public String getOwnerPassword() {
        return ownerPassword;
    }

    /**
     * Returns the intended encryption algorithm of the resulting encrypted PDF file.
     *
     * @return the encryption algorithm
     */
    public EncryptionAlgorithm getEncryptionAlgorithm() {
        return encryptionAlgorithm;
    }

    /**
     * Returns the intended type of content to encrypt for the resulting encrypted PDF file.
     *
     * @return the type of content to encrypt
     */
    public ContentEncryption getContentEncryption() {
        return contentEncryption;
    }

    /**
     * Returns the intended document permissions for the resulting encrypted PDF file.
     *
     * @return the document permissions
     */
    public Permissions getPermissions() {
        return permissions;
    }

    /**
     * Builds a {@link PasswordProtectOptions} instance.
     */
    public static class Builder {

        private String userPassword;

        private String ownerPassword;

        private EncryptionAlgorithm encryptionAlgorithm;

        private ContentEncryption contentEncryption;

        private Permissions permissions;

        /**
         * Constructs a {@code Builder} instance.
         */
        public Builder() {}

        /**
         * Sets the intended user password required for opening the encrypted PDF file. Allowed maximum length for the user
         * password is 128 bytes.
         *
         * @param userPassword the user password; can not be null or empty
         * @return this Builder instance to add any additional parameters
         */
        public Builder setUserPassword(String userPassword) {
            Objects.requireNonNull(userPassword, "User password cannot be null");
            this.userPassword = userPassword;
            return this;
        }

        /**
         * Sets the intended owner password required to control access permissions in the encrypted PDF file. This
         * password can also be used to open/view the encrypted PDF file. Allowed maximum length for the owner password is 128
         * bytes.
         *
         * @param ownerPassword the owner password; can not be null or empty
         * @return this Builder instance to add any additional parameters
         */
        public Builder setOwnerPassword(String ownerPassword) {
            Objects.requireNonNull(ownerPassword, "Owner password cannot be null");
            this.ownerPassword = ownerPassword;
            return this;
        }

        /**
         * Sets the intended encryption algorithm required for encrypting the PDF file. For AES-128 encryption, the password
         * supports LATIN-I characters only. For AES-256 encryption, passwords supports Unicode character set.
         *
         * @param encryptionAlgorithm the encryption algorithm; can not be null
         * @return this Builder instance to add any additional parameters
         */
        public Builder setEncryptionAlgorithm(EncryptionAlgorithm encryptionAlgorithm) {
            this.encryptionAlgorithm = encryptionAlgorithm;
            return this;
        }

        /**
         * Sets the intended type of content to encrypt in the PDF file.
         *
         * @param contentEncryption the type of content to encrypt; can not be null
         * @return this Builder instance to add any additional parameters
         */
        public Builder setContentEncryption(ContentEncryption contentEncryption) {
            Objects.requireNonNull(contentEncryption, "Content encryption cannot be null");
            this.contentEncryption = contentEncryption;
            return this;
        }

        /**
         * Sets the intended permissions for the encrypted PDF file. This includes permissions to allow printing, editing
         * and content copying in the PDF document. Permissions can only be used in case the owner password is set.
         *
         * @param permissions the permissions; can not be null
         * @return this Builder instance to add any additional parameters
         */
        public Builder setPermissions(Permissions permissions) {
            Objects.requireNonNull(permissions, "Permissions cannot be null");
            this.permissions = permissions;
            return this;
        }

        /**
         * Returns a new {@link PasswordProtectOptions} instance built from the current state of this builder.
         *
         * @return a new {@code ProtectPDFOptions} instance
         */
        public PasswordProtectOptions build() {
            return new PasswordProtectOptions(this);
        }
    }
}
