/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops.options.compresspdf;

import com.adobe.pdfservices.operation.pdfops.CompressPDFOperation;

import java.util.Objects;

/**
 * Parameters for reducing file size of a pdf using {@link CompressPDFOperation}.
 */
public class CompressPDFOptions {

    private CompressionLevel compressionLevel;

    private CompressPDFOptions(CompressionLevel compressionLevel) {
        this.compressionLevel = compressionLevel;
    }

    /**
     * Determines the compression level ({@link CompressionLevel}) to be used for Compress PDF
     */
    public CompressionLevel getCompressionLevel() {
        return compressionLevel;
    }

    /**
     * Creates a new {@link CompressPDFOptions} builder.
     *
     * @return a {@link CompressPDFOptions.Builder} instance
     */
    public static CompressPDFOptions.Builder compressPDFOptionsBuilder() {
        return new CompressPDFOptions.Builder();
    }

    /**
     * Builds a {@link CompressPDFOptions} instance.
     */
    public static class Builder {

        private CompressionLevel compressionLevel;

        /**
         * Constructs a {@code CompressPDFOptions.Builder} instance.
         */
        public Builder() {

        }

        /**
         * Sets compression level to be used for Compress PDF, specified by {@link CompressionLevel}.
         *
         * @param compressionLevel see {@link CompressionLevel}.
         * @return this Builder instance to add any additional parameters
         */
        public CompressPDFOptions.Builder withCompressionLevel(CompressionLevel compressionLevel) {
            Objects.requireNonNull(compressionLevel, "CompressionLevel must not be null");
            this.compressionLevel = compressionLevel;
            return this;
        }

        /**
         * Returns a new {@link CompressPDFOptions} instance built from the current state of this builder.
         *
         * @return a new {@code CompressPDFOptions} instance
         */
        public CompressPDFOptions build() {
            return new CompressPDFOptions(compressionLevel);
        }
    }
}
