/*
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops;

import com.adobe.pdfservices.operation.ExecutionContext;
import com.adobe.pdfservices.operation.Operation;
import com.adobe.pdfservices.operation.exception.ServiceApiException;
import com.adobe.pdfservices.operation.exception.ServiceUsageException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.api.FileDownloadApi;
import com.adobe.pdfservices.operation.internal.exception.OperationException;
import com.adobe.pdfservices.operation.internal.service.PDFPropertiesService;
import com.adobe.pdfservices.operation.internal.util.FileUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.adobe.pdfservices.operation.internal.util.ValidationUtil;
import com.adobe.pdfservices.operation.io.FileRef;
import com.adobe.pdfservices.operation.pdfops.options.pdfproperties.PDFPropertiesOptions;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

/**
 * An Operation that is used to fetch properties from an input PDF file. The properties are returned in a JSON file or JSON Object.
 * <p>
 * Sample Usage:
 * <pre>{@code
 * PDFPropertiesOperation pdfPropertiesOperation = PDFPropertiesOperation.createNew();
 *
 * FileRef source = FileRef.createFromLocalFile("src/test/resources/pdfPropertiesOperationInput.pdf");
 * pdfPropertiesOperation.setInputFile(source);
 *
 * JSONObject result = pdfPropertiesOperation.execute(executionContext);
 * }</pre>
 */
public class PDFPropertiesOperation implements Operation {

    private static final Logger LOGGER = LoggerFactory.getLogger(PDFPropertiesOperation.class);

    /**
     * Supported media types for this operation
     */
    private static final Set<String> SUPPORTED_SOURCE_MEDIA_TYPES =
            new HashSet<>(Collections.singletonList(ExtensionMediaTypeMapping.PDF.getMediaType()));

    /**
     * Variable to check if the operation instance was invoked more than once
     */
    private boolean isInvoked = false;

    /**
     * Field representing the extension of the operation result
     */
    private static final String TARGET_FILE_EXTENSION = ExtensionMediaTypeMapping.JSON.getExtension();

    private FileRefImpl sourceFileRef;

    private PDFPropertiesOptions pdfPropertiesOptions;

    private PDFPropertiesOperation( ) {
    }

    /**
     * Sets the options for this operation. See {@link PDFPropertiesOptions} for how to specify the
     * options for PDF Properties Operation.
     *
     * @param pdfPropertiesOptions optional parameters; use null for default values
     */
    public void setOptions(PDFPropertiesOptions pdfPropertiesOptions){
        this.pdfPropertiesOptions = pdfPropertiesOptions;
    }

    /**
     * Constructs a {@code PDFPropertiesOperation} instance.
     *
     * @return a new {@code PDFPropertiesOperation} instance
     */
    public static PDFPropertiesOperation createNew() {
        return new PDFPropertiesOperation();
    }

    /**
     * Sets an input file.
     *
     * @param sourceFileRef an input file
     */
    public void setInputFile(FileRef sourceFileRef) {
        Objects.requireNonNull(sourceFileRef, "No input was set for operation");
        this.sourceFileRef = (FileRefImpl) sourceFileRef;
    }

    /**
     * Executes this operation synchronously using the supplied context and returns a new FileRef instance for the resulting JSON file.
     * <p>
     * The resulting file may be stored in the system temporary directory (per java.io.tmpdir System property).
     * See {@link FileRef} for how temporary resources are cleaned up.
     *
     * @param context the context in which to execute the operation
     * @return the resulting JSON file containing properties of PDF File.
     * @throws ServiceApiException   if an API call results in an error response
     * @throws IOException           if there is an error in reading either the input source or the resulting JSON File Reference.
     * @throws ServiceUsageException if service usage limits have been reached or credentials quota has been exhausted.
     */
    public FileRef executeAndReturnFileRef(ExecutionContext context) throws ServiceApiException, IOException, ServiceUsageException {
        validateInvocationCount();
        InternalExecutionContext internalExecutionContext = (InternalExecutionContext) context;
        this.validate(internalExecutionContext);

        try {
            LOGGER.info("All validations successfully done. Beginning PDFProperties operation execution");
            long startTimeMs = System.currentTimeMillis();

            String location = PDFPropertiesService.getPdfProperties(internalExecutionContext, sourceFileRef, pdfPropertiesOptions, this.getClass().getSimpleName());

            String targetFileName = FileUtil.getRandomFileName(TARGET_FILE_EXTENSION);
            String temporaryDestinationPath = PathUtil.getTemporaryDestinationPath(targetFileName,
                    TARGET_FILE_EXTENSION);

            FileDownloadApi.downloadAndSaveJson(internalExecutionContext, location, temporaryDestinationPath);

            isInvoked = true;
            LOGGER.info("Operation successfully completed. Stored requisite file at {}", temporaryDestinationPath);
            LOGGER.debug("Operation Success Info - Request ID: {}, Latency(ms): {}",
                    StringUtil.getRequestIdFromLocation(location), System.currentTimeMillis() - startTimeMs);

            return FileRef.createFromLocalFile(temporaryDestinationPath);
        } catch (OperationException oe) {
            throw new ServiceApiException(oe.getErrorMessage(), oe.getRequestTrackingId(), oe.getStatusCode(), oe.getReportErrorCode());
        }

    }

    /**
     * Executes this operation synchronously using the supplied context and returns a new JSONObject instance for the resulting JSON object.
     * <p>
     *
     * @param context the context in which to execute the operation
     * @return the resulting JSON object containing properties of PDF File.
     * @throws ServiceApiException   if an API call results in an error response
     * @throws IOException           if there is an error in reading either the input source or the resulting JSON Object
     * @throws ServiceUsageException if service usage limits have been reached or credentials quota has been exhausted.
     */
    public JSONObject execute(ExecutionContext context) throws ServiceApiException, IOException, ServiceUsageException {
        validateInvocationCount();
        InternalExecutionContext internalExecutionContext = (InternalExecutionContext) context;
        this.validate(internalExecutionContext);

        try {
            LOGGER.info("All validations successfully done. Beginning PDFProperties operation execution");
            long startTimeMs = System.currentTimeMillis();

            String location = PDFPropertiesService.getPdfProperties(internalExecutionContext, sourceFileRef, pdfPropertiesOptions, this.getClass().getSimpleName());

            JSONObject pdfPropertiesJsonObject = FileDownloadApi.downloadAndReturnJSONObject(internalExecutionContext, location);

            isInvoked = true;
            LOGGER.info("Operation successfully completed.");
            LOGGER.debug("Operation Success Info - Request ID: {}, Latency(ms): {}",
                    StringUtil.getRequestIdFromLocation(location), System.currentTimeMillis() - startTimeMs);

            return pdfPropertiesJsonObject;
        } catch (OperationException oe) {
            throw new ServiceApiException(oe.getErrorMessage(), oe.getRequestTrackingId(), oe.getStatusCode(), oe.getReportErrorCode());
        }

    }

    private void validateInvocationCount() {
        if (isInvoked) {
            LOGGER.error("Operation instance must only be invoked once");
            throw new IllegalStateException("Operation instance must not be reused, can only be invoked once");
        }
    }

    private void validate(InternalExecutionContext context) {
        if (sourceFileRef == null) {
            throw new IllegalArgumentException("No input was set for operation");
        }
        if (sourceFileRef.getSourceURL() != null) {
            throw new IllegalArgumentException("Input for the PDF Properties Operation can not be sourced from a URL");
        }
        ValidationUtil.validateExecutionContext(context);
        ValidationUtil.validateMediaType(SUPPORTED_SOURCE_MEDIA_TYPES, this.sourceFileRef.getMediaType());

    }

}
