/*
 * Copyright 2020 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.pdfops;

import com.adobe.pdfservices.operation.ExecutionContext;
import com.adobe.pdfservices.operation.Operation;
import com.adobe.pdfservices.operation.exception.ServiceApiException;
import com.adobe.pdfservices.operation.exception.ServiceUsageException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.exception.OperationException;
import com.adobe.pdfservices.operation.internal.service.ExtractPDFService.ExtractPDFService;
import com.adobe.pdfservices.operation.internal.util.FileUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.adobe.pdfservices.operation.internal.util.ValidationUtil;
import com.adobe.pdfservices.operation.io.FileRef;
import com.adobe.pdfservices.operation.pdfops.options.extractpdf.ExtractPDFOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.*;

/**
 * An Operation that extracts pdf elements such as text, images, tables in a structured format from a PDF.
 */
public class ExtractPDFOperation implements Operation {

    private static final Logger LOGGER = LoggerFactory.getLogger(ExtractPDFOperation.class);

    /**
     * Supported media types for this operation
     */
    private static final Set<String> SUPPORTED_SOURCE_MEDIA_TYPE =
            new HashSet<>(Arrays.asList(ExtensionMediaTypeMapping.PDF.getMediaType()));

    /**
     * Variable to check if the operation instance was invoked more than once
     */
    private boolean isInvoked = false;

    /**
     * Field representing the extension of the operation result
     */
    private static ExtensionMediaTypeMapping resultFormat = ExtensionMediaTypeMapping.ZIP;


    /* Following are the list of inputs being set by client */

    /**
     *  Reference to input file to be extracted
     */
    private FileRefImpl sourceFileRef;

    private ExtractPDFOptions extractPDFOptions;

    private ExtractPDFOperation( ) {
    }

    /**
     * Sets the options for this operation. See {@link ExtractPDFOptions} for how to specify the
     * options for the different elements and renditions.
     * @return current {@code ExtractPDFOperation} instance
     * @param extractPDFOptions option
     */
    public ExtractPDFOperation setOptions(ExtractPDFOptions extractPDFOptions){
        this.extractPDFOptions = extractPDFOptions;
        return this;
    }

    /**
     * Constructs a {@code ExtractPDFOperation} instance.
     *
     * @return a new {@code ExtractPDFOperation} instance
     */
    public static ExtractPDFOperation createNew() {
        return new ExtractPDFOperation();
    }

    /**
     * Sets an input file.
     *
     * @param sourceFileRef an input file
     * @return current {@code ExtractPDFOperation} instance
     */
    public ExtractPDFOperation setInputFile(FileRef sourceFileRef) {
        Objects.requireNonNull(sourceFileRef, "No input was set for operation");
        this.sourceFileRef = (FileRefImpl) sourceFileRef;
        return this;
    }

    /**
     * Executes this operation synchronously using the supplied context and returns a new FileRef instance for the resulting Zip file.
     * <p>
     * The resulting file may be stored in the system temporary directory (per java.io.tmpdir System property).
     * See {@link FileRef} for how temporary resources are cleaned up.
     *
     * @param context the context in which to execute the operation
     * @return the resulting Zip file containing elements information and renditions
     * @throws ServiceApiException   if an API call results in an error response
     * @throws IOException           if there is an error in reading either the input source or the resulting Zip file
     * @throws ServiceUsageException if service usage limits have been reached or credentials quota has been exhausted.
     */
    public FileRef execute(ExecutionContext context) throws ServiceApiException, IOException, ServiceUsageException {
        validateInvocationCount();
        InternalExecutionContext internalExecutionContext = (InternalExecutionContext) context;
        this.validate(internalExecutionContext);

        try {
            LOGGER.info("All validations successfully done. Beginning ExtractPDF operation execution");
            long startTimeMs = System.currentTimeMillis();

            String location = ExtractPDFService.extractPdf(internalExecutionContext, sourceFileRef, extractPDFOptions, this.getClass().getSimpleName());

            String targetFileName = FileUtil.getRandomFileName(resultFormat.getExtension());
            String temporaryDestinationPath = PathUtil.getTemporaryDestinationPath(targetFileName,
                    resultFormat.getExtension());

            ExtractPDFService.downloadAndSave(internalExecutionContext, location, temporaryDestinationPath,
                    resultFormat);

            isInvoked = true;
            LOGGER.info("Extract Operation Successful - Transaction ID: {}", StringUtil.getRequestIdFromLocation(location));

            return FileRef.createFromLocalFile(temporaryDestinationPath);
        } catch (OperationException oe) {
            throw new ServiceApiException(oe.getErrorMessage(), oe.getRequestTrackingId(), oe.getStatusCode(), oe.getReportErrorCode());
        }

    }

    private void validateInvocationCount() {
        if (isInvoked) {
            LOGGER.error("Operation instance must only be invoked once");
            throw new IllegalStateException("Operation instance must not be reused, can only be invoked once");
        }
    }

    private void validate(InternalExecutionContext context) {
        if (sourceFileRef == null) {
            throw new IllegalArgumentException("No input was set for operation");
        }
        if (sourceFileRef.getSourceURL() != null) {
            throw new IllegalArgumentException("Input for the Extract PDF Operation can not be sourced from a URL");
        }
        ValidationUtil.validateExecutionContext(context);
        ValidationUtil.validateMediaType(SUPPORTED_SOURCE_MEDIA_TYPE, this.sourceFileRef.getMediaType());

    }

}
