/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.io;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;

import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.ExecutionContext;
import com.adobe.pdfservices.operation.Operation;
import com.adobe.pdfservices.operation.pdfops.CreatePDFOperation;

/**
 * This class represents a local file. It is typically used by {@link Operation}
 * implementations which accept or return files.
 * <p>
 * When a FileRef instance is created by this SDK while referring to a temporary file location, calling any of the overloaded
 * {@code saveAs} methods will delete the temporary file.
 */
public abstract class FileRef {

    protected FileRef() {

    }

    /**
     * Creates a FileRef instance from an input stream using the specified media type.
     * <p>
     * The {@code InputStream} is not read by this method but by consumers of file content such as
     * {@link Operation#execute(ExecutionContext)}.
     *
     * @param inputStream Stream representing the file
     * @param mediaType   Media type, to identify the file format
     * @return a FileRef instance
     */
    public static FileRef createFromStream(InputStream inputStream, String mediaType) {
        return new FileRefImpl(inputStream, mediaType);
    }

    /**
     * Creates a FileRef instance from a local file path with an explicitly specified media type.
     * <p>
     * If the media type can be inferred from the file extension, consider {@link FileRef#createFromLocalFile(String)}.
     *
     * @param localSource Local file path, either absolute path or relative to classpath
     * @param mediaType   Media type to identify the local file format
     * @return a FileRef instance
     */
    public static FileRef createFromLocalFile(String localSource, String mediaType) {
        return new FileRefImpl(localSource, mediaType);
    }

    /**
     * Creates a FileRef instance from a local file path where the media type can be inferred from the file extension.
     * <p>
     * To specify the media type explicitly, use {@link FileRef#createFromLocalFile(String, String)}.
     *
     * @param localSource Local file path, either absolute path or relative to classpath
     * @return a FileRef instance
     */
    public static FileRef createFromLocalFile(String localSource) {
        return new FileRefImpl(localSource);
    }

    /**
     * Creates a FileRef instance from the input URL. As of now, this method is only supported for creation of PDF from HTML using {@link CreatePDFOperation}.
     *
     * @param url - Input URL.
     * @return a FileRef instance.
     *
     */
    public static FileRef createFromURL(URL url) {
        return new FileRefImpl(url);
    }

    /**
     * Saves this file to the location specified by {@code targetLocation}.
     * If this FileRef instance refers to a temporary local file created by the SDK, that temporary file is deleted.
     * <p>
     * The directory mentioned in {@code targetLocation} is created if it does not exist.
     *
     * @param targetLocation Local file path, either an absolute path or relative to classpath
     * @throws java.nio.file.FileAlreadyExistsException if the file already exists at the target location
     * @throws IOException                              if an I/O error occurs while saving the file
     * @throws UnsupportedOperationException            if this FileRef instance does not represent an operation result
     */
    public abstract void saveAs(String targetLocation) throws IOException;

    /**
     * Writes the contents of this file to {@code outputStream}.
     * If this FileRef instance refers to a temporary local file created by the SDK, that temporary file is deleted.
     * <p>
     * Note: This method does not close {@code outputStream}.
     *
     * @param outputStream the destination output stream
     * @throws IOException                   if an I/O error occurs while writing to the output stream
     * @throws UnsupportedOperationException if this FileRef instance does not represent an operation result
     */
    public abstract void saveAs(OutputStream outputStream) throws IOException;


}
