/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.util;

import com.adobe.pdfservices.operation.exception.SdkException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.pagemanipulation.PageAction;
import com.adobe.pdfservices.operation.internal.http.ByteArrayPart;
import com.adobe.pdfservices.operation.internal.http.MultiPartRequest;
import com.adobe.pdfservices.operation.internal.http.StringBodyPart;
import com.adobe.pdfservices.operation.internal.options.CombineOperationInput;
import com.adobe.pdfservices.operation.internal.options.PageRange;
import com.adobe.pdfservices.operation.pdfops.options.PageRanges;
import com.adobe.pdfservices.operation.pdfops.options.documentmerge.DocumentMergeOptions;
import com.adobe.pdfservices.operation.pdfops.options.protectpdf.EncryptionAlgorithm;
import com.adobe.pdfservices.operation.pdfops.options.protectpdf.PasswordProtectOptions;
import com.adobe.pdfservices.operation.pdfops.options.protectpdf.ProtectPDFOptions;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;


public class ValidationUtil {

    private static final long INPUT_FILE_SIZE_LIMIT = 104857600; // 100 MBs in bytes
    private static final int PASSWORD_MAX_LENGTH = 128; // 128 bytes
    private static final int PAGE_ACTIONS_MAX_LIMIT = 200;
    private static final String USER_STRING = "User";
    private static final String OWNER_STRING = "Owner";
    private static final int REMOVE_PROTECTION_PASSWORD_MAX_LENGTH = 150; // 150 characters
    private static final int SPLIT_PDF_OUTPUT_COUNT_LIMIT = 20;

    private static ValidatorFactory validatorFactory = Validation.buildDefaultValidatorFactory();

    public static void validateMediaType(Set<String> allowedMediaTypes, String mediaType) {
        if (mediaType == null || !allowedMediaTypes.contains(mediaType)) {
            throw new IllegalArgumentException("Operation cannot be performed on the specified input media type : " + mediaType);
        }
    }

    public static void validateExecutionContext(InternalExecutionContext context) {
        Objects.requireNonNull(context, "Client Context not initialized before invoking the operation");
        if (context.getClientConfig() == null) {
            throw new IllegalArgumentException("Client Context not initialized before invoking the operation");
        }
        context.validate();
    }

    public static void validateFileWithPageOptions(CombineOperationInput input,
                                                   Set<String> allowedMimeTypes) {


        validateMediaType(allowedMimeTypes, input.getSourceFileRef().getMediaType());

        validatePageOptions(input);
    }


    private static void validatePageOptions(CombineOperationInput input) {
        PageRanges pageRanges = input.getPageRanges();
        if (pageRanges == null) {
            throw new IllegalArgumentException("No page options provided for combining files PDFs");
        }
        pageRanges.validate();
    }


    public static void validateOptionInstanceType(Map<ExtensionMediaTypeMapping, Class> mediaTypeOptionClassMap,
                                                  String sourceMediaType, Object options) {
        Class instanceType = mediaTypeOptionClassMap.get(ExtensionMediaTypeMapping.getFromMimeType(sourceMediaType));
        if (instanceType == null || !instanceType.isInstance(options)) {
            throw new IllegalArgumentException(format("Invalid option instance type provided for source media type %s",
                    sourceMediaType));
        }
    }

    /**
     * A validation util which checks whether the given option instance is null or is instance of one of the given
     * option classes from the provided set
     *
     * @param optionsClassSet a set of option classes
     * @param options the option object to validate
     */
    public static void validateOptionInstanceType(Set<Class> optionsClassSet, Object options) {
        if (options == null  || !optionsClassSet.contains(options.getClass())) {
            throw new IllegalArgumentException("Invalid option instance type provided for the operation");
        }
    }

    public static void validateOperationOptions(Object bean) {
        Validator validator = validatorFactory.getValidator();
        Set<ConstraintViolation<Object>> violations = validator.validate(bean);
        if (!violations.isEmpty()) {
            String message = violations.stream().map(ConstraintViolation::getMessage).collect(Collectors.joining("; "));
            throw new IllegalArgumentException(message);
        }
    }

    public static void validateMultiPartBodySize(MultiPartRequest multiPartRequest) {
        double totalSize = 0;
        StringBodyPart stringBodyPart = multiPartRequest.getStringBodyPart();
        List<ByteArrayPart> byteArrayParts = multiPartRequest.getByteArrayParts();

        // Add the size(in bytes) of stringBodyPart
        if (stringBodyPart != null && stringBodyPart.getBody() != null) {
            totalSize = stringBodyPart.getBody().getBytes().length;
        }

        // Add the size(in bytes) for each byteArrayPart
        if (byteArrayParts != null) {
            for (ByteArrayPart byteArrayPart : byteArrayParts) {
                totalSize += byteArrayPart.getBody().length;
            }
        }

        // Compare the totalSize with INPUT_FILE_SIZE_LIMIT
        if (totalSize > INPUT_FILE_SIZE_LIMIT) {
            throw new SdkException("Total input file(s) size exceeds the acceptable limit");
        }
    }

    private static void validatePassword(String password, boolean isUserPassword, String encryptionAlgorithm) {

        if (StringUtil.isEmpty(password)) {
            throw new IllegalArgumentException(format("%s Password cannot be empty", isUserPassword? USER_STRING: OWNER_STRING));
        }

        if (password.length() > PASSWORD_MAX_LENGTH) {
            throw new IllegalArgumentException(format("%s Password length cannot exceed %d bytes",
                    isUserPassword? USER_STRING: OWNER_STRING, PASSWORD_MAX_LENGTH));
        }

        // Password validation for AES_128 encryption algorithm
        if (encryptionAlgorithm.equals(EncryptionAlgorithm.AES_128.getValue())) {

            if (!StandardCharsets.ISO_8859_1.newEncoder().canEncode(password)) {
                throw new IllegalArgumentException(format("%s Password supports only LATIN-I characters for AES-128 encryption",
                        isUserPassword? USER_STRING: OWNER_STRING));
            }
        }
    }

    public static void validateProtectPDFOptions(ProtectPDFOptions protectPDFOptions) {

        // Validations for PasswordProtectOptions
        if (PasswordProtectOptions.class.isInstance(protectPDFOptions)) {
            PasswordProtectOptions passwordProtectOptions = (PasswordProtectOptions) protectPDFOptions;

            // Validate encryption algo
            if (passwordProtectOptions.getEncryptionAlgorithm() == null) {
                throw new IllegalArgumentException("Encryption algorithm cannot be null");
            }

            if (StringUtil.isNull(passwordProtectOptions.getUserPassword()) &&
                StringUtil.isNull(passwordProtectOptions.getOwnerPassword()))
            {
                throw new IllegalArgumentException("One of the password (user/owner) is required");
            }

            if(!StringUtil.isNull(passwordProtectOptions.getOwnerPassword()) && !StringUtil.isNull(passwordProtectOptions.getUserPassword())){
                if(passwordProtectOptions.getOwnerPassword().equals(passwordProtectOptions.getUserPassword())){
                    throw new IllegalArgumentException("User and owner password cannot be same");
                }
            }
            // Validate user password
            if (!StringUtil.isNull(passwordProtectOptions.getUserPassword()))
                validatePassword(passwordProtectOptions.getUserPassword(), true, passwordProtectOptions.getEncryptionAlgorithm().getValue());

            // Validate owner password
            if (!StringUtil.isNull(passwordProtectOptions.getOwnerPassword()))
                validatePassword(passwordProtectOptions.getOwnerPassword(), false, passwordProtectOptions.getEncryptionAlgorithm().getValue());

            // OwnerPassword is mandatory in case the permissions are provided
            if (passwordProtectOptions.getPermissions() != null && StringUtil.isNull(passwordProtectOptions.getOwnerPassword()))
                throw new IllegalArgumentException("The document permissions cannot be applied without setting owner password");
        }
    }

    public static void validateInsertFilesInputs(Set<String> supportedSourceMediaTypes, Map<Integer, List<CombineOperationInput>> filesToInsert) {
        if (filesToInsert == null || filesToInsert.isEmpty()) {
            throw new IllegalArgumentException("No files to insert in the base input file");
        }

        for (Map.Entry<Integer, List<CombineOperationInput>> entry : filesToInsert.entrySet()) {
            if (entry.getKey() < 1) {
                throw new IllegalArgumentException("Base file page should be greater than 0");
            }
            for (CombineOperationInput combineOperationInput : entry.getValue()) {
                if (combineOperationInput.getSourceFileRef().getSourceURL() != null) {
                    throw new IllegalArgumentException("Input for the Insert Pages Operation can not be sourced from a URL");
                }
                validateMediaType(supportedSourceMediaTypes, combineOperationInput.getSourceFileRef().getMediaType());
                validatePageRanges(combineOperationInput.getPageRanges());
            }
        }
    }

    public static void validateReplaceFilesInputs(Set<String> supportedSourceMediaTypes, Map<Integer, CombineOperationInput> filesToReplace) {
        if (filesToReplace == null || filesToReplace.isEmpty()) {
            throw new IllegalArgumentException("No files to replace with");
        }
        for (Map.Entry<Integer, CombineOperationInput> entry : filesToReplace.entrySet()) {
            if (entry.getKey() < 1) {
                throw new IllegalArgumentException("Base file page should be greater than 0");
            }
            if (entry.getValue().getSourceFileRef().getSourceURL() != null) {
                throw new IllegalArgumentException("Input for the Replace Pages Operation can not be sourced from a URL");
            }
            validateMediaType(supportedSourceMediaTypes, entry.getValue().getSourceFileRef().getMediaType());
            validatePageRanges(entry.getValue().getPageRanges());
        }
    }

    public static void validatePageRanges(PageRanges pageRanges) {
        if (pageRanges == null || pageRanges.isEmpty()) {
            throw new IllegalArgumentException("No page ranges were set for the operation");
        }
        pageRanges.validate();
    }

    public static void validatePageRangesOverlap(PageRanges pageRanges) {

        // Creating a copy in case the pageRange order matters in the original pageRanges
        List<PageRange> pageRangeList = new ArrayList<>();
        pageRangeList.addAll(pageRanges.getRanges());

        Collections.sort(pageRangeList, Comparator.comparingInt(PageRange::getStart));

        for(int i = 1; i < pageRangeList.size(); i++) {
            if(pageRangeList.get(i-1).getEnd() == null || pageRangeList.get(i).getStart() <= pageRangeList.get(i-1).getEnd()) {
                throw new IllegalArgumentException("The overlapping page ranges are not allowed");
            }
        }

    }

    public static void validateRotatePageActions(List<PageAction> pageActions) {
        if (pageActions.isEmpty()) {
            throw new IllegalArgumentException("No rotation specified for the operation");
        }

        if (pageActions.size() > PAGE_ACTIONS_MAX_LIMIT) {
            throw new IllegalArgumentException("Too many rotations not allowed.");
        }

        for (PageAction pageAction : pageActions) {
            if (pageAction.getPageRanges().isEmpty()) {
                throw new IllegalArgumentException("No page ranges were set for the operation");
            }
        }
    }

    public static void validatePasswordToRemoveProtection(String password) {

        if (StringUtil.isNull(password) || StringUtil.isEmpty(password)) {
            throw new IllegalArgumentException("Password cannot be null or empty");
        }

        if (password.length() > REMOVE_PROTECTION_PASSWORD_MAX_LENGTH) {
            throw new IllegalArgumentException(format("Allowed maximum length of password is %d characters",
                    REMOVE_PROTECTION_PASSWORD_MAX_LENGTH));
        }
    }

    public static void validateSplitPDFOperationParams(PageRanges pageRanges, Integer pageCount, Integer fileCount) {
        if (pageRanges == null && pageCount == null && fileCount == null) {
            throw new IllegalArgumentException("One of the options(page ranges/file count/page count) is required for splitting a PDF document");
        }

        if (pageRanges != null) {
            if(pageCount != null || fileCount != null) {
                throw new IllegalArgumentException("Only one of option (page ranges/page count/file count) can be specified for splitting a PDF document");
            }

            if (pageRanges.getRanges().size() > SPLIT_PDF_OUTPUT_COUNT_LIMIT) {
                throw new IllegalArgumentException("Too many page ranges specified");
            }
            validatePageRangesOverlap(pageRanges);
            pageRanges.validate();
            return;
        }

        if (pageCount != null) {
            if (fileCount != null) {
                throw new IllegalArgumentException("Only one of option (page ranges/page count/file count) can be specified for splitting a PDF document");
            }

            if (pageCount <= 0) {
                throw new IllegalArgumentException("Page count should be greater than 0");
            }
            return;
        }

        if (fileCount <= 0) {
            throw new IllegalArgumentException("File count should be greater than 0");
        }

        if (fileCount > SPLIT_PDF_OUTPUT_COUNT_LIMIT) {
            throw new IllegalArgumentException(format("Input PDF file cannot be split into more than %d documents", SPLIT_PDF_OUTPUT_COUNT_LIMIT));
        }
    }

    public static void validateDocumentMergeOptions(DocumentMergeOptions documentMergeOptions){
        if (documentMergeOptions.getJsonDataForMerge() == null) {
            throw new IllegalArgumentException("Input json data cannot be null");
        }

        if (documentMergeOptions.getJsonDataForMerge().length() == 0) {
            throw new IllegalArgumentException("Input JSON Data cannot be null or empty");
        }

        if (documentMergeOptions.getOutputFormat() == null) {
            throw new IllegalArgumentException("Output format cannot be null");
        }
    }
}
