/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.service;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.options.CombineOperationInput;
import com.adobe.pdfservices.operation.pdfops.options.PageRanges;

public class ReplacePagesService extends CombinePDFService{

    private static final int INPUT_SOURCE_LIMIT = 20;


    private static ReplaceFilesInput getFilesToReplace(FileRefImpl baseFileRef,
                                                       Map<Integer, CombineOperationInput> filesToReplace) throws IOException {
        List<FileRefImpl> fileRefList = new ArrayList<>();
        List<PageRanges> pageRangeList = new ArrayList<>();
        int baseFileStartIndex = 1;
        for (Map.Entry<Integer, CombineOperationInput> entry : filesToReplace.entrySet()) {
            if (entry.getKey() != baseFileStartIndex) {
                fileRefList.add(baseFileRef.getCopy());
                PageRanges pageRanges = new PageRanges();
                pageRanges.addRange(baseFileStartIndex,entry.getKey()-1);
                pageRangeList.add(pageRanges);
            }

            fileRefList.add(entry.getValue().getSourceFileRef());
            pageRangeList.add(entry.getValue().getPageRanges());
            baseFileStartIndex = entry.getKey() + 1;
        }
        fileRefList.add(baseFileRef);
        PageRanges basePageRanges = new PageRanges();
        basePageRanges.addAllFrom(baseFileStartIndex);
        pageRangeList.add(basePageRanges);
        return new ReplaceFilesInput(fileRefList, pageRangeList);
    }

    public static String replacePages(InternalExecutionContext context,
                                      FileRefImpl baseFileRef,
                                      Map<Integer, CombineOperationInput> filesToReplace,
                                      String operation) throws IOException {
        try {
            ReplaceFilesInput replaceFilesInput = getFilesToReplace(baseFileRef, filesToReplace);

            // Prepare the sourceFileRefList
            List<FileRefImpl> sourceFileRefList = replaceFilesInput.fileRefList;

            // Validate replace operation input source limit
            if (sourceFileRefList.size() > INPUT_SOURCE_LIMIT) {
                throw new IllegalArgumentException("Too many replacements specified for the operation");
            }

            return execute(context,operation,replaceFilesInput.pageRangesList,sourceFileRefList,sourceFileRefList.size());
        } catch (FileNotFoundException fe) {
            throw fe;
        }
    }

    private static class ReplaceFilesInput {

        private List<FileRefImpl> fileRefList;
        private List<PageRanges> pageRangesList;

        ReplaceFilesInput(List<FileRefImpl> fileRefList, List<PageRanges> pageRangesList) {
            this.fileRefList = fileRefList;
            this.pageRangesList = pageRangesList;
        }
    }
}
