/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.http.config;


import com.adobe.pdfservices.operation.internal.GlobalConfig;
import com.adobe.pdfservices.operation.internal.http.HttpClientType;

public class HttpClientConfig {

    private int connectTimeout;
    private int socketTimeout;
    private HttpClientType clientType;
    private int maxRetries;

    /**
     * Not being used since we're not doing more than 1 retry internally
     */
    private int retryBackoffInterval;

    /**
     * Not being used since we're not doing more than 1 retry internally
     */
    private double retryDelayFactor;

    /**
     * Not being used since we're not doing more than 1 retry internally
     */
    private int maxRetryInterval;


    private HttpClientConfig(Builder clientConfigBuilder) {
        this.connectTimeout = clientConfigBuilder.connectTimeout;
        this.socketTimeout = clientConfigBuilder.socketTimeout;
        this.clientType = clientConfigBuilder.clientType;
        this.maxRetries = clientConfigBuilder.maxRetries;
        this.retryBackoffInterval = clientConfigBuilder.retryBackoffInterval;
        this.retryDelayFactor = clientConfigBuilder.retryDelayFactor;
        this.maxRetryInterval = clientConfigBuilder.maxRetryInterval;
    }

    public HttpClientType getClientType() {
        return clientType;
    }

    public int getRetryBackoffInterval() {
        return retryBackoffInterval;
    }

    public int getMaxRetries() {
        return maxRetries;
    }

    public double getRetryDelayFactor() {
        return retryDelayFactor;
    }

    public int getMaxRetryInterval() {
        return maxRetryInterval;
    }

    public int getConnectTimeout() {
        return connectTimeout;
    }

    public int getSocketTimeout() {
        return socketTimeout;
    }

    public static class Builder {

        private int connectTimeout = GlobalConfig.getConnectTimeout();
        private int socketTimeout = GlobalConfig.getSocketTimeout();
        private HttpClientType clientType = GlobalConfig.getDefaultHttpClientType();
        private int maxRetries = GlobalConfig.getMaxRetries();
        private int retryBackoffInterval = GlobalConfig.getRetryBackoffInterval();
        private double retryDelayFactor = GlobalConfig.getRetryDelayFactor();
        private int maxRetryInterval = GlobalConfig.getMaxRetryInterval();


        public Builder withSocketTimeout(int socketTimeout) {
            this.socketTimeout = socketTimeout;
            return this;
        }

        public Builder withConnectTimeout(int connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public Builder clientType(HttpClientType clientType) {
            this.clientType = clientType;
            return this;
        }

        public Builder withRetryLimit(int retryLimit) {
            this.maxRetries = retryLimit;
            return this;
        }

        public Builder withRetryInterval(int retryInterval) {
            this.retryBackoffInterval = retryInterval;
            return this;
        }

        public Builder withRetryDelayFactor(int retryDelayFactor) {
            this.retryDelayFactor = retryDelayFactor;
            return this;
        }

        public Builder withMaxRetryInterval(int maxRetryInterval) {
            this.maxRetryInterval = maxRetryInterval;
            return this;
        }

        public HttpClientConfig build() {
            return new HttpClientConfig(this);
        }

    }

}
