/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.api;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.adobe.pdfservices.operation.internal.cpf.dto.response.platform.CPFContentAnalyzerResponse;
import com.adobe.pdfservices.operation.internal.http.HttpResponse;
import com.adobe.pdfservices.operation.internal.http.MultiPartHttpResponse;
import com.adobe.pdfservices.operation.internal.util.FileUtil;
import com.adobe.pdfservices.operation.internal.util.PathUtil;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.pdfservices.operation.exception.SdkException;

import javax.mail.internet.MimeBodyPart;

public class FileDownloadApi {

    private static final Logger LOGGER = LoggerFactory.getLogger(FileDownloadApi.class);

    /**
     * This method downloads the file from the given location and saves it at the given destination path
     */
    public static <T> void downloadAndSave(InternalExecutionContext context,
                                           String location,
                                           String destinationPath,
                                           Class<T> responseType) throws IOException {

        HttpResponse<T> response = CPFApi.cpfStatusApi(context, location, responseType);
        InputStream inputStream = !response.getResponseAsStreamList().isEmpty() ? response.getResponseAsStreamList().get(0) : null;
        File file = new File(destinationPath);
        try {
            LOGGER.info("Downloading file to {} ", destinationPath);
            Files.copy(inputStream, file.toPath());
        } catch (IOException e) {
            LOGGER.error("Error {} while writing downloaded file to location {} ", e, destinationPath);
            throw new SdkException("Exception encountered while downloading file", e);

          /*
           Files.copy throws a null pointer exception when input stream is null. This case can occur when there is no
           inline file content in the multipart response.
          */
        } catch (NullPointerException e) {
            LOGGER.error("Error while fetching the output file content from the multipart response");
            throw new IOException("Exception encountered while downloading file", e);

        } finally {
            try {
                response.consume();
            } catch (IOException e) {
                LOGGER.error("Error while consuming file download response ", e);
            }
        }
    }

    /**
     * This method downloads multiple files from the given location and returns the list of temporary destination paths
     * of the downloaded files.
     */
    public static <T> List<String> downloadAndSaveMultiple(InternalExecutionContext context,
                                           String location,
                                           String targetFileExtension,
                                           Class<T> responseType) throws IOException{

        HttpResponse<T> response = CPFApi.cpfStatusApi(context, location, responseType);
        List<InputStream> inputStreamList = response.getResponseAsStreamList();
        List<String> paths = new ArrayList<>();
        String temporaryDestinationPath = null;

        try {
            /*
             inputStreamList will be empty in case there is no inline file content in the multipart response. In this
             case we will raise a dummy NullPointerException (to keep the behavior in sync with downloadAndSave method)
             */
            if(inputStreamList.isEmpty()) {
                throw new NullPointerException();
            }

            for(InputStream inputStream : inputStreamList) {
                String targetFileName = FileUtil.getRandomFileName(targetFileExtension);
                temporaryDestinationPath = PathUtil.getTemporaryDestinationPath(targetFileName, targetFileName);

                File file = new File(temporaryDestinationPath);
                LOGGER.info("Downloading file to {} ", temporaryDestinationPath);
                Files.copy(inputStream, file.toPath());
                paths.add(temporaryDestinationPath);
            }

        } catch (IOException e) {
            LOGGER.error("Error {} while writing downloaded file to location {} ", e, temporaryDestinationPath);
            throw new SdkException("Exception encountered while downloading file", e);

         /*
          Null pointer exception is thrown when input stream list is empty. This case can occur when there is no
          inline file content in the multipart response.
         */
        } catch (NullPointerException e) {
            LOGGER.error("Error while fetching the output file content from the multipart response");
            throw new IOException("Exception encountered while downloading file", e);

        } finally {
            try {
                response.consume();
            } catch (IOException e) {
                LOGGER.error("Error while consuming file download response ", e);
            }
        }
        return paths;
    }

    /**
     * This method get the multipart response and saves the json output in a file
     */
    public static void downloadAndSaveJson(InternalExecutionContext context, String location, String destinationPath) {

        HttpResponse response = CPFApi.cpfStatusApi(context, location, CPFContentAnalyzerResponse.class);
        MultiPartHttpResponse multiPartData = (MultiPartHttpResponse) response;
        List<MimeBodyPart> responseData = multiPartData.getResponseBodyParts();
        try {
            InputStream inputStream = !response.getResponseAsStreamList().isEmpty() ? responseData.get(1).getInputStream() : null;
            File file = new File(destinationPath);
            LOGGER.info("Downloading file to {} ", destinationPath);
            Files.copy(inputStream, file.toPath());
        } catch (IOException e) {
            LOGGER.error("Error {} while writing downloaded file to location {} ", e, destinationPath);
            throw new SdkException("Exception encountered while downloading file", e);
        } catch (Exception e) {
            LOGGER.debug(e.getMessage());
        } finally {
            try {
                response.consume();
            } catch (IOException e) {
                LOGGER.error("Error while consuming file download response ", e);
            }
        }
    }

    /**
     * This method get the multipart response and saves the json output in a JSON Object
     */
    public static JSONObject downloadAndReturnJSONObject(InternalExecutionContext context, String location) {

        HttpResponse response = CPFApi.cpfStatusApi(context, location, CPFContentAnalyzerResponse.class);
        MultiPartHttpResponse multiPartData = (MultiPartHttpResponse) response;
        List<MimeBodyPart> responseData = multiPartData.getResponseBodyParts();
        JSONObject pdfPropertiesJsonObject = null;
        try {
            InputStream inputStream = !response.getResponseAsStreamList().isEmpty() ? responseData.get(1).getInputStream() : null;
            ObjectMapper mapper = new ObjectMapper();
            pdfPropertiesJsonObject = new JSONObject(mapper.readValue(inputStream, Map.class));
        } catch (IOException e) {
            LOGGER.error("Error {} while creating JSON Object", e);
            throw new SdkException("Exception encountered while creating JSON Object ", e);
        } catch (Exception e) {
            LOGGER.debug(e.getMessage());
        } finally {
            try {
                response.consume();
            } catch (IOException e) {
                LOGGER.error("Error while consuming file download response ", e);
            }
        }
        return pdfPropertiesJsonObject;
    }
}
