/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal.api;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.adobe.pdfservices.operation.internal.cpf.constants.RequestKey;
import com.adobe.pdfservices.operation.internal.cpf.dto.request.platform.CPFContentAnalyzerRequests;
import com.adobe.pdfservices.operation.internal.exception.OperationException;
import com.adobe.pdfservices.operation.internal.util.InputStreamUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.pdfservices.operation.exception.SdkException;
import com.adobe.pdfservices.operation.internal.ExtensionMediaTypeMapping;
import com.adobe.pdfservices.operation.internal.FileRefImpl;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;
import com.adobe.pdfservices.operation.internal.http.BaseHttpRequest;
import com.adobe.pdfservices.operation.internal.http.DefaultRequestHeaders;
import com.adobe.pdfservices.operation.internal.http.HttpClient;
import com.adobe.pdfservices.operation.internal.http.HttpClientFactory;
import com.adobe.pdfservices.operation.internal.http.HttpResponse;
import com.adobe.pdfservices.operation.internal.http.InputStreamPart;
import com.adobe.pdfservices.operation.internal.http.MultiPartRequest;
import com.adobe.pdfservices.operation.internal.http.StringBodyPart;

public class CPFApi {

    private static final Logger LOGGER = LoggerFactory.getLogger(CPFApi.class);
    private static final String CONTENT_ANALYZER_REQUESTS_STRING = "contentAnalyzerRequests";
    private static final long TIME_LIMIT_MS = 1000 * 60 * 10; // 10 minutes in ms
    private static final int ACCEPTED_SUCCESS_STATUS_CODE = 202;
    private static final int POLLING_TIMEOUT_STATUS_CODE = 0;

    // Job Polling config
    private static final long POLLING_INTERVAL = 1000; // in milliseconds
    private static final long DELAY_POLLING_INTERVAL = 1000; // in milliseconds

    public static <T> HttpResponse<T> cpfCreateOpsApi(InternalExecutionContext context,
                                                      CPFContentAnalyzerRequests CPFContentAnalyzerRequests,
                                                      List<FileRefImpl> sourceFileRefList,
                                                      Class<T> responseType,
                                                      String operation) throws FileNotFoundException {
        try {
            HttpClient httpClient = HttpClientFactory.getDefaultHttpClient();
            MultiPartRequest multipartRequest
                    = (MultiPartRequest) context.getBaseRequestFromRequestContext(RequestKey.CREATE);

            // Prepare the multipart request
            prepareMultiPartRequest(multipartRequest, CPFContentAnalyzerRequests, sourceFileRefList, operation);

            return httpClient.send(multipartRequest, responseType);
        } catch (FileNotFoundException fe){
            // Mandatory to catch and throw FileNotFoundException, since we wrap other IOException cases in a runtime exception
            throw fe;
        } catch (IOException io) {
            throw new SdkException("Unexpected error while uploading file", io);
        }
    }

    public static <T> HttpResponse<T> cpfStatusApi(InternalExecutionContext context,
                                                   String location,
                                                   Class<T> responseType) {

        BaseHttpRequest baseHttpRequest = (BaseHttpRequest) context.getBaseRequestFromRequestContext(RequestKey.STATUS);
        baseHttpRequest.withTemplate(location);

        HttpClient httpClient = HttpClientFactory.getDefaultHttpClient();
        HttpResponse<T> response =
                httpClient.send(baseHttpRequest, responseType);

        // Time bound polling
        long startTimeMs = System.currentTimeMillis();
        int count = 1;
        while (response.getStatusCode() == ACCEPTED_SUCCESS_STATUS_CODE) {
            LOGGER.debug("Polling for status ");
            long retryIntervalMs = (count > 5) ? DELAY_POLLING_INTERVAL : POLLING_INTERVAL;
            if (System.currentTimeMillis() - startTimeMs > TIME_LIMIT_MS) {
                LOGGER.debug("Polling Timeout reached. Something's wrong, file operation took too long");
                throw new OperationException("Operation execution has timed out!",
                        StringUtil.getRequestIdFromLocation(location), POLLING_TIMEOUT_STATUS_CODE);
            } else {
                // Keep polling
                try {
                    TimeUnit.MILLISECONDS.sleep(retryIntervalMs);
                } catch (InterruptedException ex) {
                    throw new SdkException("Thread interrupted while waiting for operation execution status!!", ex);
                }
                response = httpClient.send(baseHttpRequest, responseType);
                count++;
            }
        }
        return response;
    }

    /**
     * This private method prepares the string body part and byte array body parts for a given multipart request; It also
     * sets x-dcsdk-ops-info header value to the specified operation name, for the multipart request.
     * @param multipartRequest          The multipart request
     * @param contentAnalyzerRequests   The object for string body part
     * @param sourceFileRefList         The list of sources for byte array body parts
     * @param operation                 The operation name
     * @throws IOException              If source file(s) not found
     */
    private static <T> void prepareMultiPartRequest(MultiPartRequest multipartRequest,
                                                    T contentAnalyzerRequests,
                                                    List<FileRefImpl> sourceFileRefList,
                                                    String operation) throws IOException {
        // Prepare the string body part of the multipart request
        ObjectMapper objectMapper = new ObjectMapper();
        String contentAnalyzerRequestsString = objectMapper.writeValueAsString(contentAnalyzerRequests);
        StringBodyPart stringBodyPart = new StringBodyPart(CONTENT_ANALYZER_REQUESTS_STRING,
                contentAnalyzerRequestsString, ExtensionMediaTypeMapping.TXT.getMediaType());
        multipartRequest = multipartRequest.withStringBodyPart(stringBodyPart);

        // Prepare the byte array body part(s) of the multipart request
        // partNumber is required in case when the request may have multiple byte array parts i.e. multiple file uploads
        int partNumber = 1;
        for (FileRefImpl fileRef : sourceFileRefList) {
            //We need to check if input stream exists for source file, in case of HTML2PDF via URL it will be null
            if(fileRef.getAsStream() != null) {
                InputStreamPart inputStreamPart = InputStreamUtil.createInputStreamPart(fileRef, partNumber);
                multipartRequest.withInputStreamPart(inputStreamPart);
                partNumber++;
            }
        }

        // Set x-dcsdk-ops-info header
        multipartRequest.withHeader(DefaultRequestHeaders.X_DCSDK_OPS_INFO_HEADER_NAME, operation);
    }
}
