/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation.internal;

/**
 * An internal mapping of Media types and extension.
 * <p>
 * <p>
 * Refer : https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/MIME_types/Complete_list_of_MIME_types
 * https://www.iana.org/assignments/media-types/media-types.xhtml
 */
public enum ExtensionMediaTypeMapping {


    AI("application/illustrator"),
    BMP("image/bmp"),
    DOC("application/msword"),
    DOCX("application/vnd.openxmlformats-officedocument.wordprocessingml.document"),
    GIF("image/gif"),
    HTML("text/html"),
    INDD("application/x-indesign"),
    JPEG("image/jpeg"),
    JPG("image/jpeg"),
    PDF("application/pdf"),
    PNG("image/png"),
    PPT("application/vnd.ms-powerpoint"),
    PPTX("application/vnd.openxmlformats-officedocument.presentationml.presentation"),
    PSD("image/vnd.adobe.photoshop"),
    RTF("text/rtf"),
    TIF("image/tiff"),
    TIFF("image/tiff"),
    TXT("text/plain"),
    XLS("application/vnd.ms-excel"),
    XLSX("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet"),
    ZIP("application/zip"),
    JSON("application/json"),
    CSV("text/csv");


    private final String mimeType;

    ExtensionMediaTypeMapping(String mimeType) {
        this.mimeType = mimeType;
    }

    public static ExtensionMediaTypeMapping getFromMimeType(String mimeType) {

        if (mimeType == null) {
            return null;
        }
        for (ExtensionMediaTypeMapping extensionMediaTypeMapping : ExtensionMediaTypeMapping.values()) {
            if (extensionMediaTypeMapping.getMediaType().equalsIgnoreCase(mimeType)) {
                return extensionMediaTypeMapping;
            }
        }
        return null;
    }

    public static ExtensionMediaTypeMapping getFromExtension(String extension) {
        if (extension == null) {
            return null;
        }
        for (ExtensionMediaTypeMapping extensionMediaTypeMapping : ExtensionMediaTypeMapping.values()) {
            if (extensionMediaTypeMapping.getExtension().equalsIgnoreCase(extension)) {
                return extensionMediaTypeMapping;
            }
        }
        return null;
    }

    public String getMediaType() {
        return mimeType;
    }

    public String getExtension() {
        return name().toLowerCase();
    }

}
