/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation;

import java.util.Objects;

import com.adobe.pdfservices.operation.auth.Credentials;
import com.adobe.pdfservices.operation.internal.InternalExecutionContext;

/**
 * Represents the execution context of an {@link Operation}. An execution context typically consists of the desired
 * authentication credentials and client configurations such as timeouts.
 * <p>
 * For each set of credentials, a {@code ExecutionContext} instance can be reused across operations.
 * <p>
 * Sample Usage:
 * <pre>{@code    Credentials credentials = Credentials.serviceAccountCredentialsBuilder().fromFile("pdfservices-api-credentials.json").build();
 *    ExecutionContext context = ExecutionContext.create(credentials);
 *    CreatePDFOperation cpdfOperation = CreatePDFOperation.createNew();
 *    cpdfOperation.setInput(FileRef.createFromLocalFile("~/Documents/createPdfInput.docx"));
 *    cpdfOperation.execute(context);
 * }</pre>
 */
public class ExecutionContext {

    protected ExecutionContext() {
    }

    /**
     * Creates a context instance using the provided {@link Credentials}.
     *
     * @param credentials - a {@link Credentials} instance
     * @return a new {@code ExecutionContext} instance
     */
    public static ExecutionContext create(Credentials credentials){
        return create(credentials,null);
    }

    /**
     * Creates a context instance using the provided {@link Credentials} and {@link ClientConfig}
     *
     * @param credentials - a {@link Credentials} instance
     * @param clientConfig - a {@link ClientConfig} instance for providing custom http timeouts
     * @return a new {@code ExecutionContext} instance
     */
    public static ExecutionContext create(Credentials credentials, ClientConfig clientConfig) {
        Objects.requireNonNull(credentials, "Credentials cannot be null");
        return new InternalExecutionContext(credentials, clientConfig);
    }

}
