/*
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: Adobe permits you to use, modify, and distribute this file in
 * accordance with the terms of the Adobe license agreement accompanying
 * it. If you have received this file from a source other than Adobe,
 * then your use, modification, or distribution of it requires the prior
 * written permission of Adobe.
 */

package com.adobe.pdfservices.operation;

import com.adobe.pdfservices.operation.internal.InternalClientConfig;
import com.adobe.pdfservices.operation.internal.util.JsonUtil;
import com.adobe.pdfservices.operation.internal.util.StringUtil;
import com.fasterxml.jackson.databind.JsonNode;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Encapsulates the API request configurations
 */
public class ClientConfig {

    private static String CPF_SERVICES_KEY = "cpfServices";
    private static String CPF_SERVICES_OPS_CREATE_KEY = "cpfOpsCreateUri";
    private static String CONNECT_TIMEOUT_KEY = "connectTimeout";
    private static String SOCKET_TIMEOUT_KEY = "socketTimeout";
    private static String PROXY_HOST = "proxyHost";
    private static String PROXY_PORT = "proxyPort";
    private static String PROXY_SCHEME = "proxyScheme";

    /**
     * Creates a new {@code ClientConfig} builder.
     *
     * @return a {@code ClientConfig.Builder} instance
     */
    public static ClientConfig.Builder builder(){
        return new ClientConfig.Builder();
    }

    public enum ProxyScheme {
        HTTP("http"),
        HTTPS("https");

        private final String proxyScheme;
        private static final Map<String, ProxyScheme> lookup = new HashMap<>();

        static
        {
            for(ProxyScheme proxyScheme : values())
            {
                lookup.put(proxyScheme.proxyScheme, proxyScheme);
            }
        }

        ProxyScheme(String proxyScheme) {
            this.proxyScheme = proxyScheme;
        }

        public static ProxyScheme get(String text)
        {
            return lookup.get(text.toLowerCase());
        }
    }

    /**
     * Builds a {@link ClientConfig} instance.
     */
    public static class Builder {
        private Integer connectTimeout;
        private Integer socketTimeout;
        private String cpfOpsCreateUri;
        private String proxyHost;
        private Integer proxyPort;
        private ClientConfig.ProxyScheme proxyScheme;

        /**
         * Constructs a {@code Builder} instance.
         */
        public Builder(){
        }

        /**
         * Sets the connect timeout. It should be greater than zero.
         *
         * @param connectTimeout determines the timeout in milliseconds until a connection is established in the API calls. Default value is 10000 milliseconds
         * @return this Builder instance to add any additional parameters
         */
        public ClientConfig.Builder withConnectTimeout(Integer connectTimeout){
            this.connectTimeout = connectTimeout;
            return this;
        }

        /**
         * Sets the socket timeout. It should be greater than zero.
         *
         * @param socketTimeout Defines the socket timeout in milliseconds, which is the timeout for waiting for data or,
         *                      put differently, a maximum period inactivity between two consecutive data packets).
         *                      Default value is 2000 milliseconds
         * @return this Builder instance to add any additional parameters
         */
        public ClientConfig.Builder withSocketTimeout(Integer socketTimeout){
            this.socketTimeout = socketTimeout;
            return this;
        }

        /**
         * Sets the Host name of proxy server (IP or DNS name)
         *
         * @param proxyHost Host Name of Proxy Server
         * @return this Builder instance to add any additional parameters
         */
        public ClientConfig.Builder withProxyHost(String proxyHost){
            this.proxyHost = proxyHost;
            return this;
        }

        /**
         * Sets the Port Number of Proxy server. It must be greater than 0. Ignored if proxy host is not set.
         *
         * @param proxyPort Port Number of Proxy server. Proxy Scheme's default port is set if not provided.
         * @return this Builder instance to add any additional parameters
         */
        public ClientConfig.Builder withProxyPort(Integer proxyPort){
            this.proxyPort = proxyPort;
            return this;
        }

        /**
         * Sets the scheme of Proxy Server (http or https). Ignored if proxy host is not set.
         *
         * @param proxyScheme Scheme of Proxy Server (http or https). Default value is http.
         * @return this Builder instance to add any additional parameters
         */
        public ClientConfig.Builder withProxyScheme(ClientConfig.ProxyScheme proxyScheme){
            this.proxyScheme = proxyScheme;
            return this;
        }

        /**
         * Sets the connect timeout, socket timeout and proxy config using the JSON client config file path.
         * All the keys in the JSON structure are optional.
         * <p>
         * JSON structure:
         * <pre>{
         *   "connectTimeout": "4000",
         *   "socketTimeout": "20000",
         *   "proxyHost": "127.0.0.1",
         *   "proxyPort": "8080",
         *   "proxyScheme": "https"
         * }</pre>
         *
         * @return this Builder instance to add any additional parameters
         */
        public ClientConfig.Builder fromFile(String clientConfigFilePath) {
            try {
                JsonNode clientConfig = JsonUtil.readJsonTreeFromFile(new File(clientConfigFilePath));
                JsonNode cpfServicesConfig = clientConfig.get(ClientConfig.CPF_SERVICES_KEY);
                if (cpfServicesConfig != null) {
                    JsonNode cpfServicesOpsCreateUriNode = cpfServicesConfig.get(ClientConfig.CPF_SERVICES_OPS_CREATE_KEY);
                    if (cpfServicesOpsCreateUriNode != null) {
                        this.cpfOpsCreateUri = cpfServicesOpsCreateUriNode.asText();
                    }
                }
                JsonNode connectTimeoutNode = clientConfig.get(ClientConfig.CONNECT_TIMEOUT_KEY);
                if (connectTimeoutNode != null) {
                    if (StringUtil.isPositiveInteger(connectTimeoutNode.asText())) {
                        this.connectTimeout = connectTimeoutNode.asInt();
                    } else {
                        throw new IllegalArgumentException(String.format("Invalid value for connect timeout %s Must be valid integer greater than 0",
                                connectTimeoutNode.asText()));
                    }
                }
                JsonNode socketTimeoutNode = clientConfig.get(ClientConfig.SOCKET_TIMEOUT_KEY);
                if (socketTimeoutNode != null) {
                    if (StringUtil.isPositiveInteger(socketTimeoutNode.asText())) {
                        this.socketTimeout = socketTimeoutNode.asInt();
                    } else {
                        throw new IllegalArgumentException(String.format("Invalid value for socket timeout %s. Must be valid integer greater than 0",
                                socketTimeoutNode.asText()));
                    }
                }
                JsonNode proxyHostNode = clientConfig.get(ClientConfig.PROXY_HOST);
                if (proxyHostNode != null) {
                    if (StringUtil.isNotBlank(proxyHostNode.asText())) {
                        this.proxyHost = proxyHostNode.asText();
                    }
                }
                JsonNode proxyPortNode = clientConfig.get(ClientConfig.PROXY_PORT);
                if (proxyPortNode != null) {
                    if (StringUtil.isPositiveInteger(proxyPortNode.asText())) {
                        this.proxyPort = proxyPortNode.asInt();
                    } else {
                        throw new IllegalArgumentException(String.format("Invalid value for proxy Port %s. Must be valid integer greater than 0",
                                proxyPortNode.asText()));
                    }
                }
                JsonNode proxySchemeNode = clientConfig.get(ClientConfig.PROXY_SCHEME);
                if (proxySchemeNode != null) {
                    if (StringUtil.isNotBlank(proxySchemeNode.asText()) && null != ClientConfig.ProxyScheme.get(proxySchemeNode.asText())) {
                        this.proxyScheme = ClientConfig.ProxyScheme.get(proxySchemeNode.asText());
                    } else {
                        throw new IllegalArgumentException(String.format("Invalid value for proxy Scheme %s. Must be either http or https",
                                proxySchemeNode.asText()));
                    }
                }
                return this;
            } catch (IOException ioException) {
                throw new IllegalArgumentException("Invalid Client config file provided.");
            }
        }

        /**
         * Returns a new {@link ClientConfig} instance built from the current state of this builder.
         *
         * @return a new {@code ClientConfig} instance
         */
        public ClientConfig build() {
            return new InternalClientConfig(this.connectTimeout, this.socketTimeout, this.cpfOpsCreateUri, this.proxyHost, this.proxyPort,
                    (null != this.proxyScheme) ? this.proxyScheme.proxyScheme : null);
        }
    }
}
