package ca.uhn.hl7v2.concurrent;

import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

public interface BlockingMap<K, V> extends Map<K, V> {

	/**
	 * Adds an entry only if there's already a consumer waiting for the value.
	 * 
	 * @param key key for the entry to be added
	 * @param value entry to be added
	 * @return true if entry was added and a consumer is already waiting for the
	 *         value, false otherwise
	 */
	boolean give(K key, V value);

	/**
	 * Waits for an entry for the given key and returns the associated value.
	 * May return null if the producer withdraws the entry without providing a
	 * value.
	 * 
	 * @param key key for the entry
	 * @return the value of the entry
	 * @throws InterruptedException
	 */
	V take(K key) throws InterruptedException;
	
	/**
	 * Waits for an entry in a background thread.
	 * 
	 * @param key key for the entry
	 * @return Future the result
     */
	Future<V> asyncTake(K key);

	/**
	 * Waits for the specified amount of time for an entry with the given key
	 * and returns the associated value. Returns null if no value was provided
	 * within the poll time. May return null if the producer withdraws the entry
	 * without providing a value.
	 * 
	 * @param key key for the entry
	 * @param timeout timeout before the methods returns
	 * @param unit time unit used in conjunction with timout
	 * @return the value of the entry
	 * @throws InterruptedException
	 */
	V poll(K key, long timeout, TimeUnit unit) throws InterruptedException;

	/**
	 * Polls for an entry in a background thread.
	 * 
	 * @param key key for the entry
	 * @return Future the result
     */
	Future<V> asyncPoll(K key, long timeout, TimeUnit unit);
}

