package buzz.getcoco.iot.android;

import android.os.CountDownTimer;
import android.os.SystemClock;
import android.util.Log;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import buzz.getcoco.iot.Device;
import buzz.getcoco.iot.Resource;

public class ResourceAdvert extends Resource {

  private static final String TAG = "ResourceAdvert";

  private static final long EXPIRY_TIME   = 20_000; // 20 sec
  private static final long TICK_INTERVAL = 1000;   // 1 sec

  public enum Status {
    ACTIVE,
    EXPIRED
  }

  private long timeOfAdvert;

  private CountDownTimer timer;
  private MutableLiveData<Long> expiryObservable;

  public ResourceAdvert(String eui, Device parentDevice) {
    super(eui, parentDevice, null);

    timeOfAdvert = SystemClock.elapsedRealtime();

    Log.d(TAG, "ResourceAdvert: advert created: " + eui + ", at: " + timeOfAdvert);
  }

  private long millisSinceAdvert() {
    return SystemClock.elapsedRealtime() - timeOfAdvert;
  }

  public long millisToExpiry() {
    return EXPIRY_TIME - millisSinceAdvert();
  }

  public Status getStatus() {
    return (EXPIRY_TIME < millisSinceAdvert()) ? Status.EXPIRED : Status.ACTIVE;
  }

  public void resetTimers() {
    timeOfAdvert = SystemClock.elapsedRealtime();

    if (null != timer) {
      timer.cancel();
      timer.start();
    }

    if (null != expiryObservable) {
      expiryObservable.postValue(millisToExpiry());
    }
  }

  @MainThread
  public LiveData<Long> getExpiryObservable() {

    if (null == expiryObservable) {
      long millisToExpiry = millisToExpiry();

      expiryObservable = new MutableLiveData<>(millisToExpiry);

      timer = new CountDownTimer(millisToExpiry, TICK_INTERVAL) {
        @Override
        public void onTick(long millisUntilFinished) {
          expiryObservable.postValue(millisToExpiry());
        }

        @Override
        public void onFinish() {
          expiryObservable.postValue(millisToExpiry());
        }
      }.start();
    }

    return expiryObservable;
  }

  @NonNull
  @Override
  public String toString() {
    return "ResourceAdvert{" +
        "id: " + getId() +
        ", timeOfAdvert=" + timeOfAdvert +
        ", millisToExpiry=" + millisToExpiry() +
        '}';
  }
}
