package buzz.getcoco.iot.android;

import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.common.collect.ImmutableList;

import java.util.List;

import buzz.getcoco.iot.Attribute;
import buzz.getcoco.iot.Capability;
import buzz.getcoco.iot.CapabilityWrapper;
import buzz.getcoco.iot.Resource;

public class CapabilityWrapperEx implements CapabilityWrapper {

  private final Capability realCapability;

  private MutableLiveData<String> nameObservable;
  private MutableLiveData<Resource> parentResourceObservable;
  private ImmutableList<Attribute> attributeList;
  private MutableLiveData<List<Attribute>> attributeObservable;

  protected CapabilityWrapperEx(Capability realCapability) {
    this.realCapability = realCapability;
  }

  @Override
  public void internalSetName(String name) {
    FactoryUtility.postValue(nameObservable, name);
  }

  @Override
  public void internalSetParent(Resource parent) {
    FactoryUtility.postValue(parentResourceObservable, parent);
  }

  @Override
  public void internalAddAttribute(Attribute attribute) {
    postAttributes();
    postCapabilityChanged();
  }

  @Override
  public void internalRemoveAttribute(int attributeId) {
    postAttributes();
    postCapabilityChanged();
  }

  private void postAttributes() {
    if (null == attributeList)
      return;

    FactoryUtility.postValue(attributeObservable, getAttributes());
  }

  private List<Attribute> getAttributes() {
    return (attributeList = ImmutableList.copyOf(getRealCapability().getAttributeMap().values()));
  }

  private void postCapabilityChanged() {
    ResourceEx resource = (ResourceEx) getRealCapability().getParent();

    if (null == resource)
      return;

    resource.onCapabilitiesChanged(this);
  }

  @Override
  public void clearAndSetStandardCommandSet(int[] commandsSupported) {
    // add LiveData if needed
  }

  @Override
  public void markAsReady() {
    for (Attribute a : getRealCapability()) {
      AttributeEx aEx = (AttributeEx) a;
      aEx.syncCurrentValue();
    }
  }

  @Override
  public Capability getRealCapability() {
    return realCapability;
  }

  public LiveData<String> getNameObservable() {
    return (null != nameObservable) ? nameObservable : (nameObservable = FactoryUtility.createLiveData(getRealCapability().getName()));
  }

  public LiveData<Resource> getParentResourceObservable() {
    return (null != parentResourceObservable) ? parentResourceObservable : (parentResourceObservable = FactoryUtility.createLiveData(getRealCapability().getParent()));
  }

  public LiveData<List<Attribute>> getAttributeObservable() {
    return (null != attributeObservable) ? attributeObservable : (attributeObservable = FactoryUtility.createLiveData(getAttributes()));
  }
}
